<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Mail;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Pager;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\User\EmployeePool;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class WriterPools extends Controller {
    public function pools() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::has($params, 'page', 1);

        $this->setVars([
            'page_title' => "Writer Pools"
        ]); 

        $sql = new DbQuery();
        $sql->select('ep.employee_pool_id');
        $sql->from('employee_pool', 'ep');
        $sql->orderBy('ep.order ASC, ep.employee_pool_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_pools = count($result);

        $writer_pools = array();
        $objectPresenter = new ObjectPresenter(); 
        if( $total_pools > 0) {
            $params['total_items'] = $total_pools;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/pools?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $pools_result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);

            foreach($pools_result as $pool) {
                $employeePool = new EmployeePool((int) $pool['employee_pool_id']);
                if(Validate::isLoadedObject($employeePool)) {
                    $writer_pool = $objectPresenter->present($employeePool);
                    $writer_pool['badge_url'] = Configuration::get('UPLOADS_PATH') . '/' . $employeePool->badge;

                    $pool_employees = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT e.* FROM ' . Db::prefix('employee') . ' AS e WHERE employee_pool_id = ' . (int) $employeePool->id );
                    
                    $employees = [];
                    foreach($pool_employees as $employee) {
                        $employee['user_picture'] = $user->getPicture($employee['avatar'], $employee['gender']);
                        $employees[] = $employee;
                    }

                    $writer_pool['employees'] = $employees;
                    $writer_pools[] = $writer_pool;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'pools',
            'sub_view' => 'pools',
            'writer_pools' => $writer_pools
        ]);

        $this->setTemplate('dashboard/pools/index');
        $this->display();
    }

    public function addPool() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Pools › Add New"
        ]); 

        $smarty->assign([
            'view' => 'pools',
            'sub_view' => 'add_pool',
        ]);

        $this->setTemplate('dashboard/pools/add');
        $this->display();
    }

    public function viewPool( $pool_id ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $pool = new EmployeePool((int) $pool_id);
        if(!Validate::isLoadedObject($pool)) {
            $this->setTemplate('404');
            $this->display();
        }

        $objectPresenter = new ObjectPresenter();
        $writer_pool = $objectPresenter->present($pool);

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where('employee_group = ' . (int) Employee::GROUP_WRITER );
        $sql->where('is_started = 1');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $total_writers = count($result);
        $writers = array();

        if( $total_writers > 0) {
            $params['total_items'] = $total_writers;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/pools/view/' . $pool->id . '?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach( $result as $writer ) {
                $writer = new Employee( (int) $writer['employee_id'] );
                if( Validate::isLoadedObject($writer) ) {
                    $presentedWriter = (array) $writer;
    
                    $employee_pool = new EmployeePool((int) $writer->employee_pool_id );
                    $presentedWriter['employee_pool_name'] = $employee_pool->name;
                    
                    $average_rating = $writer->getWriterAverageRating();
                    $presentedWriter['average_rating'] = $average_rating;
                    $writers[] = $presentedWriter;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }
        
        $this->setVars([
            'page_title' => "Pools › Edit"
        ]); 

        $smarty->assign([
            'view' => 'pools',
            'sub_view' => 'view_pool',
            'pool' => $writer_pool,
            'writers' => $writers
        ]);

        $this->setTemplate('dashboard/pools/view');
        $this->display();
    }
}