<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Mail;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Pager;
use Proxim\Preference\ApplicationEssay;
use Proxim\Preference\Discipline;
use Proxim\Preference\WritingGuide;
use Proxim\Preference\WritingPrompt;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class WriterApplications extends Controller {
    public function writerApplications( $status = "all" ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin && !$user->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Writer Applications"
        ]); 

        $query = ArrayUtils::has($params, 'query') ? ArrayUtils::get($params, 'query') : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where('e.employee_group = ' . (int) Employee::GROUP_APPLICANT);

        if($status == "pending") {
            $sql->where('e.is_application_rejected = 0');
            $sql->where('e.is_application_approved = 0');
        } elseif($status == "rejected") {
            $sql->where('e.is_application_rejected = 1');
        }

        if($query) {
            $sql->where('
                e.employee_id LIKE \'%' . pSQL($query) . '%\' OR
                e.first_name LIKE \'%' . pSQL($query) . '%\' OR
                e.last_name LIKE \'%' . pSQL($query) . '%\' OR
                e.email LIKE \'%' . pSQL($query) . '%\'
            ');
        }
        $sql->orderBy('e.employee_id DESC');
        
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $applications = array();
        $total_applications = count($result);

        if( $total_applications > 0) {
            $params['total_items'] = $total_applications;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/applications/?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);

            foreach( $result as $application ) {
                $application = new Employee( (int) $application['employee_id'] );
                if( Validate::isLoadedObject($application) ) {
                    $applications[] = (array) $application;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'applications',
            'sub_view' => 'applications',
            'query' => $query,
            '_status' => $status,
            'total_applications' => $total_applications,
            'applications' => $applications,
        ]);

        $this->setTemplate('dashboard/applications/applications');
        $this->display();
    }

    public function writingPrompts() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin && !$user->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Writing Prompts"
        ]); 

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $sql = new DbQuery();
        $sql->select('wp.writing_prompt_id');
        $sql->from('writing_prompt', 'wp');
        $sql->orderBy('wp.writing_prompt_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $writing_prompts = array();
        $total_writing_prompts = count($result);

        if( $total_writing_prompts > 0) {
            $params['total_items'] = $total_writing_prompts;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/applications/writing-prompt?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);

            $objectPresenter = new ObjectPresenter();
            foreach( $result as $writing_prompt ) {
                $writing_prompt = new WritingPrompt( (int) $writing_prompt['writing_prompt_id'] );
                if( Validate::isLoadedObject($writing_prompt) ) {
                    $writing_prompts[] = $objectPresenter->present($writing_prompt);
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'applications',
            'sub_view' => 'writing-prompts',
            'total_writing_prompts' => $total_writing_prompts,
            'writing_prompts' => $writing_prompts
        ]);

        $this->setTemplate('dashboard/applications/writing-prompts');
        $this->display();
    }

    public function essays() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin && !$user->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Essays"
        ]); 

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $sql = new DbQuery();
        $sql->select('ae.application_essay_id');
        $sql->from('application_essay', 'ae');
        $sql->orderBy('ae.application_essay_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $essays = array();
        $total_essays = count($result);

        if( $total_essays > 0) {
            $params['total_items'] = $total_essays;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = '/applications/essays?page=%s';

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);

            $objectPresenter = new ObjectPresenter();
            foreach( $result as $essay ) {
                $essay = new ApplicationEssay( (int) $essay['application_essay_id'] );
                if( Validate::isLoadedObject($essay) ) {
                    $essays[] = $objectPresenter->present($essay);
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'applications',
            'sub_view' => 'essays',
            'total_essays' => $total_essays,
            'essays' => $essays
        ]);

        $this->setTemplate('dashboard/applications/essays');
        $this->display();
    }

    public function settings() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if( !$user->is_admin && !$user->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Settings"
        ]); 

        $this->setTemplate('dashboard/applications/settings');
        $this->display();
    }
}