<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Pager;
use Proxim\Currency;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Order\Order;
use Proxim\Order\Rating;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Ratings extends Controller {
    public function index() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_writer) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Ratings"
        ]);

        $insights = array();

        $customer_rating = Db::getInstance()->getRow('SELECT AVG(or.paper_rating) AS avg_rating, COUNT(or.rating_id) AS total_reviews FROM '. Db::prefix('order_rating') . ' AS `or` INNER JOIN '. Db::prefix('order') . ' AS `o` ON or.order_id = o.order_id AND o.writer_id = ' . (int) $user->id . ' AND or.user_type = "' . Rating::USER_CUSTOMER . '" WHERE or.paper_rating > 0 AND or.service_rating > 0' );
        $insights['average_customer_rating'] = toFixed($customer_rating['avg_rating']/2, 1);
        $insights['total_customer_reviews'] = $customer_rating['total_reviews'];

        $editor_rating = Db::getInstance()->getRow('SELECT AVG(or.paper_rating) AS avg_rating, COUNT(or.rating_id) AS total_reviews FROM '. Db::prefix('order_rating') . ' AS `or` INNER JOIN '. Db::prefix('order') . ' AS `o` ON or.order_id = o.order_id AND o.writer_id = ' . (int) $user->id . ' AND or.user_type = "' . Rating::USER_EMPLOYEE . '" WHERE or.paper_rating > 0 AND or.service_rating > 0' );
        $insights['average_editor_rating'] = toFixed($editor_rating['avg_rating']/2, 1);
        $insights['total_editor_reviews'] = $editor_rating['total_reviews'];

        $total_reviews = $customer_rating['total_reviews'] + $editor_rating['total_reviews'];

        $sql = new DbQuery();
        $sql->select('COUNT(r.rating_id) AS total_rating, r.paper_rating');
        $sql->from('order_rating', 'r');
        $sql->innerJoin('order', 'o', 'r.order_id = o.order_id AND o.writer_id = ' . (int) $user->id );
        $sql->where('r.paper_rating > 0 AND r.service_rating > 0'); 
        $sql->groupBy('r.paper_rating'); 
        $sql->orderBy('total_rating ASC');

        $customerRatingsStats = clone $sql;
        $editorRatingsStats = clone $sql;

        // customer ratings
        $customerRatingsStats->where('r.`user_type` = \'' . pSQL(Rating::USER_CUSTOMER) . '\'');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($customerRatingsStats);
        foreach($result as $rating) {
            if(!$rating['paper_rating']) $rating['paper_rating'] = 0;
            if(!$rating['total_rating']) $rating['total_rating'] = 0;
            
            $paper_rating = toFixed($rating['paper_rating']/2, 0);
            $rating['width'] = ($rating['total_rating'] > 0 && $total_reviews > 0) ? ($rating['total_rating']/$total_reviews)*100 : 0;
            $insights['customer_ratings'][$paper_rating] = $rating;
        } 

        // editor ratings
        $editorRatingsStats->where('r.`user_type` = \'' . pSQL(Rating::USER_EMPLOYEE) . '\'');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($editorRatingsStats);
        foreach($result as $rating) {
            if(!$rating['paper_rating']) $rating['paper_rating'] = 0;
            if(!$rating['total_rating']) $rating['total_rating'] = 0;
            
            $paper_rating = toFixed($rating['paper_rating']/2, 0);
            $rating['width'] = ($rating['total_rating'] > 0 && $total_reviews > 0) ? ($rating['total_rating']/$total_reviews)*100 : 0;
            $insights['editor_ratings'][$paper_rating] = $rating;
        } 

        $sql = new DbQuery();
        $sql->select('r.rating_id');
        $sql->from('order_rating', 'r');
        $sql->innerJoin('order', 'o', 'r.order_id = o.order_id AND o.writer_id = ' . (int) $user->id );
        $sql->where('r.paper_rating > 0 AND r.service_rating > 0'); 
        $sql->orderBy('r.rating_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_ratings = count($result);

        $ratings = array();

        if( $total_ratings > 0) {
            $params['total_items'] = $total_ratings;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/ratings?page=%s";
 
            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach( $result as $rating ) {
                $rating = new Rating( (int) $rating['rating_id'] );
                if( Validate::isLoadedObject($rating) ) {
                    $ratings[] = $rating->present();
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'ratings',
            'sub_view' => 'ratings',
            'ratings' => $ratings,
            'insights' => $insights
        ]);

        $this->setTemplate('dashboard/ratings');
        $this->display();
    }
}