<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Pager;
use Proxim\Currency;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Order\PaymentRequest;
use Proxim\Order\Rating;
use Proxim\PDF\PDF;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\User\EmployeePool;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Performance extends Controller {

    public function index() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_writer && !$user->is_editor) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Performance"
        ]);

        $sql = new DbQuery();
        $sql->select('ep.employee_pool_id');
        $sql->from('employee_pool', 'ep');
        $sql->where("ep.status = 'active'");
        $sql->orderBy('ep.order ASC, ep.employee_pool_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $writer_pools = array();

        $objectPresenter = new ObjectPresenter();

        $x = $initialSlide = 0;
        foreach($result as $pool) {
            $employeePool = new EmployeePool((int) $pool['employee_pool_id']);
            if(Validate::isLoadedObject($employeePool)) {
                $writer_pool = $objectPresenter->present($employeePool);
                $writer_pool['badge_url'] = Configuration::get('UPLOADS_PATH') . '/' . $employeePool->badge;

                $writer_pool['benefits_array'] = explode(PHP_EOL, $employeePool->benefits);
                $writer_pool['requirements_array'] = explode(PHP_EOL, $employeePool->requirements);

                if($user->employee_pool_id == $employeePool->id) {
                    $initialSlide = $x;
                }

                $writer_pools[] = $writer_pool;
            }

            $x++;
        }
        
        # insights
        $insights = array();

        $sql = new DbQuery();
        $sql->from("order", "o");

        if($user->is_writer) {
            $sql->select("COALESCE(SUM(writer_pay),0) AS pay, o.date_add");
            $sql->where("writer_id = " . (int) $user->id );
            $sql->where("is_writer_paid = 1");
        } elseif ($user->is_editor) {
            $sql->select("COALESCE(SUM(editor_pay),0) AS pay, o.date_add");
            $sql->where("editor_id = " . (int) $user->id );
            $sql->where("is_editor_paid = 1");
        } elseif ($user->is_sub_admin) {
            $sql->select("COALESCE(SUM(ordermanager_pay),0) AS pay, o.date_add");
            $sql->where("ordermanager_id = " . (int) $user->id );
            $sql->where("is_ordermanager_paid = 1");
        }

        $insights['total_income'] = Db::getInstance()->getValue($sql);

        /* orders chart */
        $sql = new DbQuery();
        $sql->select("COUNT(o.order_id)")->from("order", "o");

        if($user->is_writer) {
            $sql->where("writer_id = " . (int) $user->id);
        } elseif ($user->is_editor) {
            $sql->where("editor_id = " . (int) $user->id);
        } elseif ($user->is_sub_admin) {
            $sql->where("ordermanager_id = " . (int) $user->id);
        }

        $approved_orders_urgent_query = clone $sql;
        $approved_orders_pages_query = clone $sql;
        $approved_orders_total_query = clone $sql;

        $approved_orders_total_query->where('status_id = ' . (int) Order::FINISHED );
        $insights['approved_orders_total'] = Db::getInstance()->getValue($approved_orders_total_query);

        $approved_orders_urgent_query->where('status_id = ' . (int) Order::FINISHED );
        $approved_orders_urgent_query->where('hrs_customer <= 8');
        $insights['approved_orders_urgent'] = Db::getInstance()->getValue($approved_orders_urgent_query);

        $approved_orders_pages_query->where('status_id = ' . (int) Order::FINISHED );
        $approved_orders_pages_query->where('pages >= 15');
        $insights['approved_orders_pages'] = Db::getInstance()->getValue($approved_orders_pages_query);

        /** performance overview */
        $sql = new DbQuery();
        $sql->select("COUNT(DISTINCT(oe.order_id))")->from("order_employee", "oe");
        $sql->where("employee_id = " . (int) $user->id ); 

        $late_orders_query = clone $sql;
        $revision_orders_query = clone $sql;
        $confirmed_orders_query = clone $sql;
        $declined_orders_query = clone $sql;
        $total_orders_query = clone $sql;

        $total_orders_query->where('is_employee_confirmed = 1');
        $total_orders = Db::getInstance()->getValue($total_orders_query);
        $insights['total_orders'] = $total_orders;

        $late_orders_query->where('is_late = 1');
        $late_orders = Db::getInstance()->getValue($late_orders_query);

        if ($total_orders == 0) {
            $total_orders++;
        }

        $insights['lateness_level'] = ($late_orders / $total_orders) * 100;

        $revision_orders_query->where('is_revised = 1');
        $revision_orders = Db::getInstance()->getValue($revision_orders_query);

        $insights['revision_orders'] = $revision_orders;
        $insights['revision_rate'] = ($revision_orders / $total_orders) * 100;

        $confirmed_orders_query->where('is_auto_assigned = 1');
        $confirmed_orders_query->where('is_employee_confirmed = 1');

        $confirmed_orders = Db::getInstance()->getValue($confirmed_orders_query);
        $insights['confirmed_orders'] = $confirmed_orders;

        $declined_orders_query->where('is_auto_assigned = 1');
        $declined_orders_query->where('is_employee_declined = 1');
        $declined_orders = Db::getInstance()->getValue($declined_orders_query);
        $insights['declined_orders'] = $declined_orders;

        if ($confirmed_orders == 0) {
            $declined_orders++;
        }

        $insights['confirmation_rate'] = ($confirmed_orders / ($confirmed_orders + $declined_orders)) * 100;

        $customer_rating = Db::getInstance()->getRow('SELECT AVG(or.paper_rating) AS avg_rating, COUNT(or.rating_id) AS total_reviews FROM '. Db::prefix('order_rating') . ' AS `or` INNER JOIN '. Db::prefix('order') . ' AS `o` ON or.order_id = o.order_id AND o.writer_id = ' . (int) $user->id . ' AND or.user_type = "' . Rating::USER_CUSTOMER . '" WHERE or.paper_rating > 0 AND or.service_rating > 0' );
        $insights['average_customer_rating'] = toFixed($customer_rating['avg_rating']/2, 1);
        $insights['total_customer_reviews'] = $customer_rating['total_reviews'];

        $editor_rating = Db::getInstance()->getRow('SELECT AVG(or.paper_rating) AS avg_rating, COUNT(or.rating_id) AS total_reviews FROM '. Db::prefix('order_rating') . ' AS `or` INNER JOIN '. Db::prefix('order') . ' AS `o` ON or.order_id = o.order_id AND o.writer_id = ' . (int) $user->id . ' AND or.user_type = "' . Rating::USER_EMPLOYEE . '" WHERE or.paper_rating > 0 AND or.service_rating > 0' );
        $insights['average_editor_rating'] = toFixed($editor_rating['avg_rating']/2, 1);
        $insights['total_editor_reviews'] = $editor_rating['total_reviews'];

        $total_reviews = $customer_rating['total_reviews'] + $editor_rating['total_reviews'];

        $sql = new DbQuery();
        $sql->select('COUNT(r.rating_id) AS total_rating, r.paper_rating');
        $sql->from('order_rating', 'r');
        $sql->innerJoin('order', 'o', 'r.order_id = o.order_id AND o.writer_id = ' . (int) $user->id );
        $sql->where('r.paper_rating > 0 AND r.service_rating > 0'); 
        $sql->groupBy('r.paper_rating'); 
        $sql->orderBy('total_rating ASC');

        $customerRatingsStats = clone $sql;
        $editorRatingsStats = clone $sql;

        // customer ratings
        $customerRatingsStats->where('r.`user_type` = \'' . pSQL(Rating::USER_CUSTOMER) . '\'');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($customerRatingsStats);
        foreach($result as $rating) {
            if(!$rating['paper_rating']) $rating['paper_rating'] = 0;
            if(!$rating['total_rating']) $rating['total_rating'] = 0;
            
            $paper_rating = toFixed($rating['paper_rating']/2, 0);
            $rating['width'] = ($rating['total_rating'] > 0 && $customer_rating['total_reviews'] > 0) ? ($rating['total_rating']/$customer_rating['total_reviews'])*100 : 0;
            $insights['customer_ratings'][$paper_rating] = $rating;
        } 

        // editor ratings
        $editorRatingsStats->where('r.`user_type` = \'' . pSQL(Rating::USER_EMPLOYEE) . '\'');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($editorRatingsStats);
        foreach($result as $rating) {
            if(!$rating['paper_rating']) $rating['paper_rating'] = 0;
            if(!$rating['total_rating']) $rating['total_rating'] = 0;
            
            $paper_rating = toFixed($rating['paper_rating']/2, 0);
            $rating['width'] = ($rating['total_rating'] > 0 && $editor_rating['total_reviews'] > 0) ? ($rating['total_rating']/$editor_rating['total_reviews'])*100 : 0;
            $insights['editor_ratings'][$paper_rating] = $rating;
        } 

        $ratings = $user->getReviews();

        $is_fines_enabled = false;
        if(Module::isInstalled('creditfine')) {
            $is_fines_enabled = true;
            
            $sql = new DbQuery();
            $sql->select('ft.*');
            $sql->from('fine_type', 'ft');
            $fines = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

            $max_results = Configuration::get('MAX_RESULTS', null, 10);
            $sql = new DbQuery();
            $sql->select('cf.*');
            $sql->from('credit_fine', 'cf');
            $sql->where('cf.type = "fine"');
            $sql->where('cf.employee_id = ' . (int) $user->id );
            $sql->where('cf.order_id IS NOT NULL');
            $sql->orderBy('cf.credit_fine_id DESC');
            $result = Db::getInstance()->executeS($sql);
            $total_charged_fines = count($result);

            $charged_fines = array();
            if( $total_charged_fines > 0) {
                $sql->limit( $max_results);
                $result = Db::getInstance()->executeS($sql);

                foreach($result as $res) {
                    $res['desc_trucnate'] = Tools::truncateString($res['description'], 30);
                    $charged_fines[] = $res;
                }
            }

            $smarty->assign([
                'fines' => $fines,
                'charged_fines' => $charged_fines,
                'total_charged_fines' => $total_charged_fines
            ]);
        }

        $smarty->assign([
            'view' => 'performance',
            'is_fines_enabled' => $is_fines_enabled,
            'insights' => $insights,
            'ratings' => $ratings,
            'initialSlide' => $initialSlide,
            'writer_pools' => $writer_pools
        ]);

        $this->setTemplate('dashboard/performance/index');
        $this->display(); 
    }

}