<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Pager;
use Proxim\Currency;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Order\PaymentRequest;
use Proxim\PDF\PDF;
use Proxim\User\Employee;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Payments extends Controller {

    public function financialOverview( $sub_view = "unrequested" ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if($user->is_sub_admin && !Configuration::get('ORDER_MANAGER_ENABLED')) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Finacial Overview"
        ]);

        $currencyFormat = $app->writer_currency;

        $can_request_payment = false;
        $payment_ordinal_dates = array();
        $payment_dates = Configuration::get('PAYMENT_DATES', null, array());
        if(!empty($payment_dates)) {
            $payment_dates = explode(',', $payment_dates);
            $current_day = date("d");

            $can_request_payment = in_array($current_day, $payment_dates);

            foreach($payment_dates as $payment_date) {
                $payment_ordinal_dates[] = dateOrdinal($payment_date);
            }
        }

        switch( $sub_view ) {

            case 'unrequested':

                $sql = new DbQuery();
                $sql->select('o.*');
                $sql->from('order', 'o');

                if ( $user->is_writer ) {
                    $sql->where('o.`writer_id` = ' . (int) $user->id );
                    $sql->where('o.`is_writer_paid` = 0' );
                } elseif ( $user->is_editor ) {
                    $sql->where('o.`editor_id` = ' . (int) $user->id );
                    $sql->where('o.`is_editor_paid` = 0' );
                } elseif ( $user->is_sub_admin ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                    $sql->where('o.`is_ordermanager_paid` = 0' );
                }

                if( Configuration::get('PAY_DELIVERED_ORDERS') ) {
                    $sql->where( sprintf('o.status_id = %s OR o.status_id = %s',Order::DELIVERED, Order::FINISHED) );
                } else {
                    $sql->where( sprintf('o.status_id = %s', Order::FINISHED) );
                }

                $sql->leftJoin('payment_request_order', 'pro', 'pro.order_id = o.order_id AND pro.user_id = ' . (int) $user->id );
                $sql->where( 'pro.order_id IS NULL' );
                $sql->orderBy( 'o.finished_at DESC' );

                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_orders = count($result);

                $orders = array();
                $subtotal_pay = $total_pay = $completed_orders_pay = $done_orders_pay = $employee_credit = $employee_fine = 0;
                $has_employee_credit = $has_employee_fine = false;

                if( $total_orders > 0) {
                    $params['total_items'] = $total_orders;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/financial-overview/unrequested?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
        
                    foreach( $result as $order ) {
                        $order = new Order( (int) $order['order_id'] );
                        if( Validate::isLoadedObject($order) ) {
                            if($order->status_id == Order::FINISHED) {
                                if($user->is_sub_admin) {
                                    $completed_orders_pay += $order->ordermanager_pay;
                                } elseif($user->is_editor) {
                                    $completed_orders_pay += $order->editor_pay;
                                } elseif($user->is_writer) {
                                    $completed_orders_pay += $order->writer_pay;
                                }
                            } elseif ( $order->status_id == Order::DELIVERED ) {
                                if($user->is_sub_admin) {
                                    $done_orders_pay += $order->ordermanager_pay;
                                } elseif($user->is_editor) {
                                    $done_orders_pay += $order->editor_pay;
                                } elseif($user->is_writer) {
                                    $done_orders_pay += $order->writer_pay;
                                }
                            }

                            $orders[] = $order->present(false);
                        }
                    }
        
                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $total_pay = $completed_orders_pay + $done_orders_pay;
                $subtotal_pay = $total_pay;

                $additionalPaymentsByModule = Hook::exec('actionGetEmployeeAdditionalPayments', array(
                    'employee_id' => $user->id,
                    'requested' => false,
                    'resolved' => false
                ), null, true);

                if(is_array($additionalPaymentsByModule) && !empty($additionalPaymentsByModule)) {
                    foreach($additionalPaymentsByModule as $moduleName => $additionalPayments) {
                        foreach($additionalPayments as $additionalPayment) {
                            if($additionalPayment['type'] == 'credit') {
                                $has_employee_credit = true;
                                $employee_credit += (float) $additionalPayment['amount'];
                            } elseif($additionalPayment['type'] == 'fine') {
                                $has_employee_fine = true;
                                $employee_fine += (float) $additionalPayment['amount'];
                            }
                        }
                    }
                }

                $sql = new DbQuery();
                $sql->select('payment_request_id');
                $sql->from('payment_request');
                $sql->where('status = 0');
                $sql->where('user_id = ' . (int) $user->id );
                $existing_payment_request = (bool) Db::getInstance()->getValue($sql);

                $total_pay = ($total_pay + $employee_credit) - $employee_fine;
                $minimum_payment_request = 0;
                if ( $user->is_editor ) {
                    $minimum_payment_request = Configuration::get('EDITOR_PAYMENT_REQUEST_LIMIT');
                } else {
                    $minimum_payment_request = Configuration::get('WRITER_PAYMENT_REQUEST_LIMIT');
                } 

                $can_request_payment = !$existing_payment_request && ($can_request_payment && $total_pay > $minimum_payment_request) ? true: false;

                $smarty->assign([
                    'has_employee_credit' => $has_employee_credit,
                    'employee_credit' => $employee_credit,
                    'has_employee_fine' => $has_employee_fine,
                    'employee_fine' => $employee_fine,
                    'total_orders' => $total_orders,
                    'done_orders_pay' => $done_orders_pay,
                    'done_orders_pay_formatted' => formatPrice($done_orders_pay, $currencyFormat),
                    'completed_orders_pay' => $completed_orders_pay,
                    'completed_orders_pay_formatted' => formatPrice($completed_orders_pay, $currencyFormat),
                    'subtotal_pay' => $subtotal_pay,
                    'subtotal_pay_formatted' => formatPrice($subtotal_pay, $currencyFormat),
                    'total_pay' => $total_pay,
                    'total_pay_formatted' => formatPrice($total_pay, $currencyFormat),
                    'orders' => $orders,
                ]);

                break;

            case 'requested':
                
                $sql = new DbQuery();
                $sql->select('pr.*');
                $sql->from('payment_request', 'pr');
                $sql->where('pr.`user_id` = ' . (int) $user->id );
                $sql->where('pr.`status` = 0');
                $sql->orderBy('pr.`payment_request_id` DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_requests = count($result);
                
                $requests = array();
                if( $total_requests > 0) {
                    $params['total_items'] = $total_requests;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/financial-overview/unrequested?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $requests = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_requests' => $total_requests,
                    'requests' => $requests,
                ]);

                break;
            
            case 'history':
                $sql = new DbQuery();
                $sql->select('pr.*');
                $sql->from('payment_request', 'pr');
                $sql->where('pr.`user_id` = ' . (int) $user->id );
                $sql->where('pr.`status` = 1 OR pr.`status` = -1');
                $sql->orderBy('pr.`payment_request_id` DESC');
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_requests = count($result);
                
                $requests = array();
                if( $total_requests > 0) {
                    $params['total_items'] = $total_requests;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/financial-overview/history?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $requests = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_requests' => $total_requests,
                    'requests' => $requests,
                ]);

                break;

            default:
                $hookNavContentExists = Hook::getIdByName('displayFinancialOverviewNavContent');
                $hookNavbarExists = Hook::getIdByName('displayFinancialOverviewNavbar');

                if(!$hookNavbarExists && !$hookNavContentExists) {
                    $this->setTemplate('404');
                    $this->display();
                }
                
                break;
        }

        $smarty->assign([
            'view' => 'financial-overview',
            'sub_view' => $sub_view,
            'can_request_payment' => $can_request_payment,
            'payment_dates' => implode(', ', $payment_ordinal_dates),
            'currencyFormat' => $currencyFormat
        ]);

        $this->setTemplate('dashboard/financial-overview');
        $this->display(); 
    }

    public function payments( $sub_view = "overview" ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $selected_page = (int) ArrayUtils::get($params, 'page', 1);
        $date_from = ArrayUtils::get($params, 'from');
        $date_to = ArrayUtils::get($params, 'to');
        $employee_group = ArrayUtils::get($params, 'employee', "all");

        $this->setVars([
            'page_title' => "Payments"
        ]);

        $currencyFormat = $app->writer_currency;

        switch( $sub_view ) {

            case 'overview':
                $payment_overview = array();

                $sql = new DbQuery();
                $sql->select('e.employee_id, e.employee_group, e.first_name, e.last_name, e.email, e.avatar, e.gender');
                $sql->from('employee', 'e');
                
                if ($employee_group == "all") {
                    $sql->where('e.employee_group = ' . (int) Employee::GROUP_EDITOR . ' OR e.employee_group = ' . (int) Employee::GROUP_WRITER );

                    if(Configuration::get('ORDER_MANAGER_ENABLED')) {
                        $sql .= ' OR e.employee_group = ' . (int) Employee::GROUP_SUB_ADMIN . ' ';
                    }
                } elseif ( $employee_group == "writer" ) {
                    $sql->where('e.employee_group = ' . (int) Employee::GROUP_WRITER );
                } elseif ( $employee_group == "editor" ) {
                    $sql->where('e.employee_group = ' . (int) Employee::GROUP_EDITOR );
                } elseif ( $employee_group == "ordermanager" ) {
                    $sql->where('e.employee_group = ' . (int) Employee::GROUP_SUB_ADMIN );
                }

                $result =  Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_stats = count($result);

                if( $total_stats > 0) {
                    $params['total_items'] = $total_stats;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/payments/overview?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    foreach ($result as $employee ) {
                        $sql = new DbQuery();
                        if ( $employee['employee_group'] == Employee::GROUP_WRITER ) {
                            $sql->select('COUNT(o.order_id) AS total_orders, COALESCE(SUM(o.pages),0) AS total_pages, COALESCE(SUM(o.words),0) AS total_words, COALESCE(SUM(o.writer_pay),0) AS employee_pay');
                        } elseif ( $employee['employee_group'] == Employee::GROUP_EDITOR ) {
                            $sql->select('COUNT(o.order_id) AS total_orders, COALESCE(SUM(o.pages),0) AS total_pages, COALESCE(SUM(o.words),0) AS total_words, COALESCE(SUM(o.editor_pay),0) AS employee_pay');
                        } elseif ( $employee['employee_group'] == Employee::GROUP_SUB_ADMIN ) {
                            $sql->select('COUNT(o.order_id) AS total_orders, COALESCE(SUM(o.pages),0) AS total_pages, COALESCE(SUM(o.words),0) AS total_words, COALESCE(SUM(o.ordermanager_pay),0) AS employee_pay');
                        }
                        $sql->from('order', 'o');
                        
                        if ( $employee['employee_group'] == Employee::GROUP_WRITER ) {
                            $sql->where('o.`writer_id` = ' . (int) $employee['employee_id'] );
                            $sql->where('o.`writer_assigned` = 1');
                            $sql->where('o.`is_writer_paid` = 0' );
                        } elseif ( $employee['employee_group'] == Employee::GROUP_EDITOR ) {
                            $sql->where('o.`editor_id` = ' . (int) $employee['employee_id'] );
                            $sql->where('o.`editor_assigned` = 1');
                            $sql->where('o.`is_editor_paid` = 0' );
                        } elseif ( $employee['employee_group'] == Employee::GROUP_SUB_ADMIN ) {
                            $sql->where('o.`ordermanager_id` = ' . (int) $employee['employee_id'] );
                            $sql->where('o.`ordermanager_assigned` = 1');
                            $sql->where('o.`is_ordermanager_paid` = 0' );
                        }

                        if( Configuration::get('PAY_DELIVERED_ORDERS') ) {
                            $sql->where( sprintf('o.status_id = %s OR o.status_id = %s',Order::DELIVERED, Order::FINISHED) );
                        } else {
                            $sql->where( sprintf('o.status_id = %s', Order::FINISHED) );
                        }

                        if ($date_from || $date_to) {
                            if ( $date_from ) {
                                $dateFromSQL = DateUtils::date(strtotime($date_from));
                            } else {
                                $dateFromSQL = DateUtils::date( strtotime('-14 days') );
                            }

                            if ( $date_to ) {
                                $dateToSQL = DateUtils::date(strtotime($date_to));
                            } else {
                                $dateToSQL = DateUtils::date( strtotime('+14 days') );
                            }
                
                            $sql->where('o.`date_add` BETWEEN "' . pSQL($dateFromSQL) . '" AND "' . pSQL($dateToSQL) . '"');
                        }

                        $employee['user_picture_default'] = !$employee['avatar'] ? true : false;
                        $employee['user_picture'] = $user->getPicture($employee['avatar'], $employee['gender']);

                        $overview = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
                        $payment_overview[] = array_merge( $employee, $overview );
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }
                
                $smarty->assign([
                    'total_stats' => $total_stats,
                    'payment_overview' => $payment_overview,
                    'date_from' => $date_from,
                    'date_to' => $date_to,
                    'employee_group' => $employee_group,
                    'selected_page' => $selected_page
                ]);

                break;

            case 'requests':
                $sql = new DbQuery();
                $sql->select('pr.*');
                $sql->from('payment_request', 'pr');
                $sql->where('status = 0');
                $sql->orderBy( 'pr.payment_request_id DESC' );
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_requests = count($result);
                
                $requests = array();
                if( $total_requests > 0) {
                    $params['total_items'] = $total_requests;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/payments/requests?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    foreach( $result as $request ) {
                        $employee = new Employee( (int) $request['user_id'] );
                        if(Validate::isLoadedObject($employee)) {
                            $request['employee'] = (array) $employee;
                            $request['request_amount_intl'] = $request['total_pay'];
                            if(Configuration::get('MPESA_USD_KES_RATE')) {
                                $usedToKes = (int) Configuration::get('MPESA_USD_KES_RATE', PROX_SITE_ID, 100);
                                $request['request_amount_intl'] = $request['total_pay']*$usedToKes;
                            }

                            $requests[] = $request;
                        }
                        
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_requests' => $total_requests,
                    'requests' => $requests,
                ]);

                break;

            case 'history':
                $sql = new DbQuery();
                $sql->select('pr.*, e.first_name, e.last_name, e.employee_group');
                $sql->from('payment_request', 'pr');
                $sql->innerJoin('employee', 'e', 'e.`employee_id` = pr.`user_id`');
                $sql->where('pr.`status` = 1 OR pr.`status` = -1');
                $sql->orderBy( 'pr.payment_request_id DESC' );
                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_requests = count($result);
                
                $requests = array();
                if( $total_requests > 0) {
                    $params['total_items'] = $total_requests;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/payments/history?page=%s";
        
                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();
                    $requests = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                $smarty->assign([
                    'total_requests' => $total_requests,
                    'requests' => $requests,
                ]);
                
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;

        }

        $smarty->assign([
            'view' => 'payments',
            'sub_view' => $sub_view,
            'currencyFormat' => $currencyFormat
        ]);

        $this->setTemplate('dashboard/payments');
        $this->display(); 
    }

    public function downloadPaymentRequests( $request_id ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        $paymentRequest = new PaymentRequest( (int) $request_id );

        if (!Validate::isLoadedObject($paymentRequest)) {
	        $this->setTemplate('404');
            $this->display();
        }

        $paymentRequestInfo = $paymentRequest->getPaymentRequestInfo();

        $pdf = new PDF($paymentRequest, 'PaymentRequestInvoice', $smarty );
        $pdf->render();
    }
}