<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Mail;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Pager;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Order\Message;
use Proxim\Order\Bid;
use Proxim\Order\AdditionalService;
use Proxim\Order\Revision;
use Proxim\Order\File;
use Proxim\Order\Rating;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\Deadline;
use Proxim\Preference\Descriptor;
use Proxim\Preference\Discipline;
use Proxim\Preference\ItemPricing;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Deadline\DeadlinePresenter;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Site\Site;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Orders extends Controller {
    public function getOrders( $status = 'available' ) {
        $app = $this->app;
        $payload = $app->request->get();
        $user = $app->user;
        $smarty = $app->smarty;

        // $time_start = microtime(true); 

        /** Search */
        $corslevel = (int) ArrayUtils::get($payload, 'corslevel');
        $discipline_ids = ArrayUtils::get($payload, 'discipline');
        $deadline = ArrayUtils::get($payload, 'deadline');
        $pages = ArrayUtils::get($payload, 'pages');
        $cost = ArrayUtils::get($payload, 'cost');
        $order_id = ArrayUtils::get($payload, 'order_id', null);
        $status_id = ArrayUtils::get($payload, 'status_id', null);
        $toptitle = ArrayUtils::get($payload, 'toptitle');
        $trackingid = ArrayUtils::get($payload, 'trackingid');
        $is_search = ArrayUtils::has($payload, 'search') ? true : false;

        if(ArrayUtils::has($payload, 'ajaxpage')) {
            ArrayUtils::remove($payload, 'ajaxpage');
        }

        if (!is_array($discipline_ids)) {
            $discipline_ids = array();
        }

        if(ArrayUtils::has($payload, 'discipline')) {
            ArrayUtils::remove($payload, 'discipline');
        }

        if(ArrayUtils::has($payload, 'page')) {
            $selected_page = (int) ArrayUtils::get($payload, 'page');
            ArrayUtils::remove($payload, 'page');
        } else {
            $selected_page = 1;
        }

        $sql = new DbQuery();
        $sql->select('DISTINCT o.*');
        $sql->from('order', 'o');

        $orderAccess = array();
        if($user->has_academic_access) $orderAccess[] = 'o.`service_type` = ' . (int) Order::COMPLEX_ASSIGNMENT;
        if($user->has_article_access) $orderAccess[] = 'o.`service_type` = ' . (int) Order::ARTICLE_WRITING;
        if($user->has_programming_access) $orderAccess[] = 'o.`service_type` = ' . (int) Order::PROGRAMMING;
        if($user->has_calculations_access) $orderAccess[] = 'o.`service_type` = ' . (int) Order::CALCULATIONS;

        // subadmin && admin order access
        if ( ($user->is_admin && !$user->is_superadmin) || $user->is_sub_admin) {
            $sql->where(implode(' OR ', $orderAccess));
        }

        if ($is_search) {
            if ($order_id) {
                $sql->where('o.order_id LIKE \'%' . pSQL($order_id) . '%\' OR o.tracking_id LIKE \'%' . pSQL($order_id) . '%\'');
            } 

            if ($trackingid) {
                $sql->where('o.tracking_id LIKE \'%' . pSQL($trackingid) . '%\'');
            } 
        
            if($corslevel > 0 ) {
                $sql->where( 'o.academic_level_id = ' . (int) $corslevel );
            }

            if($status_id) {
                $sql->where( 'o.status_id = ' . (int) $status_id );
            }

            if( $pages == "1" ) {
                $sql->where( 'o.pages < 5' );
            } elseif($pages == "2") {
                $sql->where( 'o.pages >= 5 AND o.pages <= 10' );
            } elseif($pages == "3") {
                $sql->where('o.pages >= 10 AND o.pages <= 20');
            } elseif($pages == "4") {
                $sql->where('o.pages >= 20 AND o.pages <= 30');
            } elseif($pages == "5") {
                $sql->where( 'o.pages > 30' );
            }

            if(is_array($discipline_ids) && !empty($discipline_ids)) {
                $disciplineQ = implode(',', $discipline_ids);
                $sql->where( "o.topic_category_id IN ($disciplineQ)" );
            }

            if ($user->is_writer) {
                $cost_row = 'o.writer_pay';
            } elseif ($user->is_editor) {
                $cost_row = 'o.editor_pay';
            } else {
                $cost_row = 'o.total';
            }

            if( $cost == "1" ) {
                $sql->where( "$cost_row <= 10" );
            } elseif($cost == "2") {
                $sql->where( "$cost_row >= 10 AND $cost_row <= 30" );
            } elseif($cost == "3") {
                $sql->where( "$cost_row >= 30 AND $cost_row <= 100" );
            } elseif($cost == "4") {
                $sql->where( "$cost_row >= 100 AND $cost_row <= 500" );
            } elseif($cost == "5") {
                $sql->where( "$cost_row > 500" );
            }

            if ($user->is_writer) {
                $deadline_row = 'o.writer_deadline';
            } else {
                $deadline_row = 'o.customer_deadline';
            }

            if( $deadline == "1" ) {
                $sql->where( "$deadline_row <= DATE_ADD(NOW(), INTERVAL 24 HOUR)" );
            } elseif($deadline == "2") {
                $sql->where( "$deadline_row >= DATE_ADD(NOW(), INTERVAL 24 HOUR) AND $deadline_row <= DATE_ADD(NOW(), INTERVAL 72 HOUR)" );
            } elseif($deadline == "3") {
                $sql->where( "$deadline_row >= DATE_ADD(NOW(), INTERVAL 96 HOUR) AND $deadline_row <= DATE_ADD(NOW(), INTERVAL 168 HOUR)" );
            } elseif($deadline == "4") {
                $sql->where( "$deadline_row > DATE_ADD(NOW(), INTERVAL 168 HOUR)" );
            }

            if ($toptitle) {
                $sql->where('o.title LIKE \'%' . pSQL($toptitle) . '%\'');
            }
        }

        if ($user->is_writer && !in_array($status, ['available', 'bids', 'cancelled']) ) {

            $sql->where('o.`writer_id` = ' . (int) $user->id );
            $sql->where( 'o.writer_assigned = 1' );

        } elseif ( $user->is_editor && !in_array($status, ['available']) ) {

            $sql->where('o.`editor_id` = ' . (int) $user->id );
            $sql->where( 'o.editor_assigned = 1' );

        }

        $this->setVars([
            'page_title' => "Orders › " . Tools::ucfirst($status)
        ]);

        $statuses = array();
        switch( $status ) {
            case 'all':
                if (!$user->is_admin && !$user->is_sub_admin) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $total_orders = 0;
                foreach($user->order_stats as $stat_status => $stat_num) {
                    $total_orders += $stat_num;
                }
                break;

            case 'unpaid':
                if (!$user->is_admin && !$user->is_sub_admin) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $statuses = array(
                    Order::FAKE,
                );

                $total_orders = $user->order_stats['unpaid'];
                break;

            case 'inquiry':
                if (!$user->is_admin && !$user->is_sub_admin) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $statuses = array(
                    Order::FREE_INQUIRY
                );

                $total_orders = $user->order_stats['inquiry'];
                break;


            case 'new-paid':
                if (!$user->is_admin && !$user->is_sub_admin) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $statuses = array(
                    Order::NEW_PAID,
                );

                $total_orders = $user->order_stats['new_paid'];
                break;

            case 'bids':
                $statuses = array(
                    Order::ORDER_PUBLISHED,
                    Order::WRITER_ACCEPTED
                );

                if ( $user->is_admin || $user->is_sub_admin ) {
                    $sql->innerJoin('order_bid', 'bid', 'o.order_id = bid.order_id' );
                } else {
                    $sql->innerJoin('order_bid', 'bid', 'o.order_id = bid.order_id AND bid.writer_id = ' . $user->id );
                }

                $total_orders = $user->order_stats['bids'];
                break; 

            case 'available':
                if ( $user->is_writer || $user->is_editor) {
                    $sql->where(implode(' OR ', $orderAccess));
                } 

                $orderRules = $user->getEmployeeOrderRules(); 
                $urgentDeadlineHours = (int) Configuration::get('DEADLINE_DUE_HOURS', null, 12);

                if ($user->is_writer) { 
                    $statuses = array(
                        Order::ORDER_PUBLISHED,
                        Order::WRITER_ACCEPTED,
                        Order::INQUERY_PUBLISHED
                    );

                    $sql->leftJoin('order_bid', 'bid', 'o.order_id = bid.order_id AND bid.writer_id = ' . (int) $user->id );
                    $sql->where( 'bid.bid_id IS NULL' );

                    $sql->where( 'o.academic_level_id <= ' . (int) $orderRules['academic_level'] );
                    $sql->where( 'o.writer_category_id <= ' . (int) $orderRules['writer_category'] );

                    if(!$orderRules['urgent_orders']) {
                        $sql->where('o.writer_deadline >=  DATE_ADD(NOW(), INTERVAL ' . (int) $urgentDeadlineHours . ' HOUR)');
                    }

                    if(!$orderRules['new_customers']) {
                        $sql->leftJoin('customer', 'c', 'o.customer_id = c.customer_id' );
                        $sql->where( 'c.total_orders >= 2' );
                    }
                    
                } elseif ( $user->is_editor ) {
                    $statuses = array(
                        Order::DONE
                    );
                    $sql->where( 'o.editor_assigned = 0' );
                    $sql->where( 'o.academic_level_id <= ' . (int) $orderRules['academic_level'] );
                    $sql->where( 'o.writer_category_id <= ' . (int) $orderRules['writer_category'] );

                } else {
                    $statuses = array(
                        Order::ORDER_PUBLISHED,
                        Order::INQUERY_PUBLISHED,
                        Order::WRITER_ACCEPTED
                    );

                    $sql->leftJoin('order_bid', 'bid', 'o.order_id = bid.order_id' );
                    $sql->where( 'bid.bid_id IS NULL' );
                } 

                $total_orders = $user->order_stats['available'];
                break;

            case 'unconfirmed':
                if ( $user->is_editor ) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $statuses = array(
                    Order::WRITER_ASSIGNED
                );

                $sql->where( 'o.is_writer_confirmed = 0' );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['unconfirmed'];
                break;

            case 'progress':
                if ( $user->is_editor ) {
                    $statuses = array(
                        Order::DONE,
                    );
                } else {
                    $statuses = array(
                        Order::WRITER_ASSIGNED,
                    );
                    $sql->where( 'o.is_writer_confirmed = 1' );
                }

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['progress'];
                break;

            case 'done':
                $statuses = array(
                    Order::DONE
                );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['done'];
                break;

            case 'delivered':
                $statuses = array(
                    Order::DELIVERED,
                );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['delivered'];
                break;

            case 'dispute':
                $statuses = array(
                    Order::DISPUTE
                );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['disputes'];
                break;

            case 'revision':
                $statuses = array(
                    Order::REVISION,
                );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['revision'];
                break;

            case 'finished':
                $statuses = array(
                    Order::FINISHED,
                );

                if ( $user->is_sub_admin && (Configuration::get('ORDER_MANAGER_ENABLED') && !Configuration::get('ORDER_MANAGER_PUBLIC')) ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $user->id );
                }

                $total_orders = $user->order_stats['finished'];
                break;

            case 'cancelled':

                if($user->is_writer) {
                    $sql->innerJoin('order_employee', 'oe', 'oe.order_id = o.order_id' );
                    $sql->where('oe.`employee_type` = \'' . pSQL('writer') . '\'');
                    $sql->where('oe.`employee_id` = ' . (int) $user->id );
                    $sql->where( 'o.writer_id != ' . (int) $user->id );
                } elseif($user->is_editor) {
                    $sql->innerJoin('order_employee', 'oe', 'oe.order_id = o.order_id' );
                    $sql->where('oe.`employee_type` = \'' . pSQL('editor') . '\'');
                    $sql->where('oe.`employee_id` = ' . (int) $user->id );
                    $sql->where( 'o.editor_id != ' . (int) $user->id );
                } else {
                    $statuses = array( Order::CANCELLED );
                }

                $total_orders = $user->order_stats['cancelled'];
                break;
            default: 
                $this->setTemplate('404');
                $this->display();
                break;
        }

        if(!empty($statuses)) {
            $statuses = implode(',', $statuses);
            $sql->where("o.`status_id` IN ( $statuses )");
        }

        $sql->orderBy('o.order_id DESC'); 

        // urgent orders in available tab
        $urgentOrders = $urgentOrdersIds = array();
        $urgentQuery = clone $sql;
        if($status == 'available') {
            $deadlineDueHours = Configuration::get('DEADLINE_DUE_HOURS', PROX_SITE_ID, 12);

            if ($user->is_writer) {
                $urgentQuery->where('o.writer_deadline <=  DATE_ADD(NOW(), INTERVAL ' . (int) $deadlineDueHours . ' HOUR)');
            } else {
                $urgentQuery->where('o.customer_deadline <=  DATE_ADD(NOW(), INTERVAL ' . (int) $deadlineDueHours . ' HOUR)');
            } 

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($urgentQuery);
            foreach($result as $order) {
                $urgentOrdersIds[$order['order_id']] = $order['order_id'];
            }
        }

        // $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        // $total_orders = count($result);

        $orders = $params = array(); 

        if( $total_orders > 0) {
            $params['total_items'] = $total_orders;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/orders/$status?page=%s";

            if( \is_array($payload) && !empty($payload)) {
                $params['url'] .= "&" . http_build_query($payload);
            }

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach( $result as $order ) {
                $o = new Order();
                $order = $o->getObject( (array) $order );
                if( Validate::isLoadedObject($order) ) {
                    if(!empty($urgentOrdersIds) && ArrayUtils::has($urgentOrdersIds, $order->id)) {
                        $urgentOrders[] = $order->present();
                    } else {
                        $orders[] = $order->present();
                    }
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $disciplines = Discipline::getDisciplines();
        $groupedDisciplineGroups = ArrayUtils::groupByKey($disciplines, 'group_id');

        $disciplineGroups = array();

        if (count($groupedDisciplineGroups) > 0) {
            foreach($groupedDisciplineGroups as $disciplineGroupId => $disciplineGroupItems) {
                $xDiscipline = array();
                foreach( $disciplineGroupItems as $x ) {
                    $x['selected'] = \in_array($x['discipline_id'], $discipline_ids);
                    $xDiscipline[] = $x;
                }

                $discipline = new Discipline( (int) $disciplineGroupId );
                if(Validate::isLoadedObject($discipline)) {
                    $disciplineGroups[] = array(
                        'id' => $discipline->id,
                        'title' => $discipline->title,
                        'disciplines' => $xDiscipline
                    );
                } else {
                    $disciplineGroups[] = array(
                        'id' => Discipline::DISCIPLINE_OTHER,
                        'title' => 'Other',
                        'disciplines' => $xDiscipline
                    );
                } 
            }
        } else {
            $xDiscipline = array();

            foreach( $disciplines as $x ) {
                $x['selected'] = \in_array($x['discipline_id'], $discipline_ids);
                $xDiscipline[] = $x;
            }

            $disciplineGroups[] = array(
                'id' => Discipline::DISCIPLINE_OTHER,
                'title' => 'Other',
                'disciplines' => $xDiscipline
            );
        }

        // echo 'Total execution time in seconds: ' . (microtime(true) - $time_start);

        $smarty->assign([
            'view' => 'orders',
            'sub_view' => $status,
            'total_orders' => $total_orders,
            'urgentOrders' => $urgentOrders,
            'orders' => $orders,
            'corslevel' => $corslevel,
            'filter_status' => $status_id,
            'deadline' => $deadline,
            'pages' => $pages,
            'cost' => $cost,
            'trackingid' => $trackingid,
            'order_id' => $order_id,
            'toptitle' => $toptitle,
            'is_search' => $is_search,
            'disciplineGroups' => $disciplineGroups
        ]);
        
        $this->setTemplate("dashboard/orders/orders.$status");
        $this->display();
    } 

    public function getUserOrders( $user_group, $user_id ) {
        $app = $this->app;
        $params = $app->request->get();
        $viewer = $app->user;
        $smarty = $app->smarty;

        if(!$viewer->is_admin && !$viewer->is_sub_admin ) {
            $this->setTemplate('404');
            $this->display();
        } 
        
        /** Search */
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
        $corslevel = (int) ArrayUtils::get($params, 'corslevel');
        $discipline_ids = ArrayUtils::get($params, 'discipline');
        $deadline = ArrayUtils::get($params, 'deadline');
        $pages = ArrayUtils::get($params, 'pages');
        $cost = ArrayUtils::get($params, 'cost');
        $order_id = ArrayUtils::get($params, 'order_id', null);
        $status_id = ArrayUtils::get($params, 'status_id', null);
        $toptitle = ArrayUtils::get($params, 'toptitle');
        $trackingid = ArrayUtils::get($params, 'trackingid');
        $is_search = ArrayUtils::has($params, 'search') ? true : false;

        if (!is_array($discipline_ids)) {
            $discipline_ids = array();
        }

        $sql = new DbQuery();
        $sql->select('DISTINCT o.*');
        $sql->from('order', 'o');
        $sql->orderBy('o.order_id DESC');

        switch( $user_group ) {
            case "customer":
                $customer = new Customer( (int) $user_id );
                if(!Validate::isLoadedObject($customer)) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $this->setVars([
                    'page_title' => "Orders › Customer › " . $customer->id
                ]);

                $sql->where('o.`customer_id` = ' . (int) $customer->id );

                $smarty->assign([
                    'user_id' => $customer->id,
                    'total_spent' => formatPrice($customer->total_spent),
                    'user_group' => 'customer',
                    'sub_view' => 'customer'
                ]);
                break;

            case "employee":
            case "writer":
            case "editor":
                $employee = new Employee( (int) $user_id );
                if(!Validate::isLoadedObject($employee)) {
                    $this->setTemplate('404');
                    $this->display();
                }
 
                $this->setVars([
                    'page_title' => "Orders › Employee › " . $employee->id
                ]);

                if ($employee->is_writer) {
                    $sql->where('o.`writer_id` = ' . (int) $employee->id );
                    $smarty->assign([
                        'user_group' => 'writer'
                    ]);
                } elseif ( $employee->is_editor ) {
                    $sql->where('o.`editor_id` = ' . (int) $employee->id );
                    $smarty->assign([
                        'user_group' => 'editor'
                    ]);
                } elseif ( $employee->is_sub_admin ) {
                    $sql->where('o.`ordermanager_id` = ' . (int) $employee->id );
                    $smarty->assign([
                        'user_group' => 'ordermanager'
                    ]);
                }

                $smarty->assign([
                    'user_id' => $employee->id,
                    'total_earned' => formatPrice($employee->total_paid),
                    'sub_view' => 'employee'
                ]);
                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        if ($is_search) {
            if ($order_id) {
                $sql->where('o.order_id LIKE \'%' . pSQL($order_id) . '%\'');
            } 

            if ($trackingid) {
                $sql->where('o.tracking_id LIKE \'%' . pSQL($trackingid) . '%\'');
            } 
        
            if($corslevel > 0 ) {
                $sql->where( 'o.academic_level_id = ' . (int) $corslevel );
            }

            if($status_id) {
                $sql->where( 'o.status_id = ' . (int) $status_id );
            }

            if( $pages == "1" ) {
                $sql->where( 'o.pages < 5' );
            } elseif($pages == "2") {
                $sql->where( 'o.pages >= 5 AND o.pages <= 10' );
            } elseif($pages == "3") {
                $sql->where('o.pages >= 10 AND o.pages <= 20');
            } elseif($pages == "4") {
                $sql->where('o.pages >= 20 AND o.pages <= 30');
            } elseif($pages == "5") {
                $sql->where( 'o.pages > 30' );
            }

            if(is_array($discipline_ids) && !empty($discipline_ids)) {
                $disciplineQ = implode(',', $discipline_ids);
                $sql->where( "o.topic_category_id IN ($disciplineQ)" );
            }

            $cost_row = 'o.total';

            if( $cost == "1" ) {
                $sql->where( "$cost_row <= 10" );
            } elseif($cost == "2") {
                $sql->where( "$cost_row >= 10 AND $cost_row <= 30" );
            } elseif($cost == "3") {
                $sql->where( "$cost_row >= 30 AND $cost_row <= 100" );
            } elseif($cost == "4") {
                $sql->where( "$cost_row >= 100 AND $cost_row <= 500" );
            } elseif($cost == "5") {
                $sql->where( "$cost_row > 500" );
            }

            $deadline_row = 'o.customer_deadline';

            if( $deadline == "1" ) {
                $sql->where( "$deadline_row <= DATE_ADD(NOW(), INTERVAL 24 HOUR)" );
            } elseif($deadline == "2") {
                $sql->where( "$deadline_row >= DATE_ADD(NOW(), INTERVAL 24 HOUR) AND $deadline_row <= DATE_ADD(NOW(), INTERVAL 72 HOUR)" );
            } elseif($deadline == "3") {
                $sql->where( "$deadline_row >= DATE_ADD(NOW(), INTERVAL 96 HOUR) AND $deadline_row <= DATE_ADD(NOW(), INTERVAL 168 HOUR)" );
            } elseif($deadline == "4") {
                $sql->where( "$deadline_row > DATE_ADD(NOW(), INTERVAL 168 HOUR)" );
            }

            if ($toptitle) {
                $sql->where('o.title LIKE \'%' . pSQL($toptitle) . '%\'');
            }
        }

        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_orders = count($result);

        $orders = array();

        if( $total_orders > 0) {
            $params['total_items'] = $total_orders;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/orders/$user_group/$user_id?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach( $result as $order ) {
                $o = new Order();
                $order = $o->getObject( (array) $order );
                if( Validate::isLoadedObject($order) ) {
                    $orders[] = $order->present();
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $disciplines = Discipline::getDisciplines();
        $groupedDisciplineGroups = ArrayUtils::groupByKey($disciplines, 'group_id');

        $disciplineGroups = array();

        if (count($groupedDisciplineGroups) > 0) {
            foreach($groupedDisciplineGroups as $disciplineGroupId => $disciplineGroupItems) {
                $xDiscipline = array();
                foreach( $disciplineGroupItems as $x ) {
                    $x['selected'] = \in_array($x['discipline_id'], $discipline_ids);
                    $xDiscipline[] = $x;
                }

                $discipline = new Discipline( (int) $disciplineGroupId );
                if(Validate::isLoadedObject($discipline)) {
                    $disciplineGroups[] = array(
                        'id' => $discipline->id,
                        'title' => $discipline->title,
                        'disciplines' => $xDiscipline
                    );
                } else {
                    $disciplineGroups[] = array(
                        'id' => Discipline::DISCIPLINE_OTHER,
                        'title' => 'Other',
                        'disciplines' => $xDiscipline
                    );
                } 
            }
        } else {
            $xDiscipline = array();

            foreach( $disciplines as $x ) {
                $x['selected'] = \in_array($x['discipline_id'], $discipline_ids);
                $xDiscipline[] = $x;
            }

            $disciplineGroups[] = array(
                'id' => Discipline::DISCIPLINE_OTHER,
                'title' => 'Other',
                'disciplines' => $xDiscipline
            );
        }

        $smarty->assign([
            'view' => 'users',
            'total_orders' => $total_orders,
            'orders' => $orders,
            'status_id' => $status_id,
            'order_id' => $order_id,
            'user_group' => $user_group,
            'corslevel' => $corslevel,
            'filter_status' => $status_id,
            'deadline' => $deadline,
            'pages' => $pages,
            'cost' => $cost,
            'trackingid' => $trackingid,
            'order_id' => $order_id,
            'toptitle' => $toptitle,
            'is_search' => $is_search,
            'disciplineGroups' => $disciplineGroups
        ]);

        $this->setTemplate("dashboard/orders/orders.user");
        $this->display();
    }

    public function getOrder( $order_id, $sub_view = "instructions" ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            $this->setTemplate('404');
            $this->display();
        }

        $order->getOrderManagers();

        $this->setVars([
            'page_title' => sprintf("Orders › #%s", $order->id) 
        ]);

        $_get = ArrayUtils::get($params, 'get', 'bids');
        $query = ArrayUtils::get($params, 'query', false);
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;

        $this->setVars([
            'page_title' => "Order #$order->id"
        ]);

        $can_view_order = false;
        $is_cancelled_order = $can_view_messages = false;

        if($user->is_admin || $user->is_sub_admin) {
            $can_view_messages = $can_view_order = true;
        }

        // check order employee
        if($order->writer_id != $user->id) {
            $sql = new DbQuery();
            $sql->select('oe.*');
            $sql->from('order_employee', 'oe');
            $sql->where('oe.`order_id` = ' . (int) $order->id );
            $sql->where('oe.`employee_type` = \'' . pSQL('writer') . '\'');
            $sql->where('oe.`employee_id` = ' . (int) $user->id );
            $orderEmployee = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
            if($orderEmployee) {
                $is_cancelled_order = true;
            }
           
        } elseif($user->is_editor && $order->editor_id != $user->id) {
            $sql = new DbQuery();
            $sql->select('oe.*');
            $sql->from('order_employee', 'oe');
            $sql->where('oe.`order_id` = ' . (int) $order->id );
            $sql->where('oe.`employee_type` = \'' . pSQL('editor') . '\'');
            $sql->where('oe.`employee_id` = ' . (int) $user->id );
            $orderEmployee = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
            if($orderEmployee) {
                $is_cancelled_order = true;
            }
        }

        if($user->is_writer && !$is_cancelled_order) {
            if(!$order->is_paid) {
                $this->setTemplate('404');
                $this->display();
            }

            if($order->writer_id == $user->id) {
                $can_view_messages = true;
            }

            if( 
                ($order->status_id == Order::WRITER_ASSIGNED || 
                $order->status_id == Order::NEW_PAID || 
                $order->status_id == Order::DONE ||
                $order->status_id == Order::DELIVERED ||
                $order->status_id == Order::FINISHED ||
                $order->status_id == Order::DISPUTE ) &&
                $order->writer_id != $user->id
            ) {
                $this->setTemplate('404');
                $this->display();
            }

            $can_view_order = true;
        }

        if($user->is_editor && !$is_cancelled_order) {
            if(!$order->is_paid) {
                $this->setTemplate('404');
                $this->display();
            }

            if($order->editor_id == $user->id) {
                $can_view_messages = true;
            }

            if((
                $order->status_id == Order::NEW_PAID || 
                $order->status_id == Order::DELIVERED ||
                $order->status_id == Order::FINISHED ||
                $order->status_id == Order::DISPUTE 
                ) && $order->editor_id != $user->id
            ) {
                $this->setTemplate('404');
                $this->display();
            }

            $can_view_order = true;
        }

        if(!$can_view_order && !$is_cancelled_order) {
            $this->setTemplate('404');
            $this->display();
        }

        $orderHistory = array();

        switch( $sub_view ) {
            case 'instructions':
            case 'messages':
            case 'files':
            case 'history':
                // progressive delivery
                if($order->progressive_delivery) {
                    $sql = new DbQuery();
                    $sql->select('pd.*');
                    $sql->from('order_pd_schedule', 'pd');
                    $sql->where('pd.`order_id` = ' . (int) $order->id );
                    $sql->orderBy('pd.schedule_id DESC');
                    $pdSchedules = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                    $smarty->assign([
                        'pdSchedules' => $pdSchedules
                    ]);
                }

                // Order History
                $sql = new DbQuery();
                $sql->select('o.*');
                $sql->from('order', 'o');

                if ($user->is_writer) {
                    $sql->where('o.`writer_id` = ' . (int) $user->id );
                } elseif ($user->is_editor) {
                    $sql->where('o.`editor_id` = ' . (int) $user->id );
                }

                $sql->where('o.`customer_id` = ' . (int) $order->customer_id );
                $sql->where('o.`order_id` != ' . (int) $order->id );
                if ($user->is_writer) {
                    $sql->where( sprintf('o.`status_id` = %s OR o.`status_id` = %s', (int) Order::DELIVERED, (int) Order::FINISHED) );
                }

                if($user->is_editor) {
                    $sql->where('o.`status_id` = ' . (int) Order::FINISHED );
                }

                $sql->orderBy('o.order_id DESC');

                $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                $total_orders = count($result);

                if( $total_orders > 0) {
                    $params['total_items'] = $total_orders;
                    $params['selected_page'] = $selected_page;
                    $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                    $params['url'] = "/order/$order->id/history?page=%s";

                    $pager = new Pager( $params );
                    $limit_query = $pager->getLimitSql();

                    $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

                    foreach( $result as $orderResult ) {
                        $o = new Order();
                        $orderResult = $o->getObject( (array) $orderResult );
                        if( Validate::isLoadedObject($orderResult) ) {
                            $orderHistory[] = $orderResult->present();
                        }
                    }

                    $smarty->assign([
                        'pager' => $pager->getPager()
                    ]);
                }

                if( $user->is_writer ) {
                    if( $order->status_id == Order::ORDER_PUBLISHED || 
                        $order->status_id == Order::INQUERY_PUBLISHED ||
                        $order->status_id == Order::WRITER_ACCEPTED
                    ) {
                        $bidSql = new DbQuery();
                        $bidSql->select('bid.*');
                        $bidSql->from('order_bid', 'bid');
                        $bidSql->where('order_id = ' . (int) $order->id );
                        $bidSql->where('is_deleted = 0');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $bidSql );
        
                        $order_bids = array();
                        foreach( $result as $bid ) {
                            $writer_id = ArrayUtils::get($bid, 'writer_id');
                            $order_bids[$writer_id] = $bid;
                        }
                        
                        $progress_orders = $user->order_stats['unconfirmed'] + $user->order_stats['progress'] + $user->order_stats['revision'];

                        $orderRules = $user->getEmployeeOrderRules();
                        $number_of_takes = $orderRules['takes'];
                        $number_of_bids = $orderRules['bids'];

                        $can_take = $can_bid = false;
                        if($orderRules['takes'] == '-1') {
                            $can_take = true;
                        } else {
                            $can_take = ($progress_orders < $number_of_takes) ? true : false;
                        }

                        if($orderRules['bids'] == '-1') {
                            $can_bid = true;
                        } elseif( $user->employee_status == Employee::STATUS_PROBATION ) {
                            $can_bid = false;
                        } else {
                            $can_bid = ($user->order_stats['bids'] < $number_of_bids) ? false : true;
                        }

                        $smarty->assign([
                            'can_take' => $can_take,
                            'can_bid' => $can_bid,
                            'has_bidded' => ArrayUtils::contains($order_bids, $user->id) ? true : false,
                            'bid' => ArrayUtils::get($order_bids, $user->id)
                        ]);
                    }
                }

                break;

            case 'assign':
                
                $ordermanagers = $editors = $writers = $bids = array();
                switch( $_get ) {
                    case 'bids':

                        $sql = new DbQuery();
                        $sql->select('DISTINCT b.bid_id, b.order_id, b.writer_id, b.bid_time, b.amount, b.is_assigned, b.is_deleted, b.date_add');
                        $sql->from('order_bid', 'b');
                        $sql->where('order_id = ' . (int) $order->id );
                        $sql->where('is_deleted = 0');
                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );
                        $total_bids = count($result);

                        if( $total_bids > 0) {
                            $params['total_items'] = $total_bids;
                            $params['selected_page'] = $selected_page;
                            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                            $params['url'] = '/order/'.$order->id.'/assign?get=bids&page=%s';
                
                            $pager = new Pager( $params );
                            $limit_query = $pager->getLimitSql();
                
                            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql.$limit_query);
                
                            foreach( $result as $bid ) {
                                $writer = new Employee( (int) $bid['writer_id'] );
                                if(Validate::isLoadedObject($writer)) {
                                    $average_rating = $writer->getWriterAverageRating();
                                    $presentedWriter = (array) $writer;
                                    $presentedWriter['average_rating'] = $average_rating;
                                    $bids[] = $presentedWriter;
                                }
                            }
                
                            $app->smarty->assign([
                                'pager' => $pager->getPager()
                            ]);
                        }

                        $smarty->assign([
                            'bids' => $bids,
                            'total_bids' => $total_bids
                        ]);

                        break;

                    case 'writers':                        
                        $sql = new DbQuery();
                        $sql->select('e.employee_id');
                        $sql->from('employee', 'e');
                        $sql->where('employee_group = ' . (int) Employee::GROUP_WRITER );
                        $sql->where('is_started = 1');
                        $sql->where('employee_status = ' . (int) Employee::STATUS_WORKING . ' OR employee_status = ' . (int) Employee::STATUS_PROBATION);

                        if($query) {
                            $sql->where('
                                e.employee_id LIKE \'%' . pSQL($query) . '%\' OR
                                e.first_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.last_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.email LIKE \'%' . pSQL($query) . '%\'
                            ');
                        }

                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                        $total_writers = count($result);

                        $writers = array();
                        if( $total_writers > 0) {
                            $params['total_items'] = $total_writers;
                            $params['selected_page'] = $selected_page;
                            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                            $params['url'] = '/order/'.$order->id.'/assign?get=writers&page=%s';
                
                            $pager = new Pager( $params );
                            $limit_query = $pager->getLimitSql();
                            $sql .= $limit_query;
                            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                
                            foreach( $result as $writer ) {
                                $writer = new Employee( (int) $writer['employee_id'] );
                                if( Validate::isLoadedObject($writer) ) {
                                    $average_rating = $writer->getWriterAverageRating();
                                    $presentedWriter = (array) $writer;
                                    $presentedWriter['average_rating'] = $average_rating;
                                    $writers[] = $presentedWriter;
                                }
                            }
                
                            $app->smarty->assign([
                                'pager' => $pager->getPager()
                            ]);
                        }

                        $smarty->assign([
                            'query' => $query,
                            'total_writers' => $total_writers
                        ]);

                        break;

                    case 'editors': 
                        if(!Configuration::get('EDITOR_ACCOUNT_ENABLED')) {
                            $this->setTemplate('404');
                            $this->display();
                        }     

                        $sql = new DbQuery();
                        $sql->select('e.employee_id');
                        $sql->from('employee', 'e');
                        $sql->where('employee_group = ' . (int) Employee::GROUP_EDITOR );
                        $sql->where('is_started = 1');
                        $sql->where('employee_status = ' . (int) Employee::STATUS_WORKING);

                        if($query) {
                            $sql->where('
                                e.employee_id LIKE \'%' . pSQL($query) . '%\' OR
                                e.first_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.last_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.email LIKE \'%' . pSQL($query) . '%\'
                            ');
                        }

                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                        $total_editors = count($result);

                        if( $total_editors > 0) {
                            $params['total_items'] = $total_editors;
                            $params['selected_page'] = $selected_page;
                            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                            $params['url'] = '/order/'.$order->id.'/assign?get=editors&page=%s';
                
                            $pager = new Pager( $params );
                            $limit_query = $pager->getLimitSql();
                            $sql .= $limit_query;
                            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                
                            foreach( $result as $editor ) {
                                $editor = new Employee( (int) $editor['employee_id'] );
                                if( Validate::isLoadedObject($editor) ) {
                                    $average_rating = $editor->getWriterAverageRating();
                                    $presentedEditor = (array) $editor;
                                    $presentedEditor['average_rating'] = $average_rating;
                                    $editors[] = $presentedEditor;
                                }
                            }
                
                            $app->smarty->assign([
                                'pager' => $pager->getPager()
                            ]);
                        }

                        $smarty->assign([
                            'query' => $query,
                            'total_editors' => $total_editors
                        ]);

                        break;

                    case 'ordermanagers':    
                        if(!Configuration::get('ORDER_MANAGER_ENABLED')) {
                            $this->setTemplate('404');
                            $this->display();
                        }                    
                        $sql = new DbQuery();
                        $sql->select('e.employee_id');
                        $sql->from('employee', 'e');
                        $sql->where('employee_group = ' . (int) Employee::GROUP_SUB_ADMIN );
                        $sql->where('is_started = 1');
                        $sql->where('employee_status = ' . (int) Employee::STATUS_WORKING);

                        if($query) {
                            $sql->where('
                                e.employee_id LIKE \'%' . pSQL($query) . '%\' OR
                                e.first_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.last_name LIKE \'%' . pSQL($query) . '%\' OR
                                e.email LIKE \'%' . pSQL($query) . '%\'
                            ');
                        }

                        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                        $total_ordermanagers = count($result);

                        if( $total_ordermanagers > 0) {
                            $params['total_items'] = $total_ordermanagers;
                            $params['selected_page'] = $selected_page;
                            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
                            $params['url'] = '/order/'.$order->id.'/assign?get=ordermanagers&page=%s';
                
                            $pager = new Pager( $params );
                            $limit_query = $pager->getLimitSql();
                            $sql .= $limit_query;
                            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
                
                            foreach( $result as $ordermanager ) {
                                $ordermanager = new Employee( (int) $ordermanager['employee_id'] );
                                if( Validate::isLoadedObject($ordermanager) ) {
                                    $average_rating = $ordermanager->getWriterAverageRating();
                                    $active_pages = $ordermanager->getActivePages();
                                    $presentedOrderManager = (array) $ordermanager;
                                    $presentedOrderManager['average_rating'] = $average_rating;
                                    $presentedOrderManager['active_pages'] = $active_pages;
                                    $ordermanagers[] = $presentedOrderManager;
                                }
                            }
                
                            $app->smarty->assign([
                                'pager' => $pager->getPager()
                            ]);
                        }

                        $smarty->assign([
                            'query' => $query,
                            'total_ordermanagers' => $total_ordermanagers
                        ]);

                        break;

                    default:
                        $this->setTemplate('404');
                        $this->display();
                        break;
                }    

                $smarty->assign([
                    'bids' => $bids,
                    'editors' => $editors,
                    'writers' => $writers,
                    'ordermanagers' => $ordermanagers,
                ]);

                break;

            default:
                $this->setTemplate('404');
                $this->display();
                break;
        }

        $smarty->assign([
            'HOOK_DISPLAYORDERDETAIL' => Hook::exec('displayOrderDetail', ['order' => $order]),
            'order' => $order->present(),
            'orderHistory' => $orderHistory,
            'view' => 'orders',
            'sub_view' => $sub_view,
            'can_view_order' => $can_view_order,
            'is_cancelled_order' => $is_cancelled_order,
            'can_view_messages' => $can_view_messages,
            '_get' => $_get
        ]);

        $this->setTemplate('dashboard/order/order');
        $this->display();
    }

    public function createOrder() {
        global $globals;
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;
        $presenter = new ObjectPresenter();

        $customer_id = $app->request->get('customer_id');
        
        $customer = new Customer( (int) $customer_id );
        if(!Validate::isLoadedObject($customer)) {
            $this->setTemplate('404');
            $this->display();
        }

        $site = new Site( (int) $customer->site_id );
        if(!Validate::isLoadedObject($site)) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Orders › New"
        ]);

        $deadlinePresenter = new DeadlinePresenter();
        $serviceTypes = $academicLevels = $writerPreferences = array();

        $isProgrammingEnabled = (bool) Configuration::get('PROGRAMMING_ENABLED', $site->id, false);
        $isCalculationsEnabled = (bool) Configuration::get('CALCULATIONS_ENABLED', $site->id, false);
        $isAcademicWritingEnabled = (bool) Configuration::get('ACADEMIC_WRITING_ENABLED', $site->id, true);
        $isArticleWritingEnabled = (bool) Configuration::get('ARTICLE_WRITING_ENABLED', $site->id, false);

        // enable article writing if other options are disabled
        if(!$isProgrammingEnabled && !$isCalculationsEnabled && !$isArticleWritingEnabled) {
            $isAcademicWritingEnabled = true;
        }

        if($isAcademicWritingEnabled) {
            $serviceTypes[Order::COMPLEX_ASSIGNMENT] = array(
                'serviceId' => Order::COMPLEX_ASSIGNMENT,
                'name' => 'Academic writing',
                'shortName' => 'Academic',
                'description' => 'Custom papers',
            );
        }

        if($isProgrammingEnabled && Module::isInstalled('programming')) {
            $serviceTypes[Order::PROGRAMMING] = array(
                'serviceId' => Order::PROGRAMMING,
                'name' => 'Programming',
                'shortName' => 'Programming',
                'description' => 'Tech assignments',
            );
        }

        if($isCalculationsEnabled && Module::isInstalled('calculations')) {
            $serviceTypes[Order::CALCULATIONS] = array(
                'serviceId' => Order::CALCULATIONS,
                'name' => 'Calculations',
                'shortName' => 'Calculations',
                'description' => 'Problems solving',
            );
        }

        if($isArticleWritingEnabled && Module::isInstalled('articlewriting')) {
            $serviceTypes[Order::ARTICLE_WRITING] = array(
                'serviceId' => Order::ARTICLE_WRITING,
                'name' => 'Article Writing',
                'shortName' => 'Article',
                'description' => 'Custom content',
            );
        }

        switch(count($serviceTypes)) {
            case 1:
                $serviceTypeClass = 'col-md-12';
                break;

            case 2:
                $serviceTypeClass = 'col-md-6';
                break;

            case 3:
                $serviceTypeClass = 'col-md-4';
                break;

            case 4:
                $serviceTypeClass = 'col-md-3';
                break;

            default:
                $serviceTypeClass = 'col-md-12';
                break;
        }

        
        $academicDisciplineGroups = Discipline::getDisciplinesByGroup( Order::COMPLEX_ASSIGNMENT );

        $paperFormats = PaperFormat::getPaperFormats();
        $paperTypesPreferences = PaperType::getPaperTypes();

        $paperTypes = array();
        foreach($paperTypesPreferences as $paperType) {
            $paperType = new PaperType( $paperType['paper_type_id'] );
            if(Validate::isLoadedObject($paperType)) {
                $paperTypes[] = $presenter->present($paperType);
            }
        }

        $academicLevels = AcademicLevel::getAcademicLevels($site->id);

        foreach ( $globals['writerPreferences'] as $writerPreferenceId => $writerPreference ) {
            $writerPreference['id'] = (int) $writerPreferenceId;
            $writerPreferences[] = $writerPreference;
        }	

        ## academic
        $deadlines = Deadline::getAcademicWritingDeadlines( (int) $site->id );

        $academicTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $academicTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $academicTariffs = ArrayUtils::groupByKey( $academicTariffs, "academicLevel" );

        # programming
        $deadlines = Deadline::getProgrammingDeadlines( (int) $site->id );

        $programmingTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $programmingTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        # programming disciplines
        $programmingDisciplines = Discipline::getProgrammingDisciplines();

        $programmingItemPricing = array();
		foreach($programmingDisciplines as $discipline) {
			$sql = new DbQuery();
			$sql->select('ip.item_pricing_id');
			$sql->from('item_pricing', 'ip');
			$sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
			$sql->orderBy('ip.task_size_id ASC');
			$itemPricing = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

			foreach($itemPricing as $itemPrice) {
				$itemPricing = new ItemPricing( (int) $itemPrice['item_pricing_id'] );
				if(Validate::isLoadedObject($itemPricing)) {
                    $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

					$programmingItemPricing[] = array(
						'id' => (int) $itemPricing->id,
						'disciplineId' => (int) $discipline['discipline_id'],
						'sizeId' => (int) $itemPricing->task_size_id,
						'sizeName' => $taskSize['title'],
						'price' => (float) $itemPricing->price
					);
				}
			}
		}

        $programmingItemPricing = ArrayUtils::groupByKey($programmingItemPricing, 'disciplineId');

        # calculations
        $calculationsDisciplines = Discipline::getCalculationsDisciplines();

        $deadlines = Deadline::getCalculationsDeadlines( (int) $site->id );

        $calculationsTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $calculationsTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $calculationsItemPricing = array();
		foreach($calculationsDisciplines as $discipline) {
			$sql = new DbQuery();
			$sql->select('ip.item_pricing_id');
			$sql->from('item_pricing', 'ip');
			$sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
			$sql->orderBy('ip.task_size_id ASC');
			$itemPricing = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

			foreach($itemPricing as $itemPrice) {
				$itemPricing = new ItemPricing( (int) $itemPrice['item_pricing_id'] );
				if(Validate::isLoadedObject($itemPricing)) {
                    $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

					$calculationsItemPricing[] = array(
						'id' => (int) $itemPricing->id,
						'disciplineId' => (int) $discipline['discipline_id'],
						'sizeId' => (int) $itemPricing->task_size_id,
						'sizeName' => $taskSize['title'],
						'price' => (float) $itemPricing->price
					);
				}
			}
		}

        $calculationsItemPricing = ArrayUtils::groupByKey($calculationsItemPricing, 'disciplineId');

        # article writing
        $configsForArticleWriting = array();

        $articleCategories = Discipline::getArticleWritingsDisciplines();
        $configsForArticleWriting['categories'] = $articleCategories;

        $contentTypes = PaperType::getPaperTypes( Order::ARTICLE_WRITING );
        $configsForArticleWriting['contentTypes'] = $contentTypes;

        $deadlines = Deadline::getArticleWritingDeadlines( (int) $site->id );
        $articleWritingTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $articleWritingTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $configsForArticleWriting['tariffs'] = $articleWritingTariffs;

        # englishTypes
		$englishTypes = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT et.* FROM '. Db::prefix('english_type') . ' AS et ORDER BY et.title ASC');
		
		$configEnglishTypes = array();
		foreach($englishTypes as $englishType) {
			$configEnglishTypes[] = array(
				'id' => (int) $englishType['english_type_id'],
				'title' => $englishType['title']
			);
		}

		$configsForArticleWriting['englishTypes'] = $configEnglishTypes;

		#targetAudiences
		$targetAudiences = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ta.* FROM '. Db::prefix('target_audience') . ' AS ta ORDER BY ta.title ASC');
		
		$configTargetAudiences = array();
		foreach($targetAudiences as $targetAudience) {
			$configTargetAudiences[] = array(
				'id' => (int) $targetAudience['target_audience_id'],
				'title' => $targetAudience['title']
			);
		}

		$configsForArticleWriting['targetAudiences'] = $configTargetAudiences;

		#writingTones
		$writingTones = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT wt.* FROM '. Db::prefix('writing_tone') . ' AS wt ORDER BY wt.title ASC');
		
		$configWritingTones = array();
		foreach($writingTones as $writingTone) {
			$configWritingTones[] = array(
				'id' => (int) $writingTone['writing_tone_id'],
				'title' => $writingTone['title'],
				'description' => $writingTone['description']
			);
		}

		$configsForArticleWriting['writingTones'] = $configWritingTones;

		#writingStyles
		$writingStyles = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ws.* FROM '. Db::prefix('writing_style') . ' AS ws ORDER BY ws.title ASC');

		$configWritingStyles = array();
		foreach($writingStyles as $writingStyle) {
			$configWritingStyles[] = array(
				'id' => (int) $writingStyle['writing_style_id'],
				'title' => $writingStyle['title'],
				'description' => $writingStyle['description']
			);
		}

		$configsForArticleWriting['writingStyles'] = $configWritingStyles;

		#contentFeels
		$contentFeels = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT cf.* FROM '. Db::prefix('content_feel') . ' AS cf ORDER BY cf.title ASC');
		
		$configContentFeels = array();
		foreach($contentFeels as $contentFeel) {
			$configContentFeels[] = array(
				'id' => (int) $contentFeel['content_feel_id'],
				'title' => $contentFeel['title'],
				'description' => $contentFeel['description']
			);
		}

		$configsForArticleWriting['contentFeels'] = $configContentFeels;
        $defaultOrderServiceTypeId = (int) Configuration::get('DEFAULT_SERVICE_TYPE_ID', $site->id);
        if(!ArrayUtils::has($serviceTypes, $defaultOrderServiceTypeId)) {
            $defaultServiceType = ArrayUtils::shift($serviceTypes);
            $defaultOrderServiceTypeId = $defaultServiceType['serviceId'];
        }

        $smarty->assign([
            'serviceTypes' => $serviceTypes,
            'serviceTypeClass' => $serviceTypeClass,
            'customerId' => $customer->id,
            'isProgrammingEnabled' => $isProgrammingEnabled,
			'isCalculationsEnabled' => $isCalculationsEnabled,
			'isAcademicWritingEnabled' => $isAcademicWritingEnabled,
			'isArticleWritingEnabled' => $isArticleWritingEnabled,
			'defaultOrderServiceTypeId' => $defaultOrderServiceTypeId,
            'defaultCourseLevelId' => (int) Configuration::get('DEFAULT_COURSELEVEL', $site->id, LEVEL_COLLEGE),
            'academicLevels' => $academicLevels,
            'paperTypes' => $paperTypes,
            'writerPreferences' => $writerPreferences,
            'paperFormats' => $paperFormats,
            'academicTariffs' => $academicTariffs,
            'academicDisciplineGroups' => $academicDisciplineGroups,
            'calculationsTariffs' => $calculationsTariffs,
            'calculationsDisciplines' => $calculationsDisciplines,
            'calculationsItemPricing' => $calculationsItemPricing,
            'programmingTariffs' => $programmingTariffs,
            'programmingDisciplines' => $programmingDisciplines,
            'programmingItemPricing' => $programmingItemPricing,
            'configsForArticleWriting' => $configsForArticleWriting,
            'view' => 'create'
        ]);

        $this->setTemplate('dashboard/order/order.create');
        $this->display();
    }

    public function editOrder( $order_id ) {
        global $globals;
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;
        $presenter = new ObjectPresenter();
        
        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            $this->setTemplate('404');
            $this->display();
        }

        // $this->registerJavascript('calculator', '/content/themes/' . PROX_ACTIVE_THEME . '/js/proxim/calculator.js', ['position' => 'footer', 'media' => 'all' ]);

        $this->setVars([
            'page_title' => "Edit Order #$order->id"
        ]);

        # academic writing
        $academicLevels = $writerPreferences = array();
        $paperFormats = PaperFormat::getPaperFormats();

        $paperTypesPreferences = PaperType::getPaperTypes();

        $paperTypes = array();
        foreach($paperTypesPreferences as $paperType) {
            $paperType = new PaperType( $paperType['paper_type_id'] );
            if(Validate::isLoadedObject($paperType)) {
                $paperTypes[] = $presenter->present($paperType);
            }
        }

        $academicLevels = AcademicLevel::getAcademicLevels($order->site_id);

        foreach ( $globals['writerPreferences'] as $writerPreferenceId => $writerPreference ) {
            $writerPreference['id'] = (int) $writerPreferenceId;
            $writerPreferences[] = $writerPreference;
        }	

        $deadlines = Deadline::getAcademicWritingDeadlines( (int) $order->site_id );
        $deadlinePresenter = new DeadlinePresenter();
        $academicTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $academicTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $academicTariffs = ArrayUtils::groupByKey( $academicTariffs, "academicLevel" );

        $disciplineGroups = Discipline::getDisciplinesByGroup();

        $siteSettings = Configuration::getMultiple(
            [
                'WRITER_SAMPLES_ENABLED',
                'PROGRESSIVE_DELIVERY_ENABLED',
                'EXPERT_PROOFREADING_ENABLED',
                'COPY_OF_SOURCES_ENABLED',
            ],
            (int) $order->site_id
        ); 

        # programming
        $deadlines = Deadline::getProgrammingDeadlines( (int) $order->site_id );

        $programmingTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $programmingTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        # programming disciplines
        $programmingDisciplines = Discipline::getProgrammingDisciplines();

        $programmingItemPricing = array();
		foreach($programmingDisciplines as $discipline) {
			$sql = new DbQuery();
			$sql->select('ip.item_pricing_id');
			$sql->from('item_pricing', 'ip');
			$sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
			$sql->orderBy('ip.task_size_id ASC');
			$itemPricing = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

			foreach($itemPricing as $itemPrice) {
				$itemPricing = new ItemPricing( (int) $itemPrice['item_pricing_id'] );
				if(Validate::isLoadedObject($itemPricing)) {
                    $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

					$programmingItemPricing[] = array(
						'id' => (int) $itemPricing->id,
						'disciplineId' => (int) $discipline['discipline_id'],
						'sizeId' => (int) $itemPricing->task_size_id,
						'sizeName' => $taskSize['title'],
						'price' => (float) $itemPricing->price
					);
				}
			}
		}

        $programmingItemPricing = ArrayUtils::groupByKey($programmingItemPricing, 'disciplineId');

        # calculations
        $calculationsDisciplines = Discipline::getCalculationsDisciplines();

        $deadlines = Deadline::getCalculationsDeadlines( (int) $order->site_id );

        $calculationsTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $calculationsTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $calculationsItemPricing = array();
		foreach($calculationsDisciplines as $discipline) {
			$sql = new DbQuery();
			$sql->select('ip.item_pricing_id');
			$sql->from('item_pricing', 'ip');
			$sql->where('ip.discipline_id = ' . (int) $discipline['discipline_id'] );
			$sql->orderBy('ip.task_size_id ASC');
			$itemPricing = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

			foreach($itemPricing as $itemPrice) {
				$itemPricing = new ItemPricing( (int) $itemPrice['item_pricing_id'] );
				if(Validate::isLoadedObject($itemPricing)) {
                    $taskSize = ArrayUtils::get($globals['taskSizes'], $itemPricing->task_size_id);

					$calculationsItemPricing[] = array(
						'id' => (int) $itemPricing->id,
						'disciplineId' => (int) $discipline['discipline_id'],
						'sizeId' => (int) $itemPricing->task_size_id,
						'sizeName' => $taskSize['title'],
						'price' => (float) $itemPricing->price
					);
				}
			}
		}

        $calculationsItemPricing = ArrayUtils::groupByKey($calculationsItemPricing, 'disciplineId');

        # article writing
        $configsForArticleWriting = array();

        $articleCategories = Discipline::getArticleWritingsDisciplines();
        $configsForArticleWriting['categories'] = $articleCategories;

        $contentTypes = PaperType::getPaperTypes( Order::ARTICLE_WRITING );
        $configsForArticleWriting['contentTypes'] = $contentTypes;

        $deadlines = Deadline::getArticleWritingDeadlines( (int) $order->site_id );
        $articleWritingTariffs = array();
        foreach($deadlines as $deadline) {
            $deadline = new Deadline( $deadline['deadline_id'] );
            if(Validate::isLoadedObject($deadline)) {
                $articleWritingTariffs[] = $deadlinePresenter->present($deadline)->jsonSerialize();
            }
        }

        $configsForArticleWriting['tariffs'] = $articleWritingTariffs;

        # englishTypes
		$englishTypes = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT et.* FROM '. Db::prefix('english_type') . ' AS et ORDER BY et.title ASC');
		
		$configEnglishTypes = array();
		foreach($englishTypes as $englishType) {
			$configEnglishTypes[] = array(
				'id' => (int) $englishType['english_type_id'],
				'title' => $englishType['title']
			);
		}

		$configsForArticleWriting['englishTypes'] = $configEnglishTypes;

		#targetAudiences
		$targetAudiences = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ta.* FROM '. Db::prefix('target_audience') . ' AS ta ORDER BY ta.title ASC');
		
		$configTargetAudiences = array();
		foreach($targetAudiences as $targetAudience) {
			$configTargetAudiences[] = array(
				'id' => (int) $targetAudience['target_audience_id'],
				'title' => $targetAudience['title']
			);
		}

		$configsForArticleWriting['targetAudiences'] = $configTargetAudiences;

		#writingTones
		$writingTones = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT wt.* FROM '. Db::prefix('writing_tone') . ' AS wt ORDER BY wt.title ASC');
		
		$configWritingTones = array();
		foreach($writingTones as $writingTone) {
			$configWritingTones[] = array(
				'id' => (int) $writingTone['writing_tone_id'],
				'title' => $writingTone['title'],
				'description' => $writingTone['description']
			);
		}

		$configsForArticleWriting['writingTones'] = $configWritingTones;

		#writingStyles
		$writingStyles = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT ws.* FROM '. Db::prefix('writing_style') . ' AS ws ORDER BY ws.title ASC');

		$configWritingStyles = array();
		foreach($writingStyles as $writingStyle) {
			$configWritingStyles[] = array(
				'id' => (int) $writingStyle['writing_style_id'],
				'title' => $writingStyle['title'],
				'description' => $writingStyle['description']
			);
		}

		$configsForArticleWriting['writingStyles'] = $configWritingStyles;

		#contentFeels
		$contentFeels = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS('SELECT cf.* FROM '. Db::prefix('content_feel') . ' AS cf ORDER BY cf.title ASC');
		
		$configContentFeels = array();
		foreach($contentFeels as $contentFeel) {
			$configContentFeels[] = array(
				'id' => (int) $contentFeel['content_feel_id'],
				'title' => $contentFeel['title'],
				'description' => $contentFeel['description']
			);
		}

		$configsForArticleWriting['contentFeels'] = $configContentFeels;

        $smarty->assign([
            'order' => $order->present(),
            'siteSettings' => $siteSettings,
            'academicLevels' => $academicLevels,
            'paperTypes' => $paperTypes,
            'writerPreferences' => $writerPreferences,
            'paperFormats' => $paperFormats,
            'academicTariffs' => $academicTariffs,
            'disciplineGroups' => $disciplineGroups,
            'calculationsTariffs' => $calculationsTariffs,
            'calculationsDisciplines' => $calculationsDisciplines,
            'calculationsItemPricing' => $calculationsItemPricing,
            'programmingTariffs' => $programmingTariffs,
            'programmingDisciplines' => $programmingDisciplines,
            'programmingItemPricing' => $programmingItemPricing,
            'configsForArticleWriting' => $configsForArticleWriting,
            'view' => 'orders',
            'sub_view' => 'edit'
        ]);

        $this->setTemplate('dashboard/order/order.edit');
        $this->display();
    }
}