<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Dashboard;

use Db;
use Exception;
use Proxim\Database\DbQuery;
use Proxim\Controller;
use Proxim\Pager;
use Proxim\Site\Site;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\User\Employee;
use Proxim\User\Customer;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\DateUtils;
use Proxim\Util\ArrayUtils;

class Modules extends Controller {

    private $status;

    protected $xml_modules_list = PROX_API_MODULES_LIST;

    /*
    ** @var array map with $_GET keywords and their callback
    */
    protected $map = array(
        'check' => 'check',
        'install' => 'install',
        'uninstall' => 'uninstall',
        'configure' => 'getContent',
        'update' => 'update',
        'delete' => 'delete',
        'checkAndUpdate' => 'checkAndUpdate',
        'updateAll' => 'updateAll',
    );

    public function installedModules() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Modules › Installed"
        ]);

        $all_modules = Module::getModulesOnDisk();
        $modules_list = [];

        foreach ($all_modules as $module) {
            if($module->installed) {
                $modules_list[$module->name] = $module;
            }
        }

        ksort($modules_list);

        $smarty->assign([
            'view' => 'modules',
            'sub_view' => 'installed',
            'modules_list' => $modules_list
        ]);

        $this->setTemplate('dashboard/modules/modules');
        $this->display();
    }

    public function catalogue() {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        // Refresh modules_list.xml every day (1440)
        if (ArrayUtils::has($params, 'refresh') || !$this->isFresh(Module::CACHE_FILE_MODULES_LIST, 1440)) {
            if ($this->refresh(Module::CACHE_FILE_MODULES_LIST, 'https://' . $this->xml_modules_list)) {
                $this->status = 'refresh';
            } elseif ($this->refresh(Module::CACHE_FILE_MODULES_LIST, 'http://' . $this->xml_modules_list)) {
                $this->status = 'refresh';
            } else {
                $this->status = 'error';
            }
        } else {
            $this->status = 'cache';
        }

        $this->setVars([
            'page_title' => "Modules › Catalogue"
        ]);

        $all_modules = Module::getModulesOnDisk(true);
        $modules_list = [];

        foreach ($all_modules as $module) {
            if(!$module->installed) {
                $modules_list[$module->name] = $module;
            }
        }

        ksort($modules_list);

        $smarty->assign([
            'view' => 'modules',
            'sub_view' => 'catalogue',
            'modules_list' => $modules_list
        ]);

        $this->setTemplate('dashboard/modules/modules');
        $this->display();
    }

    public function initModule( $module_name, $controller = 'configure' ) {
        $app = $this->app;
        $user = $app->user;
        $params = $app->request->get();
        $smarty = $app->smarty;

        $module = Module::getInstanceByName($module_name);
        if(!Validate::isLoadedObject($module)) {
            $this->setTemplate('404');
            $this->display();
        }

        if(ArrayUtils::has($params, 'site_id')) {
            $site_id = ArrayUtils::get($params, 'site_id');
            $site = new Site( (int) $site_id );
            if(Validate::isLoadedObject($site)) {
                $app->site = $site;
            }
        }

        $method = ArrayUtils::get($this->map, $controller, $controller);

        $smarty->assign([
            'module_name' => $module->name,
            'module_icon' => $module->icon,
            'module_display_name' => $module->displayName
        ]);

        if(!$module->checkAccess()) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => $module->displayName
        ]);

        switch( $controller ) {
            case "configure":
                if( Module::isInstalled($module->name)) {
                    // We check if method of module exists
                    if (!method_exists($module, $method)) {
                        $this->setTemplate('404');
                        $this->display();
                    }

                    $output = $module->{$method}();

                    $smarty->assign('module_content', $output);
                    $this->setTemplate('dashboard/modules/module');
                    return $this->display();
                }
                break;

            default:
                if (!method_exists($module, $method)) {
                    $this->setTemplate('404');
                    $this->display();
                }

                $this->init();
                $module->{$method}();
                break;
        }
    }

    public function purchaseModule( $module_name ) {
        $app = $this->app;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!$user->is_admin) {
            $this->setTemplate('404');
            $this->display();
        }

        $this->setVars([
            'page_title' => "Modules › Purchase"
        ]);

        $this->setTemplate('dashboard/modules/purchase');
        $this->display();
    }
}
