<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Proxim\Controller;
use Proxim\Mail;
use Proxim\Preference\ApplicationEssay;
use Proxim\Preference\WritingGuide;
use Proxim\Preference\WritingPrompt;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class WriterApplications extends Controller {

    public function applications() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin && !$user->is_sub_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $action = ArrayUtils::get($payload, 'handle');
        $id = (int) ArrayUtils::get($payload, 'id');

        switch($action) {
            case 'approve':
                $employee = new Employee((int) $id);
                if(!Validate::isLoadedObject($employee)) {
                    return $this->modal("ERROR", "Not Found", "This employee does not exist"); 
                }

                $employee->is_application_approved = 1;
                $employee->is_application_rejected = 0;
                $employee->employee_group = Employee::GROUP_WRITER;
                $employee->employee_status = Employee::STATUS_PROBATION;
                $employee->is_started = 1;
                $employee->application_approved_at = DateUtils::now();
                $employee->update();

                if(Validate::isEmail($employee->email)) {
                    try {
                        Mail::send(
                            'application_approved',
                            'Writer Application Approved',
                            array(
                                'username' => $employee->first_name . ' ' . $employee->first_name
                            ),
                            $employee->email
                        );
                    } catch(\Exception $e) {
                    }
                }

                return $this->modal("SUCCESS", "Application Approved", "Writer application approved successfully"); 
                break;  

            case 'decline':
                $employee = new Employee((int) $id);
                if(!Validate::isLoadedObject($employee)) {
                    return $this->modal("ERROR", "Not Found", "This employee does not exist"); 
                }

                $employee->is_application_rejected = 1;
                $employee->is_application_approved = 0;
                $employee->application_rejected_at = DateUtils::now();
                $employee->update();

                if(Validate::isEmail($employee->email)) {
                    try {
                        Mail::send(
                            'application_rejected',
                            'Writer Application Rejected',
                            array(
                                'username' => $employee->first_name . ' ' . $employee->first_name
                            ),
                            $employee->email
                        );
                    } catch(\Exception $e) {
                    }
                }

                return $this->modal("SUCCESS", "Application Rejected", "Writer application approved successfully"); 
                break;
        }

    }

    public function writingPrompts( $action ) {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        switch($action) {
            case 'add':
            case 'edit':
                $insert = true;
                if(ArrayUtils::has($payload, 'writing_prompt_id')) {
                    $writing_prompt_id = ArrayUtils::get($payload, 'writing_prompt_id');
                    $writing_prompt = new WritingPrompt( (int) $writing_prompt_id  );
                    if(!Validate::isLoadedObject($writing_prompt)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Writing prompt does not exist"
                        ]);
                    }

                    $insert = false;
                } else {
                    $writing_prompt = new WritingPrompt();
                }

                $title = ArrayUtils::get($payload, 'title');

                if(!$title) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter the title"
                    ]);
                }

                $writing_prompt->title = $title;

                if($insert) {
                    $writing_prompt->add();

                    return $this->modal("SUCCESS", "Added Successfully", "Writing prompt added successfully"); 
                    break;
                } else {
                    $writing_prompt->update();

                    return $this->response([
                        "success" => true,
                        "message" => "Writing prompt has been updated successfully"
                    ]);
                }
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
            
        }
    }

    public function essays( $action ) {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        switch($action) {
            case 'add':
            case 'edit':
                $insert = true;
                if(ArrayUtils::has($payload, 'essay_id')) {
                    $essay_id = ArrayUtils::get($payload, 'essay_id');
                    $essay = new ApplicationEssay( (int) $essay_id  );
                    if(!Validate::isLoadedObject($essay)) {
                        return $this->response([
                            "error" => true,
                            "message" => "Essay does not exist"
                        ]);
                    }

                    $insert = false;
                } else {
                    $essay = new ApplicationEssay();
                }

                $title = ArrayUtils::get($payload, 'title');

                if(!$title) {
                    return $this->response([
                        "error" => true,
                        "message" => "Please enter the title"
                    ]);
                }

                $essay->title = $title;

                if($insert) {
                    $essay->add();

                    return $this->modal("SUCCESS", "Added Successfully", "Essay added successfully"); 
                    break;
                } else {
                    $essay->update();

                    return $this->response([
                        "success" => true,
                        "message" => "Essay has been updated successfully"
                    ]);
                }
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
            
        }
    }
}