<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Exception;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Site\Site;
use Proxim\Coupon;
use Proxim\Currency;
use Proxim\Mail;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\Controller;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Order\File;
use Proxim\Order\Message;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\ImageManager;
use Proxim\Preference\Deadline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Uploader;

class MySites extends Controller {
    public function updateSettings() {
        global $globals;
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $do = ArrayUtils::get($params, 'do');
        $site_id = ArrayUtils::get($payload, 'site_id');

        switch( $do ) {
            case 'add':
                $site_name = ArrayUtils::get($payload, 'site_name');
                $site_email = ArrayUtils::get($payload, 'site_email');
                $site_domain = ArrayUtils::get($payload, 'site_domain');

                if(!$site_name || !Validate::isGenericName($site_name)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid site name"
                    ]);
                }

                if(!$site_email || !Validate::isEmail($site_email)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid site email"
                    ]);
                }

                if(!$site_domain || !Validate::isAbsoluteUrl($site_domain)) {
                    return $this->response([
                        "error" => true,
                        "message" => "<strong>$site_domain</strong> is not a valid url"
                    ]);
                }

                $site = new Site();
                $site->name = $site_name;
                $site->email = $site_email;
                $site->domain = rtrim($site_domain, '/');

                $success = $site->add();

                if(!$success) {
                    return $this->response([
                        "error" => true,
                        "message" => "Site could not be added. Refresh page and try again"
                    ]);
                }

                /** Add default site configuration */
                Configuration::updateValue('SITE_ID', $site->id, false, $site->id);
                Configuration::updateValue('SITE_NAME', $site->name, false, $site->id);
                Configuration::updateValue('SITE_EMAIL', $site->email, false, $site->id);
                Configuration::updateValue('SITE_DOMAIN', $site->domain, false, $site->id);

                $defaultCurrencyId = Currency::getIdByIsoCode('USD');
                Configuration::updateValue('CURRENCY_DEFAULT', $defaultCurrencyId, false, $site->id);

                Configuration::updateValue('ACADEMIC_WRITING_ENABLED', true, false, $site->id);
                Configuration::updateValue('DEFAULT_SERVICE_TYPE_ID', 37, false, $site->id); 
                Configuration::updateValue('CPP_PERCENTAGE', 100, false, $site->id); 

                $deadlines = $globals['deadlines']; 

                foreach($deadlines as $deadlineId => $deadline) {
                    $tarrif = new Deadline();
                    $tarrif->site_id = $site->id;
                    $tarrif->is_academic_writing = true;
                    $tarrif->academic_level_id = $deadline['academicLevel'];
                    $tarrif->name = $deadline['name'];
                    $tarrif->price_per_page = $deadline['pricePerPage'];
                    $tarrif->min_pages = $deadline['minPages'];
                    $tarrif->hrs_customer = $deadline['hrs'];
                    $tarrif->hrs_writer = $deadline['hrsWriter'];
                    $tarrif->add();
                }

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => $site->id,
                    'event' => 'add_site'
                ]);

                return $this->response([
                    'callback' => 'window.location.href="/my-sites/'.$site->id.'/edit";'
                ]);
                break;

            case 'general_settings':
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This site does not exist"
                    ]);
                }

                $site_name = ArrayUtils::get($payload, 'site_name');
                $site_email = ArrayUtils::get($payload, 'site_email');
                $site_email_logo = ArrayUtils::get($payload, 'site_email_logo');
                $site_brand_logo = ArrayUtils::get($payload, 'site_brand_logo');
                $site_domain = ArrayUtils::get($payload, 'site_domain');

                if(!$site_name || !Validate::isGenericName($site_name)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid site name"
                    ]);
                }

                if(!$site_email || !Validate::isEmail($site_email)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid site email"
                    ]);
                }

                if(!$site_domain || !Validate::isAbsoluteUrl($site_domain)) {
                    return $this->response([
                        "error" => true,
                        "message" => "<strong>$site_domain</strong> is not a valid url"
                    ]);
                }

                $site->name = $site_name;
                $site->email = $site_email;
                $site->domain = $site_domain;
                $site->update();

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => $site->id,
                    'event' => 'general_update'
                ]);

                Configuration::updateValue('SITE_ID', $site->id, false, $site->id);
                Configuration::updateValue('SITE_NAME', $site_name, false, $site->id);
                Configuration::updateValue('SITE_EMAIL', $site_email, false, $site->id);
                Configuration::updateValue('SITE_EMAIL_LOGO', $site_email_logo, false, $site->id);
                Configuration::updateValue('SITE_BRAND_LOGO', $site_brand_logo, false, $site->id);
                Configuration::updateValue('SITE_DOMAIN', $site_domain, false, $site->id);
                Configuration::updateValue('UK_FORM', ArrayUtils::has($payload, 'ukForm') ? true : false, false, $site->id);
                Configuration::updateValue('SHARED_AUTH', ArrayUtils::has($payload, 'shareCustomers') ? true : false, false, $site->id);
                Configuration::updateValue('TAWK_CHAT_CODE', ArrayUtils::get($payload, 'tawkChatCode'), true, $site->id);
                Configuration::updateValue('AUTH_REQUIRE_NAME', ArrayUtils::has($payload, 'requireName') ? true : false, false, $site->id);
                Configuration::updateValue('AUTH_REQUIRE_PHONE', ArrayUtils::has($payload, 'requirePhone') ? true : false, false, $site->id);
                Configuration::updateValue('SIGNUP_DEFAULT', ArrayUtils::has($payload, 'signupDefault') ? true : false, false, $site->id);
                Configuration::updateValue('DEFAULT_COURSELEVEL', (int) ArrayUtils::get($payload, 'default_course_level'), false, $site->id);
                Configuration::updateValue('URGENT_DEADLINE_MAX_PAGES', (int) ArrayUtils::get($payload, 'urgent_deadline_max_pages'), false, $site->id);
                Configuration::updateValue('CURRENCY_DEFAULT', (int) ArrayUtils::get($payload, 'default_currency'), false, $site->id);
                Configuration::updateValue('CPP_PERCENTAGE', (float) ArrayUtils::get($payload, 'cpp_percentage'), false, $site->id);

                Configuration::updateValue('ACADEMIC_WRITING_ENABLED', ArrayUtils::has($payload, 'academicwriting_enabled') ? true : false, false, $site->id);
                Configuration::updateValue('ARTICLE_WRITING_ENABLED', ArrayUtils::has($payload, 'articlewriting_enabled') ? true : false, false, $site->id);
                Configuration::updateValue('CALCULATIONS_ENABLED', ArrayUtils::has($payload, 'calculations_enabled') ? true : false, false, $site->id);
                Configuration::updateValue('PROGRAMMING_ENABLED', ArrayUtils::has($payload, 'programming_enabled') ? true : false, false, $site->id);

                Configuration::updateValue('DEFAULT_SERVICE_TYPE_ID', (int) ArrayUtils::get($payload, 'default_service_type'), false, $site->id);

                $default_deadlines = (array) ArrayUtils::get($payload, 'default_deadlines');
                $default_deadlines = implode(',', $default_deadlines);
                Configuration::updateValue('DEFAULT_DEADLINES', $default_deadlines , false, $site->id);

                Configuration::updateValue('ANONYMOUS_ENABLED', ArrayUtils::has($payload, 'anonymousId') ? true: false, false, $site->id);

                #pwa
                if(ArrayUtils::has($payload, 'pwa_enabled')) {
                    $pwa_app_icon = ArrayUtils::get($payload, 'pwa_app_icon');

                    if($pwa_app_icon && $pwa_app_icon != Configuration::get('PWA_APP_ICON')) {
                        $icon_dir = PROX_DIR_UPLOADS . $pwa_app_icon;
                      
                        $icon_info = explode('.', $pwa_app_icon);
                        $icon_ext = pathinfo($icon_dir, PATHINFO_EXTENSION);
    
                        $sizes = [72, 96, 128, 144, 152, 192, 384, 512];
                        foreach($sizes as $size) {
                            $fileName = $icon_info[0] . '-' . (int) $size . 'x' . (int) $size . '.' . $icon_ext;
                            $filePath = PROX_DIR_UPLOADS . $fileName;
                            ImageManager::resize($icon_dir, $filePath, $size, $size, $icon_ext);
                        }
                    }

                    Configuration::updateValue('PWA_APP_ICON', $pwa_app_icon, false, $site->id);
                }

                Configuration::updateValue('PWA_ENABLED', ArrayUtils::has($payload, 'pwa_enabled') ? true: false, false, $site->id);
                Configuration::updateValue('PWA_APP_NAME', ArrayUtils::get($payload, 'pwa_app_name'), false, $site->id);
                Configuration::updateValue('PWA_SHORT_NAME', ArrayUtils::get($payload, 'pwa_short_name'), false, $site->id);
                Configuration::updateValue('PWA_THEME_COLOR', ArrayUtils::get($payload, 'pwa_theme_color'), false, $site->id);
                Configuration::updateValue('PWA_BACKGROUND_COLOR', ArrayUtils::get($payload, 'pwa_background_color'), false, $site->id);

                Hook::exec('actionAfterSiteSettingsGeneralUpdate', [
                    'site_id' => $site->id,
                    'payload' => $payload
                ]);
                break;

            case 'features_settings':
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This site does not exist"
                    ]);
                }

                Configuration::updateValue('ACADEMIC_WRITING_ENABLED', ArrayUtils::has($payload, 'order_academic_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('PROGRAMMING_ENABLED', ArrayUtils::has($payload, 'order_programming_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('CALCULATIONS_ENABLED', ArrayUtils::has($payload, 'order_calculations_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('ARTICLE_WRITING_ENABLED', ArrayUtils::has($payload, 'order_article_enabled') ? true : false , false, $site->id);

                Configuration::updateValue('ORDER_FORM_COUPONS', ArrayUtils::has($payload, 'order_form_coupons') ? true : false , false, $site->id);
                Configuration::updateValue('DISCOUNTS_ENABLED', ArrayUtils::has($payload, 'discounts_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('INQUIRY_ENABLED', ArrayUtils::has($payload, 'order_inquiry_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('REQUEST_REFUND_ALLOWED', ArrayUtils::has($payload, 'request_refund_enabled') ? true : false , false, $site->id);

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => $site->id,
                    'event' => 'features_settings'
                ]);
                break;

            case 'add_services_settings':
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This site does not exist"
                    ]);
                }

                Configuration::updateValue('WRITER_SAMPLES_ENABLED', ArrayUtils::has($payload, 'writer_samples_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('PROGRESSIVE_DELIVERY_ENABLED', ArrayUtils::has($payload, 'p_delivery_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('EXPERT_PROOFREADING_ENABLED', ArrayUtils::has($payload, 'expert_proofreading_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('COPY_OF_SOURCES_ENABLED', ArrayUtils::has($payload, 'copy_of_sources_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('VIP_SUPPORT_ENABLED', ArrayUtils::has($payload, 'vip_support_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('PLAGIARISM_REPORT_ENABLED', ArrayUtils::has($payload, 'plagiarism_report_enabled') ? true : false , false, $site->id);
                Configuration::updateValue('DRAFT_OUTLINE_ENABLED', ArrayUtils::has($payload, 'draft_outline_enabled') ? true : false , false, $site->id);

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => $site->id,
                    'event' => 'add_services_settings'
                ]);
                break;

            case 'email_settings':
                $site = new Site( (int) $site_id );
                if(!Validate::isLoadedObject($site)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This site does not exist"
                    ]);
                }

                Configuration::updateValue('MAIL_METHOD', ArrayUtils::get($payload, 'mail_method'), false, $site->id);
                Configuration::updateValue('MAIL_SERVER', ArrayUtils::get($payload, 'smtp_server'), false, $site->id);
                Configuration::updateValue('MAIL_USER', ArrayUtils::get($payload, 'smtp_username'), false, $site->id);
                Configuration::updateValue('MAIL_PASSWD', ArrayUtils::get($payload, 'smtp_password'), false, $site->id);
                Configuration::updateValue('MAIL_SMTP_PORT', ArrayUtils::get($payload, 'smtp_port'), false, $site->id);
                Configuration::updateValue('MAIL_SMTP_ENCRYPTION', ArrayUtils::get($payload, 'smtp_encryption'), false, $site->id);
                
                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => $site->id,
                    'event' => 'email_settings'
                ]);
                break;
        }

        return $this->response([
			"success" => true,
			"message" => "System settings have been updated"
		]);
    }
}
