<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Exception;
use Proxim\Controller;
use Proxim\Currency as CurrencyModel;
use Proxim\Validate;
use Proxim\Util\ArrayUtils;
use Proxim\Hook;

class Currency extends Controller {
    public function currency() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $do = ArrayUtils::get($params, 'do');

        switch( $do ) {
            case 'add':
                $title = ArrayUtils::get($payload, 'title');
                if (!$title || !Validate::isName($title)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid currency title"
                    ]);
                }
        
                $iso_code = ArrayUtils::get($payload, 'iso_code');
                if (!$iso_code || !Validate::isCurrencyCode($iso_code)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid currency code"
                    ]);
                }
                
                $currencyExists = CurrencyModel::getIdByIsoCode($iso_code);
                if($currencyExists) {
                    return $this->response([
                        "error" => true,
                        "message" => "The currency you entered already exists"
                    ]);
                }
        
                $symbol = ArrayUtils::get($payload, 'symbol');
                if (!$symbol) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter the currency Symbol"
                    ]);
                } 
        
                $conversion_rate = (float) ArrayUtils::get($payload, 'conversion_rate', 1);
                
                $currency = new CurrencyModel();
                $currency->title = $title;
                $currency->iso_code = $iso_code;
                $currency->symbol = $symbol;
                $currency->conversion_rate = $conversion_rate;
                $currency->active = 1;
                
                $currency->add();

                Hook::exec('activitylog', [
                    'object' => 'currency',
                    'object_id' => $currency->id,
                    'event' => 'add_currency'
                ]);
        
                return $this->response([
                    'callback' => 'window.location.href="/system/currency/edit/'.$currency->id.'";'
                ]);
                break;
            
            case 'edit':
                $currency_id = ArrayUtils::get($payload, 'currency_id');

                $currency = new CurrencyModel( (int) $currency_id );
                if(!Validate::isLoadedObject($currency)) {
                    return $this->response([
                        "error" => true,
                        "message" => "This currency does not exist"
                    ]);
                }

                $title = ArrayUtils::get($payload, 'title');
                if (!$title || !Validate::isName($title)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid currency title"
                    ]);
                }
        
                $iso_code = ArrayUtils::get($payload, 'iso_code');
                if (!$iso_code || !Validate::isCurrencyCode($iso_code)) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter a valid currency code"
                    ]);
                }
                
                $symbol = ArrayUtils::get($payload, 'symbol');
                if (!$symbol) {
                    return $this->response([
                        "error" => true,
                        "message" => "Enter the currency Symbol"
                    ]);
                }
        
                $conversion_rate = (float) ArrayUtils::get($payload, 'conversion_rate', 1);

                $active = ArrayUtils::has($payload, 'active') ? true : false;

                $currency->title = $title;
                $currency->iso_code = $iso_code;
                $currency->symbol = $symbol;
                $currency->conversion_rate = $conversion_rate;
                $currency->active = $active;
                
                $currency->update();

                Hook::exec('activitylog', [
                    'object' => 'currency',
                    'object_id' => $currency->id,
                    'event' => 'edit_currency'
                ]);

                return $this->response([
                    "success" => true,
                    "message" => "System settings have been updated"
                ]);
                break;

            default:
                $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }
}