<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Db;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Mail;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\MailTemplate;

class Auth extends Controller {
    public function signin() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        if ($user->isLogged()) {
            return $this->response([
				'callback' => 'window.location.reload();'
			]);	
        }

        $email = trim(ArrayUtils::get($payload, 'email'));
		if ( !$email || !Validate::isEmail($email) ) {
			return $this->response([
				"error" => true,
				"message" => "Enter a valid email"
			]);
        }
        
        $password = ArrayUtils::get($payload, 'password');
		if ( !$password ) {
            return $this->response([
				"error" => true,
				"message" => "Password is required"
			]);
		}
		
		/* check brute-force attack detection */
		if(Configuration::get('BRUTE_FORCE_DETECTION_ENABLED')) {
			$brute_force_lockout_time = Configuration::get('BRUTE_FORCE_LOCKOUT_TIME', null, 10) * 60; /* convert to minutes */
			if( ($user->failed_login_ip == $app->request->getIp() ) && 
				($user->failed_login_count >= Configuration::get('BRUTE_FORCE_BAD_LOGIN_LIMIT')) && 
				(time() - strtotime($user->first_failed_login) <  $brute_force_lockout_time) 
			) {
				return $this->response([
					"error" => true,
					"message" => "Your account currently locked out, Please try again later!"
				]);
            }
		}

		Hook::exec('actionAuthenticationBefore');

        $user = $user->getByEmail(
			$email,
			$password
		);

		if (!Validate::isLoadedObject($user)) {
			/* check brute-force attack detection */
			if(Configuration::get('BRUTE_FORCE_DETECTION_ENABLED')) {
				if( time() - strtotime($user->first_failed_login)  >  Configuration::get('BRUTE_FORCE_LOCKOUT_TIME') ) {
					$user->first_failed_login++;
					$user->failed_login_ip = $app->request->getIp();
                } else {
					$user->failed_login_count++;
				}
				$user->update();
			}

            return $this->response([
				"error" => true,
				"message" => "Incorrect email or password"
            ]);
		}
		
		if ($user->is_banned) {
			return $this->response([
				"error" => true,
				"message" => "Your account has been disabled. If you think this is an error please contact us at <strong>" . Configuration::get('SITE_EMAIL') . "</strong>."
            ]);
		}  
		
		Hook::exec('actionAuthentication', ['employee' => $user]);

		$user->failed_login_count = 0;
        $user->last_login = DateUtils::now();
		$user->update();

        $app->updateUser( $user );

		Hook::exec('activitylog', [
            'object' => 'employee',
            'object_id' => $user->id,
            'event' => 'signin'
        ]); 

        return $this->response([
            'callback' => 'window.location.reload();'
        ]);	
    }

    public function signup() {
        $app = $this->app;
		$payload = $app->request->post();
		$user = $app->user;

        if ($user->isLogged()) {
            return $this->response([
				'callback' => 'window.location.reload();'
			]);	
        }

        $first_name = ArrayUtils::get($payload, 'first_name');
		if ( $first_name && !Validate::isName($first_name) ) {
			return $this->response([
				"error" => true,
				"message" => "Enter a valid first name"
			]);
		}
		
		if ( Tools::strlen($first_name) > 20 ) {
			return $this->response([
				"error" => true,
				"message" => "Your first name must be less than 20 characters long. Please try another"
			]);
		}
		if ( Tools::strlen($first_name) < 3 ) {
			return $this->response([
				"error" => true,
				"message" => "Your first name must be at least 3 characters long. Please try another"
			]);
		}

		$last_name = ArrayUtils::get($payload, 'last_name');
		if ( $last_name && !Validate::isName($last_name) ) {
			return $this->response([
				"error" => true,
				"message" => "Enter a valid last name"
			]);
		}

		if ( Tools::strlen($last_name) > 20 ) {
			return $this->response([
				"error" => true,
				"message" => "Your last name must be less than 20 characters long. Please try another"
			]);
		}
		if ( Tools::strlen($last_name) < 3 ) {
			return $this->response([
				"error" => true,
				"message" => "Your last name must be at least 3 characters long. Please try another"
			]);
		}

        $email = ArrayUtils::get($payload, 'email');
		if ( !$email || !Validate::isEmail($email) ) {
			return $this->response([
				"error" => true,
				"message" => "Enter a valid email address"
			]);
        }
        
        $employee = $user->getByEmail($email);
		if (  Validate::isLoadedObject($employee) ) {
			return $this->response([
				"error" => true,
				"message" => "Sorry, it looks like <strong>".$email."</strong> belongs to an existing account"
			]);
        }
        
        $gender = ArrayUtils::get($payload, 'gender');
		if ( !$gender || !in_array($gender, array('male', 'female')) ) {
			return $this->response([
				"error" => true,
				"message" => "Select a valid gender"
			]);
		}

        $password = ArrayUtils::get($payload, 'password');
		if ( !$password || !Validate::isPasswd($password) ) {
            return $this->response([
				"error" => true,
				"message" => "Password must be long than 6 characters"
			]);
        }

		Hook::exec('actionSignupBefore');
 
        $user = new Employee();
		$user->employee_group =  (PROX_ACTIVE_THEME == 'phantom') ? Employee::GROUP_WRITER : Employee::GROUP_APPLICANT;
		$user->email = $email;
		$user->first_name = $first_name; 
		$user->last_name = $last_name; 
        $crypto = new Hashing();
		$user->password = $crypto->hash(
            $password,
            PROX_COOKIE_KEY
		);
		
        $user->gender = $gender; 
		$user->policy_accepted = 1;
		$user->policy_accepted_at = DateUtils::now();
        $user->reg_date = DateUtils::now();
        
        $success = $user->add(true, true);
        if ( !$success ) {
            return $this->response([
                "error" => true,
                "message" => "Sorry, there was a problem creating your account"
            ]);	
        }

		Hook::exec('actionSignupBefore', array(
			'employee' => $user
		));

        $app->updateUser( $user );

		Hook::exec('activitylog', [
            'object' => 'employee',
            'object_id' => $user->id,
            'event' => 'signup'
        ]);

        return $this->response([
            'callback' => 'window.location.reload();'
        ]);	
	}   
	
	public function resetPassword() {
		$app = $this->app;
		$payload = $app->request->post();
		$user = $app->user;

        if ($user->isLogged()) {
            return $this->response([
				'callback' => 'window.location.reload();'
			]);	
		}

		$email = ArrayUtils::get($payload, 'email');

		if (!Validate::isEmail($email)) {
			return $this->response([
				"error" => true,
				"message" => "Please enter a valid email address"
			]);
		}

		$user = $user->getByEmail($email);
		if (!Validate::isLoadedObject($user)) {
			return $this->response([
				"error" => true,
				"message" => "Sorry, it looks like ".$email." doesn't belong to any account"
			]);
		}

		$user->stampResetPasswordToken();
		$user->update();

		Hook::exec('activitylog', [
            'object' => 'employee',
            'object_id' => $user->id,
            'event' => 'reset_password'
        ]);

		$mailParams = array(
			'email' => $user->email,
			'username' => $user->first_name . " " . $user->last_name,
			'reset_url' => Configuration::get('SITE_DOMAIN') . '/reset/' . $user->reset_password_token,
		);

		if (
			Mail::Send(
				MailTemplate::TEMPLATE_EMPLOYEE_PASSWORD_CHANGE,
				'Confirm password change',
				$mailParams,
				$user->email
			)
		) {
			$this->modal("SUCCESS", "Check Your Email", "We sent you an email with a confirmation link. Click on it to continue to reset your password");
		} else {
			return $this->response([
				"error" => true,
				"message" => "Activation key email could not be sent!"
			]);
		}
	}

	public function changePassword() {
		$app = $this->app;
		$payload = $app->request->post();
		$user = $app->user;

        if ($user->isLogged()) {
            return $this->response([
				'callback' => 'window.location.reload();'
			]);	
		}

		$reset_key = ArrayUtils::get($payload, 'reset_key');
		$new = ArrayUtils::get($payload, 'new');
		$confirm = ArrayUtils::get($payload, 'confirm');

		$sql = new DbQuery();
		$sql->select('e.`employee_id`');
		$sql->from('employee', 'e');
		$sql->where('e.`reset_password_token` = \'' . pSQL($reset_key) . '\'');

		$employee_id = Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($sql);

		$user = new Employee( (int) $employee_id );
		if ( !Validate::isLoadedObject($user) ) {
			return $this->response([
				"error" => true,
				"message" => "Recovery link is expired"
			]);
		} 

		if ( $user->getValidResetPasswordToken() != $reset_key ) {
			return $this->response([
				"error" => true,
				"message" => "Recovery link is expired"
			]);
		}

		/* validate all fields */
		if(!$new || !$confirm ) {
			return $this->response([
				'error' => true,
				'message' => 'You must fill in all of the fields'
			]);
		}

		/* validate new password */
		if($new != $confirm ) {
			return $this->response([
				'error' => true,
				'message' => 'Your passwords do not match'
			]);
		}

		if( !Validate::isPasswd($new) ) {
			return $this->response([
				'error' => true,
				'message' => 'New password must be at least 6 characters long. Please try another'
			]);
		}

		$crypto = new Hashing();
		$user->password = $crypto->hash(
			$new,
			PROX_COOKIE_KEY
		);
        $user->last_passwd_gen = DateUtils::now();
		$user->removeResetPasswordToken();
		$user->update();

		Hook::exec('activitylog', [
            'object' => 'employee',
            'object_id' => $user->id,
            'event' => 'change_password'
        ]);

		return $this->modal("SUCCESS", "Password updated", "Your password has been updated. Now you can log in", $app->base_uri . '/signin');
	}
}