<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Api;

use Aws\S3\S3Client;
use Db;
use Exception;
use Proxim\Controller;
use Proxim\Database\DbQuery;
use Proxim\Configuration;
use Proxim\Site\Site;
use Proxim\Coupon;
use Proxim\Currency;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Order\Order;
use Proxim\Order\File;
use Proxim\Order\Message;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Crypto\Hashing;
use Proxim\Hook;
use Proxim\ImageManager;
use Proxim\Order\Rating;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\ApplicationEssay;
use Proxim\Preference\Deadline;
use Proxim\Preference\Descriptor;
use Proxim\Preference\ItemPricing;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Preference\UploadType;
use Proxim\Preference\WritingGuide;
use Proxim\Preference\WritingPrompt;
use Proxim\Ticket\Ticket;
use Proxim\Ticket\TicketCategory;
use Proxim\Uploader;
use Proxim\User\EmployeeFile;
use Proxim\User\EmployeePool;

class Admin extends Controller {
    
    public function settings() {
        $app = $this->app;
        $payload = $app->request->post();
        $params = $app->request->get();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $edit = ArrayUtils::get($params, 'edit');

        switch($edit) {
            case 'general_settings':

                $notification_emails = ArrayUtils::get($payload, 'notification_emails');

                if(!empty($notification_emails)) {
                    $notification_emails = explode(',', $notification_emails);
                    foreach($notification_emails as $notification_email) {
                        $notification_email = trim($notification_email);
                        if(!Validate::isEmail($notification_email)) {
                            return $this->response([
                                "error" => true,
                                "message" => "<strong>" . $notification_email . "</strong> is not a valid email"
                            ]);
                        }
                    }
                } else {
                    $notification_emails = array();
                }

                Configuration::updateValue('SITE_LOGO', ArrayUtils::get($payload, 'site_logo'));
                Configuration::updateValue('SITE_DARK_LOGO', ArrayUtils::get($payload, 'site_dark_logo'));
                Configuration::updateValue('SITE_FAVICON', ArrayUtils::get($payload, 'site_favicon'));
                Configuration::updateValue('SITE_EMAIL_LOGO', ArrayUtils::get($payload, 'site_email_logo'));
                Configuration::updateValue('SITE_NAME', ArrayUtils::get($payload, 'site_name'));
                Configuration::updateValue('SITE_EMAIL', ArrayUtils::get($payload, 'site_email'));
                Configuration::updateValue('NOTIFICATION_EMAILS', implode(',', $notification_emails) );
                Configuration::updateValue('SITE_DOMAIN', ArrayUtils::get($payload, 'site_domain'));
                Configuration::updateValue('DEFAULT_TIMEZONE', ArrayUtils::get($payload, 'default_timezone'));
                Configuration::updateValue('REGISTRATION_ENABLED', ArrayUtils::has($payload, 'registration_enabled') ? true : false );
                Configuration::updateValue('COOKIE_CONSENT_ENABLED', ArrayUtils::has($payload, 'cookie_consent_enabled') ? true : false );
                Configuration::updateValue('ARTICLE_WRITING_ENABLED', ArrayUtils::has($payload, 'article_writing_enabled') ? true : false );
                Configuration::updateValue('CSS_CUSTOMIZE', ArrayUtils::has($payload, 'css_customize') ? true : false );
                Configuration::updateValue('HEADER_COLOR', ArrayUtils::get($payload, 'header_color'));
                Configuration::updateValue('BUTTON_PRIMARY', ArrayUtils::get($payload, 'btn_primary_color'));
                Configuration::updateValue('MENU_BACKGROUND', ArrayUtils::get($payload, 'menu_background_color'));
                Configuration::updateValue('CUSTOM_CSS', ArrayUtils::get($payload, 'custom_css'));
                Configuration::updateValue('COOKIE_LIFETIME', (int) ArrayUtils::get($payload, 'cookie_lifetime'));

                #pwa
                $pwa_app_icon = ArrayUtils::get($payload, 'pwa_app_icon');

                if($pwa_app_icon && $pwa_app_icon != Configuration::get('PWA_APP_ICON')) {
                    $icon_dir = PROX_DIR_UPLOADS . $pwa_app_icon;
                  
                    $icon_info = explode('.', $pwa_app_icon);
                    $icon_ext = pathinfo($icon_dir, PATHINFO_EXTENSION);

                    $sizes = [72, 96, 128, 144, 152, 192, 384, 512];
                    foreach($sizes as $size) {
                        $fileName = $icon_info[0] . '-' . (int) $size . 'x' . (int) $size . '.' . $icon_ext;
                        $filePath = PROX_DIR_UPLOADS . $fileName;
                        ImageManager::resize($icon_dir, $filePath, $size, $size, $icon_ext);
                    }
                }

                Configuration::updateValue('PWA_ENABLED', ArrayUtils::has($payload, 'pwa_enabled') ? true: false);
                Configuration::updateValue('PWA_APP_ICON', $pwa_app_icon);
                Configuration::updateValue('PWA_APP_NAME', ArrayUtils::get($payload, 'pwa_app_name'));
                Configuration::updateValue('PWA_SHORT_NAME', ArrayUtils::get($payload, 'pwa_short_name'));
                Configuration::updateValue('PWA_THEME_COLOR', ArrayUtils::get($payload, 'pwa_theme_color'));
                Configuration::updateValue('PWA_BACKGROUND_COLOR', ArrayUtils::get($payload, 'pwa_background_color'));

                $customizePayload = ArrayUtils::get($payload, 'customize', []);
                $customizeDefaultValues = array(
                    'style' => 'ui-default',
                    'header' => 'is-light',
                    'aside' => 'is-light',
                    'skin' => 'default'
                );
                
                Configuration::updateValue('CUSTOMIZE_UI_STYLE', ArrayUtils::get($customizePayload, 'style', $customizeDefaultValues['style']));
                Configuration::updateValue('CUSTOMIZE_HEADER_STYLE', ArrayUtils::get($customizePayload, 'header', $customizeDefaultValues['header']));
                Configuration::updateValue('CUSTOMIZE_ASIDE_STYLE', ArrayUtils::get($customizePayload, 'aside', $customizeDefaultValues['aside']));
                Configuration::updateValue('CUSTOMIZE_UI_SKIN', ArrayUtils::get($customizePayload, 'skin', $customizeDefaultValues['skin']));

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'general_settings'
                ]);
                break;

            case 'orders_settings':

                Configuration::updateValue('ATTACH_COMPLETED_FILES', ArrayUtils::has($payload, 'attach_completed_files') ? true : false );
                Configuration::updateValue('EDITOR_ACCOUNT_ENABLED', ArrayUtils::has($payload, 'editor_account') ? true : false );
                Configuration::updateValue('PAY_DELIVERED_ORDERS', ArrayUtils::has($payload, 'pay_delivered_orders') ? true : false );
                Configuration::updateValue('CONFIRM_ORDER_ASSIGN', ArrayUtils::has($payload, 'confirm_order_assign') ? true : false );

                Configuration::updateValue('HIGHLIGHT_ORDER_DEADLINE', ArrayUtils::has($payload, 'higlight_order_deadline') ? true : false );
                Configuration::updateValue('DEADLINE_DUE_HOURS', (int) ArrayUtils::get($payload, 'deadline_due_hours') );

                Configuration::updateValue('SUBADMIN_CLIENT_COST', ArrayUtils::has($payload, 'subadmin_client_cost') ? true : false );
                Configuration::updateValue('SUBADMIN_WRITER_INFO', ArrayUtils::has($payload, 'subadmin_writer_info') ? true : false );
                Configuration::updateValue('AUTOASSIGN_REQUESTED_WRITER', ArrayUtils::has($payload, 'autoassign_requested_writer') ? true : false );
                
                Configuration::updateValue('ORDER_MANAGER_ENABLED', ArrayUtils::has($payload, 'order_manager_enabled') ? true : false );
                Configuration::updateValue('ORDER_MANAGER_PUBLIC', ArrayUtils::has($payload, 'order_manager_public') ? true : false );
                Configuration::updateValue('ORDER_MANAGER_CPP_TYPE', ArrayUtils::get($payload, 'ordermanager_cpp_type', null, "fixed"));
                Configuration::updateValue('ORDER_MANAGER_AMOUNT_CPP', ArrayUtils::get($payload, 'ordermanager_cost_per_page', null, 0));
                Configuration::updateValue('ORDER_MANAGER_PERCENTAGE_CPP', ArrayUtils::get($payload, 'ordermanager_percent_per_page', null, 0) );

                Configuration::updateValue('ORDER_AUTOAPPROVE_TIME', ArrayUtils::get($payload, 'autoapprove_time'));

                if(ArrayUtils::has($payload, 'words_per_page')) {
                    $words_per_page = (int) ArrayUtils::get($payload, 'words_per_page');
                    if(!$words_per_page) {
                        return $this->response([
                            "error" => true,
                            "message" => "Words Per Page must be more than zero"
                        ]);
                    }

                    Configuration::updateValue('WORDS_PER_PAGE', $words_per_page);
                }

                $order_autoincrement = (int) ArrayUtils::get($payload, 'order_autoincrement');
                if(!$order_autoincrement) {
                    return $this->response([
                        "error" => true,
                        "message" => "Order autoincrement value should be more than zero"
                    ]);
                }
                Configuration::updateValue('ORDER_AUTOINCREMENT', $order_autoincrement);

                Configuration::updateValue('NEW_ORDER_STATUS', (int) ArrayUtils::get($payload, 'new_order_status'));

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'orders_settings'
                ]);
                break;

            case 'email_settings':
                Configuration::updateValue('MAIL_METHOD', ArrayUtils::get($payload, 'mail_method'));
                Configuration::updateValue('MAIL_SERVER', ArrayUtils::get($payload, 'smtp_server'));
                Configuration::updateValue('MAIL_USER', ArrayUtils::get($payload, 'smtp_username') );
                Configuration::updateValue('MAIL_PASSWD', ArrayUtils::get($payload, 'smtp_password') );
                Configuration::updateValue('MAIL_SMTP_PORT', ArrayUtils::get($payload, 'smtp_port') );
                Configuration::updateValue('MAIL_SMTP_ENCRYPTION', ArrayUtils::get($payload, 'smtp_encryption') );

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'email_settings'
                ]);
                break;

            case 'payments_settings':
                Configuration::updateValue('WRITER_PAYMENT_REQUEST_LIMIT', ArrayUtils::get($payload, 'writer_payment_request_limit'));
                Configuration::updateValue('EDITOR_PAYMENT_REQUEST_LIMIT', ArrayUtils::get($payload, 'editor_payment_request_limit'));
                $payment_dates = ArrayUtils::get($payload, 'payment_dates', array());
                Configuration::updateValue('PAYMENT_DATES', implode(',', $payment_dates) );

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'payments_settings'
                ]);
                break;

            case 'writer_payments':
                # writer cpp type
                Configuration::updateValue('WRITER_PAY_CURRENCY', ArrayUtils::get($payload, 'writer_pay_currency'));

                Configuration::updateValue('WRITER_CPP_TYPE', ArrayUtils::get($payload, 'writer_cpp_type', null, "fixed"));
                Configuration::updateValue('WRITER_AMOUNT_CPP', ArrayUtils::get($payload, 'writer_cost_per_page', null, 0));
                Configuration::updateValue('WRITER_PERCENTAGE_CPP', ArrayUtils::get($payload, 'writer_percent_per_page', null, 0) );

                # writer cpw type
                Configuration::updateValue('WRITER_CPW_TYPE', ArrayUtils::get($payload, 'writer_cpw_type', null, "fixed"));
                Configuration::updateValue('WRITER_AMOUNT_CPW', ArrayUtils::get($payload, 'writer_cost_per_word', null, 0));
                Configuration::updateValue('WRITER_PERCENTAGE_CPW', ArrayUtils::get($payload, 'writer_percent_per_word', null, 0) );

                Configuration::updateValue('WRITER_W_CATEGORY_COST', ArrayUtils::get($payload, 'writer_category_cost', null, 0) );
                Configuration::updateValue('WRITER_COMPLEX_ASSIGNMENT_COST', ArrayUtils::get($payload, 'complex_assignment_cost', null, 0) );
                Configuration::updateValue('WRITER_USED_SOURCES_COST', ArrayUtils::get($payload, 'used_sources_cost', null, 0) );
                Configuration::updateValue('WRITER_PROGRESSIVE_DELIVERY_COST', ArrayUtils::get($payload, 'p_delivery_cost', null, 0) );
                Configuration::updateValue('WRITER_PROVIDE_ME_SAMPLES_COST', ArrayUtils::get($payload, 'writer_sample_cost', null, 0) );
                Configuration::updateValue('WRITER_EXPERT_PROOFREADING_COST', ArrayUtils::get($payload, 'expert_proofreading_cost', null, 0) );
                Configuration::updateValue('WRITER_VIP_SUPPORT_COST', ArrayUtils::get($payload, 'vip_support_cost', null, 0) );
                Configuration::updateValue('WRITER_PLAGIARISM_REPORT_COST', ArrayUtils::get($payload, 'plagiarism_report_cost', null, 0) );
                Configuration::updateValue('WRITER_DRAFT_OUTLINE_COST', ArrayUtils::get($payload, 'draft_outline_cost', null, 0) );
                Configuration::updateValue('WRITER_PREFFERED_WRITER_COST', ArrayUtils::get($payload, 'preffered_writer_cost', null, 0) );

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'writer_payments'
                ]);
                break;

            case 'editor_payments':
                # editor cpp
                Configuration::updateValue('EDITOR_CPP_TYPE', ArrayUtils::get($payload, 'editor_cpp_type', null, "fixed"));
                Configuration::updateValue('EDITOR_AMOUNT_CPP', ArrayUtils::get($payload, 'editor_cost_per_page', null, 0));
                Configuration::updateValue('EDITOR_PERCENTAGE_CPP', ArrayUtils::get($payload, 'editor_percent_per_page', null, 0) );

                # editor cpw
                Configuration::updateValue('EDITOR_CPW_TYPE', ArrayUtils::get($payload, 'editor_cpw_type', null, "fixed"));
                Configuration::updateValue('EDITOR_AMOUNT_CPW', ArrayUtils::get($payload, 'editor_cost_per_word', null, 0));
                Configuration::updateValue('EDITOR_PERCENTAGE_CPW', ArrayUtils::get($payload, 'editor_percent_per_word', null, 0) );

                Configuration::updateValue('EDITOR_W_CATEGORY_COST', ArrayUtils::get($payload, 'writer_category_cost', null, 0) );
                Configuration::updateValue('EDITOR_COMPLEX_ASSIGNMENT_COST', ArrayUtils::get($payload, 'complex_assignment_cost', null, 0) );
                Configuration::updateValue('EDITOR_USED_SOURCES_COST', ArrayUtils::get($payload, 'used_sources_cost', null, 0) );
                Configuration::updateValue('EDITOR_PROGRESSIVE_DELIVERY_COST', ArrayUtils::get($payload, 'p_delivery_cost', null, 0) );
                Configuration::updateValue('EDITOR_PROVIDE_ME_SAMPLES_COST', ArrayUtils::get($payload, 'writer_sample_cost', null, 0) );
                Configuration::updateValue('EDITOR_EXPERT_PROOFREADING_COST', ArrayUtils::get($payload, 'expert_proofreading_cost', null, 0) );
                Configuration::updateValue('EDITOR_VIP_SUPPORT_COST', ArrayUtils::get($payload, 'vip_support_cost', null, 0) );
                Configuration::updateValue('EDITOR_PLAGIARISM_REPORT_COST', ArrayUtils::get($payload, 'plagiarism_report_cost', null, 0) );
                Configuration::updateValue('EDITOR_DRAFT_OUTLINE_COST', ArrayUtils::get($payload, 'draft_outline_cost', null, 0) );

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'editor_payments'
                ]);
                break;

            case 'uploads':
                
                Configuration::updateValue('SHARED_FOLDER_PERMISSIONS', ArrayUtils::has($payload, 'shared_folder_permissions') ? true : false );
                Configuration::updateValue('INTERNAL_FILE_STORAGE', ArrayUtils::has($payload, 'internal_file_storage') ? true : false );
                Configuration::updateValue('ALLOWED_EXTENSIONS', ArrayUtils::get($payload, 'file_extensions'));

                Hook::exec('activitylog', [
                    'object' => 'site',
                    'object_id' => PROX_SITE_ID,
                    'event' => 'uploads'
                ]);
                break;

            case 's3':
                $s3_enabled = ArrayUtils::has($payload, 's3_enabled') ? true : false;
                Configuration::updateValue('S3_ENABLED', $s3_enabled);

                if($s3_enabled) {
                    /* if enabled is set -> disable all other options [digitalocean] */
                    Configuration::updateValue('DIGITALOCEAN_ENABLED', false);
                }
                
                Configuration::updateValue('S3_BUCKET', ArrayUtils::get($payload, 's3_bucket'));
                Configuration::updateValue('S3_REGION', ArrayUtils::get($payload, 's3_region'));
                Configuration::updateValue('S3_KEY', ArrayUtils::get($payload, 's3_key'));
                Configuration::updateValue('S3_SECRET', ArrayUtils::get($payload, 's3_secret'));
                break;

            case 'digitalocean':
                $digitalocean_enabled = ArrayUtils::has($payload, 'digitalocean_enabled') ? true : false;
                Configuration::updateValue('DIGITALOCEAN_ENABLED', $digitalocean_enabled);

                if($digitalocean_enabled) {
                    /* if enabled is set -> disable all other options [s3] */
                    Configuration::updateValue('S3_ENABLED', false);
                }

                Configuration::updateValue('DIGITALOCEAN_SPACE_NAME', ArrayUtils::get($payload, 'digitalocean_space_name'));
                Configuration::updateValue('DIGITALOCEAN_SPACE_REGION', ArrayUtils::get($payload, 'digitalocean_space_region'));
                Configuration::updateValue('DIGITALOCEAN_KEY', ArrayUtils::get($payload, 'digitalocean_key'));
                Configuration::updateValue('DIGITALOCEAN_SECRET', ArrayUtils::get($payload, 'digitalocean_secret'));
                break;
        }

        return $this->response([
			"success" => true,
			"message" => "System settings have been updated"
		]);
    }

    public function test() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;
        
        $handle = ArrayUtils::get($payload, 'handle');

        Hook::exec('actionBeforeTestConnection', [
            'payload' => $payload
        ]);
 
        switch($handle) {
            case 's3':
                /* test */
                global $globals;
                $uploader = new Uploader();
				$uploader->setAcceptTypes($globals['extensions']);

                try {
                    $uploader->awsS3Test();
                    /* return */
                    return $this->response( array('success' => true, 'message' => "Connection established successfully") );
                } catch( Exception $e ) {
                    return $this->response( array('error' => true, 'message' => $e->getMessage()) );
                }
                break;

            case 'digitalocean':
                /* test */
                global $globals;
                $uploader = new Uploader();
				$uploader->setAcceptTypes($globals['extensions']);
                
                try {
                    $uploader->digitaloceanSpaceTest();
                    /* return */
                    return $this->response( array('success' => true, 'message' => "Connection established successfully") );
                } catch( Exception $e ) {
                    return $this->response( array('error' => true, 'message' => $e->getMessage()) );
                }
                break;
        }

        Hook::exec('actionAfterTestConnection', [
            'payload' => $payload
        ]);
    }

    public function getPaymentRequestOrders( $request_id ) {
        $app = $this->app;
        $this->assignVariables();
        $user = $app->user;
        $smarty = $app->smarty;

        // initialize the return array
        $return = array();

        $sql = new DbQuery();
        $sql->select('pr.*');
        $sql->from('payment_request', 'pr');
        $sql->where('pr.`payment_request_id` = ' . (int) $request_id );
        $payment_request = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);

        if (!$payment_request) {
	        return $this->modal("ERROR", "Error", "This request does not exists");
        }

        $employee = new Employee( (int) $payment_request['user_id'] );
        if(!Validate::isLoadedObject($employee)) {
	        return $this->modal("ERROR", "Error", "This request does not exists");
        }

        $currencyFormat = $app->writer_currency;
        
        $sql = new DbQuery();
        $sql->select('pro.order_id');
        $sql->from('payment_request_order', 'pro');
        $sql->where('pro.`payment_request_id` = ' . (int) $payment_request['payment_request_id'] );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $orders = array();
        $subtotal_pay = $total_pay = $completed_orders_pay = $done_orders_pay = 0;

        foreach( $result as $order ) {
            $order = new Order( (int) $order['order_id'] );
            if( Validate::isLoadedObject($order) ) {

                if($order->status_id == Order::FINISHED) {
                    if($employee->is_sub_admin) {
                        $completed_orders_pay += $order->ordermanager_pay;
                    } elseif($employee->is_editor) {
                        $completed_orders_pay += $order->editor_pay;
                    } elseif($employee->is_writer) {
                        $completed_orders_pay += $order->writer_pay;
                    }
                } elseif ( $order->status_id == Order::DELIVERED ) {
                    if($employee->is_sub_admin) {
                        $done_orders_pay += $order->ordermanager_pay;
                    } elseif($employee->is_editor) {
                        $done_orders_pay += $order->editor_pay;
                    } elseif($employee->is_writer) {
                        $done_orders_pay += $order->writer_pay;
                    }
                }

                $orders[] = $order->present(false);
            }
        }
        
        $subtotal_pay = $completed_orders_pay+$done_orders_pay;
        $total_pay = ($subtotal_pay+$payment_request['total_credit'])-$payment_request['total_fine'];

        $smarty->assign([
            'base_uri' => $app->base_uri,
            'employee' => (array) $employee,
            'done_orders_pay' => (float) $done_orders_pay,
            'completed_orders_pay' => (float) $completed_orders_pay,
            'total_credit' => (float) $payment_request['total_credit'],
            'total_fine' => (float) $payment_request['total_fine'],
            'subtotal_pay' => (float) $subtotal_pay,
            'total_pay' => (float) $total_pay,
            'orders' => $orders,
            'currencyFormat' => $currencyFormat
        ]);

        /* return */
        $return['template'] = $smarty->fetch("ajax.payment-request-orders.tpl");
        $return['callback'] = "$('#modal').modal('show'); $('.modal-content:last').html(response.template);";
        
        return $this->response($return);
    }

    public function withdraw() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin) {
	        return $this->modal("ERROR", "Error", "You don't have the right permission to access this");
        }

        $payment_request_id = (int) ArrayUtils::get($payload, 'id');
        $handle =  ArrayUtils::get($payload, 'handle');

        $sql = new DbQuery();
        $sql->select('pr.*');
        $sql->from('payment_request', 'pr');
        $sql->where('pr.`payment_request_id` = ' . (int) $payment_request_id );
        $payment_request = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);

        if (!$payment_request) {
	        return $this->modal("ERROR", "Error", "This request does not exists");
        } 

        $employee = new Employee( (int) $payment_request['user_id'] );
        if(!Validate::isLoadedObject($employee)) {
	        return $this->modal("ERROR", "Error", "This request does not exists");
        }

        $sql = new DbQuery();
        $sql->select('pro.order_id');
        $sql->from('payment_request_order', 'pro');
        $sql->where('pro.`payment_request_id` = ' . (int) $payment_request['payment_request_id'] );
        $orders = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        switch( $handle ) {
            /* approve request */
            case 'approve':
                $success = Db::getInstance()->update('payment_request', array(
                    'status' => '1',
                    'paid_at' => DateUtils::now(),
                ), 'payment_request_id = ' . (int) $payment_request_id, 1, true);
                
                if (!$success) {
	                return $this->modal("ERROR", "Error", "Could not approve this request. Try again later");
                }

                foreach( $orders as $order ) {
                    $order = new Order( (int) $order['order_id'] );
                    if( Validate::isLoadedObject($order) ) {
                        if ($employee->is_writer) {
                            $order->is_writer_paid = true;
                            $order->writer_paid_at = DateUtils::now();
                            $order->update();
                        } elseif( $employee->is_editor ) {
                            $order->is_editor_paid = true;
                            $order->editor_paid_at = DateUtils::now();
                            $order->update();
                        } elseif( $employee->is_sub_admin ) {
                            $order->is_ordermanager_paid = true;
                            $order->ordermanager_paid_at = DateUtils::now();
                            $order->update();
                        }
                    }
                }

                $last_pay = (float) $payment_request['total_pay'];

                $employee->last_pay = $last_pay;
                $employee->total_paid = $employee->total_paid + $last_pay;
                $employee->update();

                Hook::exec('activitylog', [
                    'object' => 'payment_request',
                    'object_id' => (int) $payment_request['payment_request_id'],
                    'event' => 'approve_payment_request'
                ]);

                Hook::exec('actionUpdateRequestedAdditionalPayments', array(
                    'employee_id' => $employee->id,
                    'resolved' => true
                )); 

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);
                break;

            /* decline request */
            case 'decline':
                
                $success = Db::getInstance()->update('payment_request', array(
                    'status' => '-1',
                ), 'payment_request_id = ' . (int) $payment_request_id, 1, true);
                
                if (!$success) {
	                return $this->modal("ERROR", "Error", "Could not decline this request. Try again later");
                }

                foreach( $orders as $order ) {
                    Db::getInstance()->delete('payment_request_order', 'order_id = ' . (int) $order['order_id']);
                }

                Hook::exec('actionUpdateRequestedAdditionalPayments', array(
                    'employee_id' => $employee->id,
                    'resolved' => false
                )); 

                Hook::exec('activitylog', [
                    'object' => 'payment_request',
                    'object_id' => (int) $payment_request['payment_request_id'],
                    'event' => 'decline_payment_request'
                ]);

                return $this->response([
                    'callback' => 'window.location.reload();'
                ]);
                break;

            default:
                return $this->response([
                    "error" => true,
                    "message" => "Error 404. Resource not found"
                ]);
                break;
        }
    }

    public function delete() {
        $app = $this->app;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $this->modal("ERROR", "Error", "You don't have the right permission to access this"); 
        }

        $id = ArrayUtils::get($payload, 'id');
        $handle = ArrayUtils::get($payload, 'handle');

		Hook::exec('actionBeforeAdminDelete', [
            'handle' => $handle,
            'id' => $id,
        ]);

        switch( $handle ) {
            case 'order':
                $order = new Order( (int) $id ); 
                if (!Validate::isLoadedObject($order)) {
	                return $this->modal("ERROR", "Not Found", "This order does not exist"); 
                }

                $order->delete();
                break;

            case 'message':
                $message = new Message( (int) $id ); 
                if (!Validate::isLoadedObject($message)) {
                    return $this->modal("ERROR", "Not Found", "This message does not exist"); 
                }

                $message->delete();
                break;

            case 'rating':
                $rating = new Rating( (int) $id );
                if (!Validate::isLoadedObject($rating)) {
	                return $this->modal("ERROR", "Not Found", "This rating does not exist"); 
                }
        
                $rating->delete();
                break;

            case 'file':
                $file = new File( (int) $id ); 
                if (!Validate::isLoadedObject($file)) {
	                return $this->modal("ERROR", "Not Found", "This file does not exist"); 
                }
        
                $file->delete();

                break;

            case 'coupon':
                $coupon = new Coupon( (int) $id ); 
                if (!Validate::isLoadedObject($coupon)) {
	                return $this->modal("ERROR", "Not Found", "This coupon does not exist"); 
                }
        
                $coupon->delete();
                break;

            case 'customer':
                $customer = new Customer( (int) $id ); 
                if (!Validate::isLoadedObject($customer)) {
	                return $this->modal("ERROR", "Not Found", "This customer does not exist");  
                }
        
                $success = $customer->delete();
                if(!$success) {
	                return $this->modal("ERROR", "Not Found", "This customer does not exist");  
                }

                $sql = new DbQuery();
                $sql->select('o.`order_id`');
                $sql->from('order', 'o');
                $sql->where('o.`customer_id` = ' . (int) $id );
                $orders = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

                foreach( $orders as $order ) {
                    $order = new Order( (int) $order['order_id'] );
                    if(Validate::isLoadedObject($order) ) {
                        $order->delete();
                    }
                }

                break;

            case 'employee':
                $employee = new Employee( (int) $id ); 
                if (!Validate::isLoadedObject($employee)) {
	                return $this->modal("ERROR", "Not Found", "This employee does not exist");
                }
                
                if( $employee->id == Employee::START_ID) {
	                return $this->modal("ERROR", "Not Found", "You can't delete the super admin"); 
                }

                $employee->delete();

                break;

            case 'currency':
                $currency = new Currency( (int) $id ); 
                if (!Validate::isLoadedObject($currency)) {
	                return $this->modal("ERROR", "Not Found", "This currency does not exist"); 
                }
        
                $currency->delete();

                break;

            case 'site':
                $site = new Site( (int) $id ); 
                if (!Validate::isLoadedObject($site)) {
	                return $this->modal("ERROR", "Not Found", "This site does not exist"); 
                }
                
                Db::getInstance()->delete('configuration', 'site_id = ' . (int) $site->id );
                Db::getInstance()->delete('deadline', 'site_id = ' . (int) $id );

                $site->delete();

                break;

            case 'deadline':
                $deadline = new Deadline( (int) $id );
                if (!Validate::isLoadedObject($deadline)) {
	                return $this->modal("ERROR", "Not Found", "This deadline does not exist"); 
                }

                $deadline->delete();

                break;

            case 'paper_type':
                $paper_type = new PaperType( (int) $id );
                if (!Validate::isLoadedObject($paper_type)) {
                    return $this->modal("ERROR", "Not Found", "This paper type does not exist"); 
                }

                $paper_type->delete();
                break;

            case 'upload_type':
                $upload_type = new UploadType( (int) $id );
                if (!Validate::isLoadedObject($upload_type)) {
                    return $this->modal("ERROR", "Not Found", "This upload type does not exist"); 
                }

                $upload_type->delete();
                break;

            case 'academic_level':
                $academicLevel = new AcademicLevel( (int) $id );
                if (!Validate::isLoadedObject($academicLevel)) {
                    return $this->modal("ERROR", "Not Found", "This academic level does not exist"); 
                }

                $academicLevel->delete();

                Db::getInstance()->delete('deadline', 'academic_level_id = ' . (int) $id );

                break;

            case 'order':
                $order = new Order( (int) $id ); 
                if (!Validate::isLoadedObject($order)) {
                    return $this->modal("ERROR", "Not Found", "This order does not exist"); 
                }

                $order->delete();
                break;

            case 'descriptor':
                $descriptor = new Descriptor( (int) $id );
                if(!Validate::isLoadedObject($descriptor)) {
                    return $this->modal("ERROR", "Not Found", "This descriptor does not exist"); 
                }

                $descriptor->delete();
                break;

            case 'item_pricing':
                $item_pricing = new ItemPricing( (int) $id );
                if(!Validate::isLoadedObject($item_pricing)) {
                    return $this->modal("ERROR", "Not Found", "This Item Pricing does not exist"); 
                }

                $item_pricing->delete();
                break;

            case 'mail':
                Db::getInstance()->delete('mailing_list', 'mail_id = ' . (int) $id );
                break;
            
            case 'discipline':
                Db::getInstance()->delete('discipline', 'discipline_id = ' . (int) $id );
                break;

            case 'session':
                Db::getInstance()->delete('user_session', 'session_id = ' . (int) $id );
                break;

            case 'pd_schedule':
                Db::getInstance()->delete('order_pd_schedule', 'schedule_id = ' . (int) $id );
                break;

            case 'englishType':
                Db::getInstance()->delete('english_type', 'english_type_id = ' . (int) $id );
                break;

            case 'targetAudience':
                Db::getInstance()->delete('target_audience', 'target_audience_id = ' . (int) $id );
                break;

            case 'writingTone':
                Db::getInstance()->delete('writing_tone', 'writing_tone_id = ' . (int) $id );
                break;

            case 'writingStyle':
                Db::getInstance()->delete('writing_style', 'writing_style_id = ' . (int) $id );
                break;

            case 'contentFeel':
                Db::getInstance()->delete('content_feel', 'content_feel_id = ' . (int) $id );
                break;

            case 'mail_template':
                Db::getInstance()->delete('mail_template', 'mail_template_id = ' . (int) $id );
                break;

            case 'writing_guide':
                $writing_guide = new WritingGuide( (int) $id );
                if(Validate::isLoadedObject($writing_guide)) {
                    $writing_guide->delete();
                }
                break;

            case 'writing_prompt':
                $writing_prompt = new WritingPrompt( (int) $id );
                if(Validate::isLoadedObject($writing_prompt)) {
                    $writing_prompt->delete();
                }
                break;

            case 'essay':
                $essay = new ApplicationEssay( (int) $id );
                if(Validate::isLoadedObject($essay)) {
                    $essay->delete();
                }
                break;

            case 'software_programming':
                Db::getInstance()->delete('software_programming', 'software_programming_id = ' . (int) $id );
                break;

            case 'application-file':
                $file = new EmployeeFile( (int) $id );
                if(Validate::isLoadedObject($file)) {
                    $file->delete();
                }
                break;

            case 'writer-pool':
                $writer_pool = new EmployeePool( (int) $id );
                if(Validate::isLoadedObject($writer_pool)) {
                    $writer_pool->delete();
                }
                break;
    
        }

        Hook::exec('activitylog', [
            'object' => $handle,
            'object_id' => $id,
            'event' => 'delete_' . $handle
        ]);

        Hook::exec('actionAfterAdminDelete', [
            'handle' => $handle,
            'id' => $id,
        ]);
        
        return $this->modal("SUCCESS", "Deleted", "This item has been deleted successfuly"); 
    } 
}