<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2020 Proxim
 * @version    1.0
 * @since      File available since Release 1.0
*/

namespace Proxim\Presenter\Order;

use Proxim\Presenter\AbstractLazyArray;
use Doctrine\Common\Annotations\AnnotationException;
use Proxim\Application;
use Proxim\Order\Message as OrderMessage;
use Proxim\Order\Order;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class OrderMessageLazyArray extends AbstractLazyArray {
    /** @var Application */
    private $application;

    /** @var User */
    public $user;

    /** @var Order */
    private $order;

    /** @var OrderMessage */
    private $orderMessage;

    /** @var Array Departments */
    private $departments = array();

    /** 
     * OrderMessageLazyArray constructor.
     *
     * @throws AnnotationException
     * @throws ReflectionException
     */
    public function __construct(OrderMessage $orderMessage)
    {
		global $globals;

        $this->application = Application::getInstance();
        $this->user = $this->application->user;
        $this->order = new Order( (int) $orderMessage->order_id);
        $this->orderMessage = $orderMessage;
		$this->departments = ArrayUtils::get($globals, 'interlocutor_departments');

        parent::__construct();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getId()
    {
        return (int) $this->orderMessage->id;
    }
    
    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getOrderId()
    {
        return (int) $this->order->id;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getOrderReference() {
        return (string) $this->order->tracking_id ? $this->order->tracking_id : $this->order->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSenderId()
    {
        return (int) $this->orderMessage->sender_id;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsSender()
    {
        $user = $this->user;
        return $user->id == $this->orderMessage->sender_id ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getParentId()
    {
        return (int) $this->orderMessage->parent_id;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getFromDepartment()
    {
        $user = $this->user;
        $from_department = $this->orderMessage->from_department;
        $fromWide = ArrayUtils::get($this->departments, $from_department);

        if( $user->is_writer && $this->orderMessage->from_department == DEPARTMENT_WRITER ) {
			$fromWide = 'Me';
        }
        
        return array(
            'id' => (int) $from_department,
            'name' => $fromWide
        );
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getToDepartment()
    {
        $user = $this->user;
        $to_department = $this->orderMessage->to_department;
        $toWide = ArrayUtils::get($this->departments, $to_department);

		if( $user->is_writer && $this->orderMessage->to_department == DEPARTMENT_WRITER ) {
			$toWide = 'Me';
        }
        
        return array(
            'id' => (int) $to_department,
            'name' => $toWide
        );
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getSubject()
    {
		$body_clean = Tools::getStringClean($this->orderMessage->body);
        return $this->orderMessage->subject ? $this->orderMessage->subject : Tools::truncateString( $body_clean );
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getBody()
    {
        $orderMessage = $this->orderMessage->body;

        if(
            $this->orderMessage->from_department == DEPARTMENT_CUSTOMER && 
            (!$this->user->is_admin && !$this->user->is_sub_admin)
        ) {
            $orderMessage = Tools::obfuscate_email($orderMessage);
        }

        // $orderMessage = Tools::hyperlinkText($orderMessage);

        return Tools::nl2br($orderMessage);
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsDeleted()
    { 
        return (bool) $this->orderMessage->is_deleted;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsViewed()
    { 
        return (bool) $this->orderMessage->is_viewed;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsVerified()
    { 
        return (bool) $this->orderMessage->is_verified;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getReadAt()
    {
        $user = $this->user;
        return "Read at " . DateUtils::convertUtcDateTimeToTimeZone($this->orderMessage->date_add, $user->timezone);
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCreatedAt()
    {
        $user = $this->user;
        return DateUtils::convertUtcDateTimeToTimeZone($this->orderMessage->date_add, $user->timezone);
    }
}