<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2020 Proxim
 * @version    1.0
 * @since      File available since Release 1.0
*/

namespace Proxim\Presenter\Order;

use Db;
use Proxim\Presenter\AbstractLazyArray;
use Doctrine\Common\Annotations\AnnotationException;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Coupon;
use Proxim\Currency;
use Proxim\Database\DbQuery;
use Proxim\Order\File as OrderFile;
use Proxim\Order\Order;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Site\Site;
use Proxim\Tools;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class CalculationsOrderLazyArray extends AbstractLazyArray {
    /** @var Application */
    public $application;

    /** @var User */
    public $user;

    /** @var Order */
    public $order;

    /** @var Currency */
    public $currency;

    /** @var Currency */
    public $writer_currency;

    /** @var ObjectPresenter */
    public $objectPresenter;

    /** @var MessagePresenter */
    public $messagePresenter;

    /** @var array $orderFiles */
    public $orderFiles = array();

    /** @var array $customerFiles */
    public $customerFiles = array();

    /** @var array $writerFiles */
    public $writerFiles = array();

    /** @var array $pdSchedule */
    public $pdSchedule = array();

    /** @var array $customerFilesCount */
    public $customerFilesCount = 0;

    /** @var array $writerFilesCount */
    public $writerFilesCount = 0;

    /** @var array $orderFilesCount */
    public $orderFilesCount = 0;

    /** @var array $unseenFilesCount */
    public $unseenFilesCount = 0;

    /** @var array $allMessagesCount */
    public $allMessagesCount = 0;

    /** @var array $countUnreadMessages */
    public $countUnreadMessages = 0;

    /** @var bool $latestFileIsDownloaded */
    public $latestFileIsDownloaded = false;

    /**
     * OrderArray constructor.
     *
     * @throws AnnotationException
     * @throws ReflectionException
     */
    public function __construct(Order $order)
    {
        $this->application = Application::getInstance();
        $this->objectPresenter = new ObjectPresenter();
        $this->messagePresenter = new OrderMessagePresenter();
        $this->user = $this->application->user;
        $this->order = $order;

        $currency = new Currency( (int) $order->currency_id );
        if(!Validate::isLoadedObject($currency)) {
            $default_currency_id = (int) Configuration::get('CURRENCY_DEFAULT');
            $currency = new Currency( $default_currency_id );
        }

        $this->currency = $currency;

        /** Order Messages */
        $orderMessages = $order->getMessages();

        $countUnreadMessages = 0;
        foreach( $orderMessages as $message ) {
            if ( $message->sender_id != $order->customer_id && !$message->is_viewed ) {
                $countUnreadMessages++;
            }
        }

        $this->countUnreadMessages = $countUnreadMessages;
        $this->allMessagesCount = count($orderMessages);

        $customerFilesCount = $writerFilesCount = $unseenFilesCount = 0;
        $writerFiles = $customerFiles = array();

        $filePresenter = new OrderFilePresenter();

        $orderFiles = $order->getFiles();
        foreach ($orderFiles as $orderFile) {
            $orderFile = new OrderFile( (int) $orderFile['file_id']); 
            if( Validate::isLoadedObject($orderFile) ) {
                if ( $orderFile->uploader_id == $this->order->customer_id ) {
                    $customerFiles[] = $filePresenter->present($orderFile);
                    $customerFilesCount++;
                } else {
                    $writerFiles[] = $filePresenter->present($orderFile);
                    $writerFilesCount++;

                    if( !$orderFile->downloaded && $this->order->status_id == Order::DELIVERED || $this->order->status_id == Order::FINISHED ) {
                        $unseenFilesCount++;
                    }
                }
            } 
        }

        $this->orderFilesCount = $customerFilesCount+$writerFilesCount;
        $this->customerFilesCount = $customerFilesCount;
        $this->writerFilesCount = $writerFilesCount;
        $this->unseenFilesCount = $unseenFilesCount;

        $this->customerFiles = $customerFiles;
        $this->writerFiles = $writerFiles;

        $this->orderFiles = $orderFiles;

        if ( $writerFilesCount > 0 ) {
            $lastFile = ArrayUtils::shift($writerFiles);
            if( Validate::isLoadedObject($lastFile) ) {
                $this->latestFileIsDownloaded = (bool) $lastFile->downloaded;
            }
        }

        parent::__construct();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getId()
    {
        return (int) $this->order->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getParentId()
    {
        return (int) $this->order->parent_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getTrackingId()
    {
        return (string) $this->order->tracking_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getReferenceId()
    {
        return (string) $this->order->tracking_id ? $this->order->tracking_id : $this->order->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getUkForm()
    {
        return (bool) Configuration::get('UK_FORM', $this->order->site_id, false);
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCustomerId()
    {
        return (int) $this->order->customer_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCustomer()
    {
        $customer = $this->order->getCustomer();
        return $this->objectPresenter->present($customer);
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCustomerCountry()
    {
        $customer = $this->order->getCustomer();
        return $customer->country;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getWriterId()
    {
        return (int) $this->order->writer_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getWriter()
    {
        $writer = $this->order->getWriter();
        return (array) $writer;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getEditorId()
    {
        return (int) $this->order->editor_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getEditor()
    {
        $editor = $this->order->getEditor();
        return $this->objectPresenter->present($editor);
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getOrderManagerId()
    {
        return (int) $this->order->ordermanager_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getOrderManager()
    {
        $orderManager = $this->order->getOrderManager();
        return $this->objectPresenter->present($orderManager);
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getRequestedWriterId()
    {
        return (int) $this->order->requested_writer_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getRequestedWriter()
    {
        $requestedWriter = new Employee( (int) $this->order->requested_writer_id );
        return $this->objectPresenter->present($requestedWriter);
    }
    
    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSiteId()
    {
        return (int) $this->order->site_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSite()
    {
        $orderSite = new Site( (int) $this->order->site_id );
        return $this->objectPresenter->present($orderSite);
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCurrencyId()
    {
        return (int) $this->currency->id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCurrencyRate()
    {
        return (float) $this->currency->conversion_rate;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getServiceTypeId()
    {
        return (int) $this->order->service_type;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getServiceTypeTitle()
    {
        global $globals;

        $serviceType = ArrayUtils::get($globals['serviceTypes'], (int) $this->order->service_type);
        return (string) ArrayUtils::get($serviceType, 'title');
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsProgramming() {
        return $this->order->service_type == Order::PROGRAMMING ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsCalculations() {
        return $this->order->service_type == Order::CALCULATIONS ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsAcademicWriting() {
        return $this->order->service_type == Order::COMPLEX_ASSIGNMENT ? true : false;
    }
    
    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsArticleWriting() {
        return $this->order->service_type == Order::ARTICLE_WRITING ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getStatus()
    {
        return (int) $this->order->status_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getStatusId()
    {
        return (int) $this->order->status_id;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getStatusTitle()
    {
        global $globals;
        $status = ArrayUtils::get($globals['orderStatuses'], (int) $this->order->status_id);
        return (string) ArrayUtils::get($status, 'title');
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getStatusDescription()
    {
        global $globals;
        $status = ArrayUtils::get($globals['orderStatuses'], (int) $this->order->status_id);
        return (string) ArrayUtils::get($status, 'description');
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getStatusBadge()
    {
        switch($this->order->status_id) {
            case Order::FAKE:
            case Order::FREE_INQUIRY:
            case Order::CONFIRMATION_PENDING:
                return 'warning';
                break;

            case Order::ORDER_PUBLISHED:
            case Order::INQUERY_PUBLISHED:
            case Order::NEW_PAID:
                return 'primary';
                break;

            case Order::WRITER_ACCEPTED:
            case Order::WRITER_ASSIGNED:
                return 'info';
                break;

            case Order::DONE:
                return 'secondary';
                break;

            case Order::DELIVERED:
            case Order::FINISHED:
                return 'success';
                break;

            case Order::CANCELLED:
            case Order::DISPUTE:
            case Order::REVISION:
                return 'danger';
                break;
            
            default:
                return 'gray';
                break;
        }
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getTopicCategoryId() {
        return (int) $this->order->topic_category_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getTopicCategory() { 
        $discipline = new Discipline( (int) $this->order->topic_category_id );
        return (string) $discipline->title;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getTopicCategoryOption() {
        return $this->order->topic_category_option;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSizeId() {
        return (int) $this->order->task_size_id;
    }
    
    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getSizeTitle() {
        global $globals;

        $taskSize = ArrayUtils::get($globals['taskSizes'], $this->order->task_size_id, []);
        return (string) ArrayUtils::get($taskSize, 'title');
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDescriptor() {
 
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDeliverables() {
        $query = new DbQuery();
        $query->select('d.deliverable_id, d.name, d.slug, des.example');
        $query->leftJoin('descriptor_deliverable', 'dd', 'dd.descriptor_id = des.descriptor_id');
        $query->leftJoin('deliverable', 'd', 'd.deliverable_id = dd.deliverable_id');
        $query->from('descriptor', 'des');
        $query->where('des.discipline_id = ' . (int) $this->order->topic_category_id );
        $query->where('des.task_size_id = ' . (int) $this->order->task_size_id );
        
        if($this->order->service_type == Order::PROGRAMMING) {
            $query->where('des.is_programming = 1' );
        } else {
            $query->where('des.is_calculations = 1' );
        }

        $deliverables = Db::getInstance()->executeS($query);

        return $deliverables;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPaperDetails() {
        $paperDetails = $this->order->paper_details;
        if(!$this->user->is_admin && !$this->user->is_sub_admin) {
            $paperDetails = Tools::obfuscate_email($paperDetails);
        }

        return (string) Tools::nl2br($paperDetails);
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPaperDetailsRaw() {
        return (string) $this->order->paper_details;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getSoftware() {
        return (string) $this->order->software;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getComments() {
        return (string) $this->order->comments;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getPaymentMethod() {
        return (string) $this->order->payment_method;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getTariffId() {
        return (int) $this->order->tariff_id;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getTariffHrs() {
        return (int) $this->order->hrs_customer;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getTariffPricePerPage() {
        return (float) $this->order->price_per_page;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPrice() {
        return (float) $this->order->total;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getPriceFormatted() {
        return formatPrice($this->order->total);
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getWriterPay() {
        return $this->order->writer_pay;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getWriterPayFormatted() {
        $writerPay = $this->getWriterPay();

        return formatPrice($writerPay, $this->application->writer_currency);
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsWriterPaid() {
        return (bool) $this->order->is_writer_paid;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getOrderServices(){
        $servicesById = $this->order->calculate();
        return $servicesById;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsWriterAssigned() {
        return (bool) $this->order->writer_assigned;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getWriterAssignedAt() {
        return $this->order->writer_assigned_at;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsWriterConfirmed() {
        return (bool) $this->order->is_writer_confirmed;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getEditorPay() {
        $user = $this->user;
        $editorPay = $this->order->editor_pay;

        if( $user->is_editor && 
            $this->order->editor_id != $user->id
        ) {
            if( $user->cpp_type == "custom" && $user->cpp_amount > 0 ) {
                Configuration::set('EDITOR_CPP_TYPE', "fixed", PROX_SITE_ID);
                Configuration::set('EDITOR_AMOUNT_CPP', $user->cpp_amount , PROX_SITE_ID);

                $calc = $this->order->calculate();
                $editorPay = (float) ArrayUtils::get($calc, 'editorCost');
            }
        }

        return $editorPay;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getEditorPayFormatted() {
        $editorPay = $this->getEditorPay();
        
        return formatPrice($editorPay, $this->application->writer_currency);
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsEditorPaid() {
        return (bool) $this->order->is_editor_paid;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsEditorAssigned() {
        return (bool) $this->order->editor_assigned;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getEditorAssignedAt() {
        return $this->order->editor_assigned_at;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsEditorConfirmed() {
        return (bool) $this->order->is_editor_confirmed;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getOrderManagerPay() {
        return (float) $this->order->ordermanager_pay;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getOrderManagerPayFormatted() {
        return formatPrice($this->order->ordermanager_pay, $this->application->writer_currency);
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsOrderManagerPaid() {
        return (bool) $this->order->is_ordermanager_paid;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getOrderManagerAssigned() {
        return (bool) $this->order->ordermanager_assigned;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getOrderManagerAssignedAt() {
        return $this->order->ordermanager_assigned_at;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getTax() {
        return (float) $this->order->tax;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getNoDiscountPrice() {
        return (float) $this->order->raw_cost;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getNoDiscountPriceFormatted() {
        return formatPrice($this->order->raw_cost);
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getCoupon() {
        $coupon = new Coupon( (int) $this->order->coupon_id );
        return (array) $coupon;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCouponType() {
        return (int) $this->order->discount_type;
    }

    /**
     * @arrayAccess
     *
     * @return float
     */
    public function getCouponValue() {
        return (float) $this->order->discount;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsPaid() {
        return (bool) $this->order->is_paid;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getProgressiveDeliveryHidden() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getAreSamplesRequired() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getAreSourcesRequired() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getExpertProofreading() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getVipSupport() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getPlagiarismReport() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getDraftOutline() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getServices() {
        $additionalServices = $this->order->getAdditionalServices();

        return $additionalServices;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getOrderReviews() {
        $orderReviews = $this->order->getOrderReviews();

        return $orderReviews;
    }

    /**
     * @arrayAccess
     *
     * @return array
     */
    public function getAdditionalPayments() {
        return $this->order->getAdditionalPayments();
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getAllFilesCount() {
        return (int) $this->orderFilesCount;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getNewFilesCount() {
        return (int) $this->unseenFilesCount;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getWithAdditionalMaterials() {
        return count($this->customerFiles) > 0 ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getLatestFileIsDownloaded() {
        return (bool) $this->latestFileIsDownloaded;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getAllMessagesCount() {
        return (int) $this->allMessagesCount;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getNewMessagesCount() {
        return (int) $this->countUnreadMessages;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getMessages() {
        $orderMessages = array();

        $messages = $this->order->getMessages();
        foreach( $messages as $orderMessage ) {
            $orderMessages[] = $this->messagePresenter->present($orderMessage);
        }

        return $orderMessages;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getCustomerFiles() {
        return $this->customerFiles;
    }

    /**
     * @arrayAccess
     *
     * @return int
     */
    public function getWriterFiles() {
        return $this->writerFiles;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getHasPDSchedule() {
        return count($this->pdSchedule) ? true : false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getPdSchedule() {
        return $this->pdSchedule;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsPdScheduleAccepted() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsPdScheduleRejected() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getShowScheduleApproveRejectButton() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsOnTimeQuestionNeeded() {
        return true;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getHasReassignments() {
        return false;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getCanSubmitOrder() {
        $canSubmitOrder = false;

        if( $this->writerFilesCount > 0 ) {
            $canSubmitOrder = true;
        }

        return (bool) $canSubmitOrder;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCompletedAt() {
        if( $this->order->completed_at ) {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->completed_at, $this->user->timezone);
        }

        return;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getFinishedAt() {
        if( $this->order->delivered_at ) {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->delivered_at, $this->user->timezone);
        }

        return;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCancelledAt() {
        if( $this->order->status_id == Order::CANCELLED ) {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->cancelled_at, $this->user->timezone);
        }

        return;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getApprovedAt() {
        if( $this->order->status_id == Order::FINISHED ) {
        return DateUtils::convertUtcDateTimeToTimeZone($this->order->approved_at, $this->user->timezone);
        }

        return;
    }

    /**
     * @arrayAccess
     *
     * @return bool
     */
    public function getIsNewCustomer() {
        $customer = $this->order->getCustomer();
        return $customer->total_orders > 2 ? false : true;
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDeadlineHighlight() {
        $user = $this->user;

        if($user->is_writer && Validate::isDate($this->order->writer_deadline)) {
            $timeLeft = DateUtils::timeLeft($this->order->writer_deadline, true);
        } else {
            $timeLeft = DateUtils::timeLeft($this->order->customer_deadline, true);
        }

        if($timeLeft['passed']) {
            return 'danger';
        } else {
            $hours = ($timeLeft['days']*24)+$timeLeft['hours'];

            if($hours <= 4) {
                return 'danger';
            } elseif( $hours <= 10 ) {
                return 'warning';
            } 
        }

        return 'success';
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getRealDeadline() {
        return DateUtils::convertUtcDateTimeToTimeZone($this->order->customer_deadline, $this->user->timezone);
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getWriterDeadline() {
        if(Validate::isDate($this->order->writer_deadline)) {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->writer_deadline, $this->user->timezone);
        } else {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->customer_deadline, $this->user->timezone);
        }
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getDeadline() {
        $user = $this->user;
        if($user->is_writer && Validate::isDate($this->order->writer_deadline)) {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->writer_deadline, $this->user->timezone);
        } else {
            return DateUtils::convertUtcDateTimeToTimeZone($this->order->customer_deadline, $this->user->timezone);
        }
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getTimeLeft() {
        $deadline = $this->getDeadline();
        return $deadline ? DateUtils::timeLeft( $deadline ) : 'N/A';
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getTimeLeftHtml() {
        $deadline = $this->getDeadline();
        return $deadline ? DateUtils::timeLeft( $deadline, false, true ) : 'N/A';
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getWriterTimeLeftHtml() {
        $deadline = $this->getWriterDeadline();
        return $deadline ? DateUtils::timeLeft( $deadline, false, true ) : 'N/A';
    }

    /**
     * @arrayAccess
     *
     * @return string
     */
    public function getCreatedAt() {
        $user = $this->user;
        return DateUtils::convertUtcDateTimeToTimeZone($this->order->date_add, $user->timezone);
    }
}