<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\PDF;

use Proxim\Application;
use Proxim\Order\Order;
use Proxim\Order\PaymentRequest;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Site\Site;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

/**
 * @since 1.5
 */
class HTMLTemplatePaymentRequestInvoice extends HTMLTemplate
{
    public $payment_request;

    /**
     * @param PaymentRequest $payment_request
     * @param $smarty
     *
     * @throws Exception
     */
    public function __construct(PaymentRequest $payment_request, $smarty)
    {
        $this->payment_request = $payment_request;
        $this->smarty = $smarty;
       
        $this->title = sprintf('INV %s', $this->payment_request->id);

        $site = Application::getInstance()->site;
        $this->site = new Site((int) $site->id );
    }

    /**
     * Returns the template's HTML content.
     *
     * @return string HTML content
     */
    public function getContent()
    {
        $employee = new Employee( (int) $this->payment_request->user_id );

        $payment_request = $this->payment_request->getPaymentRequestInfo();

        $ordersPresented = array();

        if(ArrayUtils::has($payment_request, 'orders')) {
            $orders = ArrayUtils::get($payment_request, 'orders');
            foreach($orders as $order_id) {
                $order = new Order( (int) $order_id);
                if(Validate::isLoadedObject($order)) {
                    $ordersPresented[] = $order->present(false);
                }
            }
        }

        $payment_request['ordersPresented'] = $ordersPresented;

        $this->smarty->assign([
            'employee' => $employee,
            'payment_request' => $payment_request,
            'style_tab' => $this->smarty->fetch($this->getTemplate('invoice.style-tab'))
        ]);

        return $this->smarty->fetch($this->getTemplate('payment-request'));
    }

    /**
     * Returns the template filename.
     *
     * @return string filename
     */
    public function getFilename()
    {
        return sprintf('INV%06d', $this->payment_request->id) . '.pdf';
    }

    /**
     * Returns the template filename when using bulk rendering.
     *
     * @return string filename
     */
    public function getBulkFilename()
    {
        return 'invoices.pdf';
    }

    /**
     * Returns the template's HTML header.
     *
     * @return string HTML header
     */
    public function getHeader()
    {
        $this->assignCommonHeaderData();
        $this->smarty->assign(['header' => 'Payment Request']);
        return $this->smarty->fetch($this->getTemplate('header'));
    }
}