<?php
/**
 * @package    Proxim 
 * @author     Davison Pro <davis@davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Order;

use Db;
use Proxim\Database\DbQuery;
use Proxim\ObjectModel;
use Proxim\User\Employee;
use Proxim\Validate;

/**
 * PaymentRequest
 */
class PaymentRequest extends ObjectModel
{
	/** @var $id PaymentRequest ID */
	public $id;

	/** @var int user_id */
    public $user_id;
    
    /** @var float total_done_pay */
    public $total_done_pay = 0;

    /** @var float total_completed_pay */
    public $total_completed_pay = 0;

    /** @var float total_pay */
    public $total_pay = 0;

    /** @var float total_credit */
    public $total_credit = 0;

    /** @var float total_fine */
    public $total_fine = 0;

    /** @var bool status */
    public $status = 0;

    /** @var string paid_at */
    public $paid_at;

	/** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
	public $date_add;

	/**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'payment_request',
        'primary' => 'payment_request_id',
        'fields' => array(
            'user_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt', 'required' => true),
            'total_done_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_completed_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_credit' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_fine' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'status' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'paid_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
			'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
		)
	);

	/**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null)
    {
        parent::__construct($id);
	}

    public function getPaymentRequestInfo() {
        $return = array();

        $employee = new Employee( (int) $this->user_id );
        if(!Validate::isLoadedObject($employee)) {
	        return $return;
        }

        $sql = new DbQuery();
        $sql->select('pro.order_id');
        $sql->from('payment_request_order', 'pro');
        $sql->where('pro.`payment_request_id` = ' . (int) $this->id );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        $subtotal_pay = $total_pay = $completed_orders_pay = $done_orders_pay = 0;
 
        $orders = array();
        foreach( $result as $order ) {
            $order = new Order( (int) $order['order_id'] );
            if( Validate::isLoadedObject($order) ) {
                if($order->status_id == Order::FINISHED) {
                    if($employee->is_sub_admin) {
                        $completed_orders_pay += $order->ordermanager_pay;
                    } elseif($employee->is_editor) {
                        $completed_orders_pay += $order->editor_pay;
                    } elseif($employee->is_writer) {
                        $completed_orders_pay += $order->writer_pay;
                    }
                } elseif ( $order->status_id == Order::DELIVERED ) {
                    if($employee->is_sub_admin) {
                        $done_orders_pay += $order->ordermanager_pay;
                    } elseif($employee->is_editor) {
                        $done_orders_pay += $order->editor_pay;
                    } elseif($employee->is_writer) {
                        $done_orders_pay += $order->writer_pay;
                    }
                }

                $orders[] = $order->id;
            }
        }
        
        $subtotal_pay = $completed_orders_pay+$done_orders_pay;
        $total_pay = ($subtotal_pay+$this->total_credit)-$this->total_fine;

        $return = [
            'total_credit' => (float) $this->total_credit,
            'total_fine' => (float) $this->total_fine,
            'done_orders_pay' => (float) $done_orders_pay,
            'completed_orders_pay' => (float) $completed_orders_pay,
            'subtotal_pay' => (float) $subtotal_pay,
            'total_pay' => (float) $total_pay,
            'orders' => $orders
        ];

        return $return;
    }
}