<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Order;

use Db;
use Proxim\Hook;
use Proxim\ObjectModel;
use Proxim\Tools;
use Proxim\Validate;

class OrderHistory extends ObjectModel
{
    /** @var int Order id */
    public $id_order;

    /** @var int Order status id */
    public $status_id;

    /** @var int Employee id for this history entry */
    public $employee_id;

    /** @var string Object creation date */
    public $date_add;

    /** @var string Object last modification date */
    public $date_upd;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = [
        'table' => 'order_history',
        'primary' => 'order_history_id',
        'fields' => [
            'order_id' => ['type' => self::TYPE_INT, 'validate' => 'isUnsignedId', 'required' => true],
            'status_id' => ['type' => self::TYPE_INT, 'validate' => 'isUnsignedId', 'required' => true],
            'employee_id' => ['type' => self::TYPE_INT, 'validate' => 'isUnsignedId'],
            'date_add' => ['type' => self::TYPE_DATE, 'validate' => 'isDate'],
        ],
    ];

    /**
     * Sets the new state of the given order.
     *
     * @param int $new_order_state
     * @param int/object $id_order
     * @param bool $use_existing_payment
     */
    public function changeIdOrderState($new_order_state, $id_order, $use_existing_payment = false)
    {
        if (!$new_order_state || !$id_order) {
            return;
        }

        if (!is_object($id_order) && is_numeric($id_order)) {
            $order = new Order((int) $id_order);
        } elseif (is_object($id_order)) {
            $order = $id_order;
        } else {
            return;
        }

        $new_os = new OrderState((int) $new_order_state);
        $old_os = $order->getCurrentOrderState();

        // executes hook
        Hook::exec('actionOrderStatusUpdate', ['newOrderStatus' => $new_os, 'id_order' => (int) $order->id]);

        if (Validate::isLoadedObject($order) && ($new_os instanceof OrderState)) {

        }

        $this->status_id = (int) $new_order_state;

        // changes invoice number of order ?
        if (!Validate::isLoadedObject($new_os) || !Validate::isLoadedObject($order)) {
            die(Tools::displayError('Invalid new order status'));
        }

        // the order is valid if and only if the invoice is available and the order is not cancelled
        $order->current_state = $this->status_id;
        $order->valid = $new_os->logable;
        $order->update();

        // executes hook
        Hook::exec('actionOrderStatusPostUpdate', ['newOrderStatus' => $new_os, 'id_order' => (int) $order->id]);
    }

    /**
     * Returns the last order status.
     *
     * @param int $id_order
     *
     * @return OrderState|bool
     *
     * @deprecated 1.5.0.4
     * @see Order->current_state
     */
    public static function getLastOrderState($id_order)
    {
        $status_id = Db::getInstance()->getValue('
            SELECT `status_id`
            FROM '. Db::prefix('order_history') .'
            WHERE `order_id` = ' . (int) $id_order . '
            ORDER BY `date_add` DESC, `order_history_id` DESC'
        );

        // returns false if there is no state
        if (!$status_id) {
            return false;
        }

        // else, returns an OrderState object
        return new OrderState($status_id);
    }

    /**
     * @param bool $autodate Optional
     * @param array $template_vars Optional
     *
     * @return bool
     */
    public function addWithemail($autodate = true, $template_vars = false)
    {
        $order = new Order($this->id_order);

        if (!$this->add($autodate)) {
            return false;
        }

        Order::cleanHistoryCache();

        if (!$this->sendEmail($order, $template_vars)) {
            return false;
        }

        return true;
    }

    /**
     * @param Order $order
     * @param array|false $template_vars
     *
     * @return bool
     */
    public function sendEmail($order, $template_vars = false)
    {
        
    }

    public function add($autodate = true, $null_values = false)
    {
        if (!parent::add($autodate)) {
            return false;
        }

        $order = new Order((int) $this->id_order);
        // Update status_id attribute in Order
        $order->current_state = $this->status_id;
        $order->update();

        Hook::exec('actionOrderHistoryAddAfter', ['order_history' => $this], null, false, true, false, $order->id_shop);

        return true;
    }
}
