<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Order;

use Db; 
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\Application;
use Proxim\ObjectModel;
use Proxim\Order\File;
use Proxim\PriceCalculator;
use Proxim\Configuration;
use Proxim\Coupon;
use Proxim\Currency;
use Proxim\Hook;
use Proxim\Mail;
use Proxim\MailTemplate;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Presenter\Order\OrderFilePresenter;
use Proxim\Presenter\Order\OrderMessagePresenter;
use Proxim\Presenter\Order\OrderPresenter;
use Proxim\Site\Site;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;
use Symfony\Component\DependencyInjection\Variable;

/**
 * Order
 */
class Order extends ObjectModel
{
    const START_ORDER_ID = 11233;

    /**
	 * Dispute - is the status of the order when customer is not satisfied
	 * with the quality of the paper, or this order was delivered late and
	 * customer wants to receive a refund.
	 *
	 * We have a dispute manager who deals with all the disputed orders,
	 * however Support Team Members should know basic information about it.
	 */
	const DISPUTE = 1; 
	/**
	 * When customer tries to pay for the order, but his payment doesn't
	 * go through, we receive "Fake" order.
	 *
	 * Basically, this status means that order has 96% possibility to become
	 * a sale, as customer had filled out the whole form and clicked on "Pay
	 * for the order")
	 *
	 * When you receive such order, it is your responsibility to contact
	 * customer and to help him to complete the payment.
	 */
	const FAKE = 0;
	/**
	 * Confirmation Pending - inquiry with instructions for the writer to
	 * start working on it, we just need customer to proceed with the payment.
	 */
	const CONFIRMATION_PENDING = 2;
	
	/**
	 * Writer Assigned - is a paid order that was assigned to a specific writer
	 */
	const WRITER_ASSIGNED = 6;

	/**
	 * Writer Accepted - is a paid order that was already published, and few writers applied for it.
	 * Such orders have a list of writers who are willing to work on them.
	 * At this point supporter has to assign this order to one of the writers
	 */
	const WRITER_ACCEPTED = 7;
	
	/**
	 * Done – status of the order when we receive completed paper from the writer.
	 * At this point customer doesn't have access to this order yet.
	 * Supporter needs to check this paper for plagiarism, paper format, etc.
	 *
	 * In case if there is plagiarism or this order was written in a wrong format,
	 * it should be set on "Revision" status.
	 *
	 * If the order is completed correctly, then the preview version should be sent to customer.
	 */
	const DONE = 8;
	/**
	 * Order can move to "Canceled" status at any stage.
	 * But there are several main reasons to "cancel" the order:
	 *
	 * Customer doesn't want to pay for his inquiry
	 * Customer accidentally placed 2 or more identical inquiries or orders.
	 * Writer wasn't assigned and we issued customer a full refund
	 *  
	 */
	const CANCELLED = 9;
	/**
	 * Order moves to Finished  status when customer approves the order
	 * and receives MS Word version of the paper.
	 *
	 * By approving the order, customer confirms that product is of expected quality.
	 */
	const FINISHED = 10;
	/**
	 * This status means that the Preview version of the order is available for customer
	 * and he/she can review it in .pdf format.
	 *
	 * Preview version is protected with a password,
	 * so technically customer cannot print or copy/paste it.
	 */
	const DELIVERED = 11;
	/**
	 * New Paid - is a status of a new order that we receive from
	 * customer. This is a stage when order is paid.

	* At this point order is not visible for writers.

	* You have to check such order and make sure that there is no customer's
	* personal information and that we have all required instructions.
	*/
	const NEW_PAID = 12;
	/**
	 *  Free Inquiry -  is a transitional status. When we receive it, we
	 *  have to check instructions and call customer to turn this inquiry into
	 *  a paid order (thus status will be changed to "New Paid").
	 *
	 *  In case if we can not contact customer, or he/she refuses to
	 *  complete the payment, status will be changed as well ("Reply Pending",
	 *  "Confirmation Pending", "Canceled")
	 */
	const FREE_INQUIRY = 13;
	/**
	 * Revision – Order is in a process of editing and revising.
	 *
	 * It is necessary when customer is not satisfied with the quality of a product.
	 *
	 * Also order can move to this status when plagiarism or
	 * format mistakes were found by Support Team Member.
	 */
	const REVISION = 14;
	/**
	 * Reply Pending - inquiry with the lack of instructions for us to look
	 * for the writer, thus we need to ask customer to specify/add instructions
	 * for the assignment.
	 *
	 * When we will receive such information customer can proceed with
	 * the payment.
	 */
	const REPLY_PENDING = 15;
	/**
	 * This status visually looks like "Order Published," but it is NOT
	 * PAID. Orders with such status are visible for writers.
	 *
	 * This status should be applied in following cases:
	 *
	 * - big orders over $300
	 * - difficult topic: programming for ex.
	 * - specific sources are required
	 */
	const INQUERY_PUBLISHED = 16;
	/**
	 * When writers apply for orders in "Inquiry Published" status,
	 * they move to "Inquiry Writer Accepted" status.
	 *
	 * Keep in mind that it's not paid yet, so we cannot assign this order
	 * to the writer until we will receive the payment.
	 */
	const INQUERY_WRITER_ACCEPTED = 17;
	/**
	 * Order published - is a paid order, which was checked by the supporter
	 * on personal information, deadline, attachments.
	 *
	 * At this point order is visible for all writers and they can place their bids.
	 */
	const ORDER_PUBLISHED = 19;
	const CONVERT_FAILED = 20;
	const CALL_PENDING = 21;

	// aliases
	const INQUIRY = self::FREE_INQUIRY;
	const LACK_OF_INSTRUCTIONS = self::REPLY_PENDING;

	const SPACING_SINGLE = 'single';
	const SPACING_DOUBLE = 'double';
	
	/** Services **/
	const WRITING_PAGES = 1;
	const WRITING_SLIDES = 2;
	const WRITING_CHARTS = 31;
	const WRITING_EXCEL_SHEETS = 32;
	const CHOOSE_WRITER = 3;
	const PROVIDE_ME_SAMPLES = 4;
	const PROGRESSIVE_DELIVERY = 5;
	const DISCOUNT = 6;
	const USED_SOURCES = 21;
	const COMPLEX_ASSIGNMENT = 37;
	const COUPON_WRT_PAGES = 10;
	const REVISION_MINOR_PAGES = 26;
	const REVISION_MINOR_SLIDES = 27;
	const REVISION_MINOR_CHARTS = 33;
	const REVISION_MINOR_EXCEL_SHEETS = 34;
	const REVISION_MAJOR_PAGES = 28;
	const REVISION_MAJOR_SLIDES = 29;
	const REVISION_MAJOR_EXCEL_SHEETS = 30;
	const REVISION_MAJOR_CHARTS = 35;
    const EXPERT_PROOFREADING = 41;
    const VIP_SUPPORT = 44;
    const PLAGIARISM_REPORT = 45;
    const DRAFT_OUTLINE = 46;
	const SHORTEN_DEADLINE = 42;

	const REVISION_MINOR = 42;
    const REVISION_MAJOR = 43;

    const RESUME = 49;
    const CALCULATIONS = 50;
    const PROGRAMMING = 51;
    const ARTICLE_WRITING = 52;
	const WRITING_WORDS = 53;

    const ORDER_TYPE_ACADEMIC = "academic";
    const ORDER_TYPE_ARTICLE = "article";
    
    /** @var $id Order ID */
	public $id;
    
    /** @var string $tracking_id Tracking ID */
    public $tracking_id;
    
    /** @var int site_id */
    public $site_id = PROX_SITE_ID;

    /** @var int form_type_id */
    public $form_type_id;

    /** @var string form_type */
    public $form_type;

    /** @var string payment_method */
    public $payment_method;

    /** @var int currency_id */
    public $currency_id;

    /** @var float currency_to_usd_rate */
    public $currency_to_usd_rate;

    /** @var int gmt */
    public $gmt;

    /** @var int coupon_id */
    public $coupon_id;

    /** @var int customer_id */
    public $customer_id;

    /** @var int writer_id */
    public $writer_id;

    /** @var int current_writer_id */
    public $current_writer_id;

    /** @var int requested_writer_id */
    public $requested_writer_id;

    /** @var int editor_id */
    public $editor_id;

    /** @var int status_id */
    public $status_id = self::FAKE;

    /** @var int parent_id */
    public $parent_id;

    /** @var int service_type */
    public $service_type = self::COMPLEX_ASSIGNMENT;

    /** @var int tariff_id */
    public $tariff_id;

    /** @var string title */
    public $title;

    /** @var int academic_level_id */
    public $academic_level_id;

    /** @var int paper_type_id */
    public $paper_type_id;

    /** @var string paper_type_option */
    public $paper_type_option;

    /** @var int topic_category_id */
    public $topic_category_id;

    /** @var string topic_category_option */
    public $topic_category_option;

    /** @var bool is_complex_assignment */
    public $is_complex_assignment = 0;

    /** @var int paper_format_id */
    public $paper_format_id;

    /** @var string paper_format_option */
    public $paper_format_option;

    /** @var int writer_category_id */
    public $writer_category_id;

    /** @var int writer_percent */
    public $writer_percent = 0;

    /** @var int task_size_id */
    public $task_size_id = 0;

    /** @var int pages */
    public $pages = 0;

    /** @var int charts */
    public $charts = 0;

    /** @var int slides */
    public $slides = 0;

    /** @var int excel_sheets */
    public $excel_sheets = 0;

    /** @var int words */
    public $words = 0;

    /** @var int sources */
    public $sources = 0;

    /** @var string spacing */
    public $spacing = self::SPACING_DOUBLE;

    /** @var string paper_details */
    public $paper_details;

    /** @var string software */
    public $software;

    /** @var string keywords */
    public $keywords;

    /** @var int english_type_id */
    public $english_type_id;

    /** @var int target_audience_id */
    public $target_audience_id;

    /** @var int writing_tone_id */
    public $writing_tone_id;

    /** @var int writing_style_id */
    public $writing_style_id;

    /** @var int content_feel_id */
    public $content_feel_id;

    /** @var string comments */
    public $comments;

    /** @var bool writer_samples */
    public $writer_samples = 0;

    /** @var bool used_sources */
    public $used_sources = 0;

    /** @var bool expert_proofreading */
    public $expert_proofreading = 0;

    /** @var bool vip_support */
    public $vip_support = 0;

    /** @var bool plagiarism_report */
    public $plagiarism_report = 0;

    /** @var bool draft_outline */
    public $draft_outline = 0;

    /** @var bool progressive_delivery */
    public $progressive_delivery = 0;

    /** @var float raw_cost */
    public $raw_cost = 0;

    /** @var float total */
    public $total = 0;

    /** @var float tax */
    public $tax = 0;

    /** @var int discount */
    public $discount = 0;

     /** @var int discount_type */
    public $discount_type = 0;

    /** @var float price_per_page */
    public $price_per_page = 0;

    /** @var float price_per_word */
    public $price_per_word = 0;

    /** @var float price_modifier */
    public $price_modifier = 0;

    /** @var float task_item_price */
    public $task_item_price = 0;
    
    /** @var float total_paid */
    public $total_paid = 0;

    /** @var string paid_at */
    public $paid_at;

    /** @var bool is_paid */
    public $is_paid = 0;

    /** @var float writer_pay */
    public $writer_pay = 0;

    /** @var bool is_writer_paid */
    public $is_writer_paid = 0;

    /** @var string writer_paid_at */
    public $writer_paid_at;

    /** @var bool is_writer_confirmed */
    public $is_writer_confirmed = 0;

    /** @var int ordermanager_id */
    public $ordermanager_id;

    /** @var float ordermanager_pay */
    public $ordermanager_pay = 0;

    /** @var bool is_ordermanager_paid */
    public $is_ordermanager_paid = 0;

    /** @var string ordermanager_paid_at */
    public $ordermanager_paid_at;

    /** @var bool is_editor_confirmed */
    public $is_editor_confirmed = 0;

    /** @var float price_per_page */
    public $editor_pay = 0;

    /** @var bool is_editor_paid */
    public $is_editor_paid = 0;

    /** @var string editor_paid_at */
    public $editor_paid_at;
    
    /** @var int hrs_customer */
    public $hrs_customer = 0;

    /** @var string customer_deadline */
    public $customer_deadline;

    /** @var int hrs_writer */
    public $hrs_writer = 0;

    /** @var string writer_deadline */
    public $writer_deadline;

    /** @var bool writer_assigned */
    public $writer_assigned = 0;

    /** @var string writer_assigned_at */
    public $writer_assigned_at;

    /** @var string editor_assigned */
    public $editor_assigned = 0;

    /** @var string editor_assigned_at */
    public $editor_assigned_at;

    /** @var string ordermanager_assigned */
    public $ordermanager_assigned = 0;

    /** @var string ordermanager_assigned_at */
    public $ordermanager_assigned_at;

    /** @var string approved_at */
    public $approved_at;

    /** @var string completed_at */
    public $completed_at;

    /** @var string cancelled_at */
    public $cancelled_at;

    /** @var string delivered_at */
    public $delivered_at;

    /** @var string finished_at */
    public $finished_at;

    /** @var bool is_pd_schedule_accepted */
    public $is_pd_schedule_accepted = 0;

    /** @var string pd_schedule_accepted_at */
    public $pd_schedule_accepted_at;

    /** @var bool is_pd_schedule_rejected */
    public $is_pd_schedule_rejected = 0;

    /** @var string pd_schedule_rejected_at */
    public $pd_schedule_rejected_at;

    /** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
    public $date_add;

    public $winbackCoupons = array();

    public $presentFiles = array();

    public $currency;

    public $orderWriter;
    
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'order',
        'primary' => 'order_id',
        'fields' => array(
            'tracking_id' => array('type' => self::TYPE_STRING, 'size' => 255),
            'site_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'form_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'form_type' => array('type' => self::TYPE_STRING),
            'payment_method' => array('type' => self::TYPE_STRING),
            'currency_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'currency_to_usd_rate' => array('type' => self::TYPE_FLOAT, 'validate' => 'isUnsignedFloat', 'required' => false),
            'gmt' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'coupon_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'customer_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'writer_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'current_writer_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'requested_writer_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'editor_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'status_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'parent_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'service_type' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'tariff_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'title' => array('type' => self::TYPE_STRING),
            'academic_level_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_option' => array('type' => self::TYPE_STRING),
            'topic_category_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'topic_category_option' => array('type' => self::TYPE_STRING),
            'is_complex_assignment' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'paper_format_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_format_option' => array('type' => self::TYPE_STRING),
            'writer_category_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'writer_percent' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'task_size_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'pages' => array('type' => self::TYPE_FLOAT),
            'charts' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'slides' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'excel_sheets' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'words' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'sources' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'spacing' => array('type' => self::TYPE_STRING),
            'paper_details' => array('type' => self::TYPE_HTML),
            'software' => array('type' => self::TYPE_STRING),
            'keywords' => array('type' => self::TYPE_STRING),
            'english_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'target_audience_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'writing_tone_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'writing_style_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'content_feel_id' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'comments' => array('type' => self::TYPE_STRING),
            'writer_samples' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'used_sources' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'expert_proofreading' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'vip_support' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'plagiarism_report' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'draft_outline' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'progressive_delivery' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'raw_cost' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'tax' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'discount' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'discount_type' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'price_per_page' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'price_per_word' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'price_modifier' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'task_item_price' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'total_paid' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'paid_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'is_paid' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'writer_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'is_writer_paid' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'writer_paid_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'is_writer_confirmed' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'editor_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'is_editor_paid' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'editor_paid_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'is_editor_confirmed' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'ordermanager_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'ordermanager_pay' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'is_ordermanager_paid' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'ordermanager_paid_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'hrs_customer' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'customer_deadline' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'hrs_writer' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'writer_deadline' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'writer_assigned' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'writer_assigned_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'editor_assigned' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'editor_assigned_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'ordermanager_assigned' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'ordermanager_assigned_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'approved_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'completed_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'cancelled_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'delivered_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'finished_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'is_pd_schedule_accepted' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'pd_schedule_accepted_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'is_pd_schedule_rejected' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'pd_schedule_rejected_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        )
    );

    protected static $_historyCache = [];
    
    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null)
    {
        parent::__construct($id);

        if($this->id) {
            $this->currency = $this->getCurrency();

            if($this->writer_id && !$this->current_writer_id) {
                $this->setCurrentWriter($this->writer_id, true);
            }

            $orderWriter = $this->getCurrentWriterFull();
            if($orderWriter) {
                $this->orderWriter = new OrderEmployee((int) $orderWriter['order_employee_id']);
            }
        }
    }

    public function getCurrency() {
        // currency 
        $default_currency_id = Configuration::get('CURRENCY_DEFAULT', $this->site_id, 1);

        $currency_id = $this->currency_id ? $this->currency_id : $default_currency_id;
        $currency = new Currency((int) $currency_id);
        $currency->conversion_rate = $this->currency_to_usd_rate ? (float) $this->currency_to_usd_rate : (float) $currency->conversion_rate;

        return $currency;
    }
 
    public function getObject( $order = array() ) { 
        if( !ArrayUtils::has($order, 'order_id') ) {
            return $this;
        }

        $this->id = $order['order_id'];

        foreach ($order as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }
        
        $this->currency = $this->getCurrency();

        return $this;
    }

    public function add($autodate = true, $null_values = true) {
		if ( !$this->getNextOrderId() ) {
			$this->id = self::START_ORDER_ID;
		} else {
            $order_autoincrement = Configuration::get('ORDER_AUTOINCREMENT', PROX_SITE_ID, 2);
			$next_order_id = Db::getInstance()->getValue(
                'SELECT MAX(`order_id`)+' . (int) $order_autoincrement . ' FROM ' . Db::prefix('order')
            );
			$this->id = $next_order_id;
        }
        
        $this->force_id = true;
        
        if(!$this->currency_id) {
            $currency = new Currency( (int) $this->currency_id );
            if(Validate::isLoadedObject($currency)) {
                $this->currency_id = $currency->id;
                $this->currency_to_usd_rate = $currency->conversion_rate;
            }
        }

		return parent::add($autodate, $null_values);
    }

    public function delete() {
        $success = parent::delete();

        if($success) {
            Db::getInstance()->delete('additional_service', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('paypal_payment', 'node_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_bid', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_view', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_file', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_message', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_revision', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_meta', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('order_rating', 'order_id = ' . (int) $this->id );
            Db::getInstance()->delete('notification', 'action = "order" AND node_url = ' . (int) $this->id );
        }
    }

    public function getItemTitle($id) {
        switch ($id) {
            case 2: 
                
                return $this->service_type == Order::ARTICLE_WRITING ? 'Standard' : 'Best available';
                break;

            case 3: 
                return $this->service_type == Order::ARTICLE_WRITING ? 'Premium' : 'Advanced';
                break;

            case 4: 
                return $this->service_type == Order::ARTICLE_WRITING ? 'Ultimate' : 'ENL';
                break;

            default: 
                return $this->service_type == Order::ARTICLE_WRITING ? 'Standard' : 'Best available';
                break;
        }
    }

    public function getItemTitleUk($id, $academicLevel) {
        switch ($academicLevel) {
            case LEVEL_HIGHSCHOOL:
              switch ($id) {
                case 2: 
                    return 'Grade B';
                    break;

                case 3: 
                    return 'Grade A';
                    break;

                case 4: 
                    return 'Grade A+';
                    break;

                default: 
                    return $this->getItemTitle($id);
                    break;
            }

            case LEVEL_COLLEGE:
            case LEVEL_UNIVERSITY:
                switch ($id) {
                    case 2: 
                        return 'Undergraduate 2:2';
                        break;

                    case 3: 
                        return 'Undergraduate 2:1';
                        break;

                    case 4: 
                        return 'Undergraduate 1st';
                        break;

                    default: 
                        return  $this->getItemTitle($id);
                        break;
                }
            case LEVEL_MASTERS:
              switch ($id) {
                case 2: 
                    return 'Pass';
                    break;
                case 3: 
                    return 'Merit';
                    break;
                case 4: 
                    return 'Distinction';
                    break;
                default: 
                    return $this->getItemTitle($id);
                    break;
              }
            default: 
                return $this->getItemTitle($id);
          }
    } 

    public function isTechinicalOrder() {
        return (bool) $this->service_type !== Order::PROGRAMMING && $this->service_type !== Order::CALCULATIONS;
    }

    public function present() {
        $orderPresenter = new OrderPresenter();
        return $orderPresenter->present($this);
    }

    public function calculate( $options = array() ) {
        switch($this->service_type) {
            case self::COMPLEX_ASSIGNMENT;
                $priceCalculator = new PriceCalculator();
                $defaultOrder = $this->defaultOrder();

                $defaultOrder = array_merge($defaultOrder, $options);

                return $priceCalculator->calculate( $defaultOrder );
                break;

            case self::PROGRAMMING:
            case self::CALCULATIONS:
	            $cost = PriceCalculator::normalizePrice($this->task_item_price * $this->price_modifier);

                $discount = 0;
                if( $this->discount ) {
                    $discount = $this->discount_type == Coupon::PERCENT ? ($cost * $this->discount) / 100 : $this->discount;
                }

                $servicesById = array(
                    self::PROGRAMMING => array(
                        'type_id' => self::PROGRAMMING,
                        'title' => $this->topic_category_option,
                        'cost' => $cost,
                        'winbackReduction' => 0,
                        'active' => true
                    ),
                    self::DISCOUNT => array(
                        'winbackReduction' => 0,
                        'cost' => $discount
                    ),
                );

                return array(
                    'servicesById' => $servicesById,
                    'totalCost' => PriceCalculator::normalizePrice($cost - $discount)
                );
                break;

            case self::ARTICLE_WRITING:
                $defaultOrder = $this->defaultOrder();

                $cost = PriceCalculator::normalizePrice($this->price_per_word * $this->words);

                $discount = 0;
                if( $this->discount ) {
                    $discount = $this->discount_type == Coupon::PERCENT ? ($cost * $this->discount) / 100 : $this->discount;
                }

                $writerPercent = $this->writer_percent;
                $prefferedWriterPrice = PriceCalculator::normalizePrice(
                    ($cost * $writerPercent) / 100
                );

	            $baseCost = $cost + $prefferedWriterPrice;

                $servicesById = array(
                    self::WRITING_WORDS => array(
                        'type_id' => (int) self::WRITING_WORDS,
                        'quantity' => (int) $this->words,
                        'itemPrice' => (float) $this->price_per_word,
                        'cost' => (float) $cost,
                        'winbackReduction' => 0,
                        'active' => true
                    ),
                    self::CHOOSE_WRITER => array(
                        'type_id' => (int) self::CHOOSE_WRITER,
                        'cost' => (float) $prefferedWriterPrice,
                        'winbackReduction' => 0,
                        'active' => true
                    ),
                    self::DISCOUNT => array(
                        'winbackReduction' => 0,
                        'cost' => (float) $discount
                    ),
                );

                $defaultOrder = array_merge($defaultOrder, $options);

                $writerCostPerWordType = ArrayUtils::get($defaultOrder, 'writerCpwType');
                if ($writerCostPerWordType == "fixed") {
                    $writerPricePerWord = (float) ArrayUtils::get($defaultOrder, 'writerAmountCpw');
                } else {
                    $writerPercentageCpw = (float) ArrayUtils::get($defaultOrder, 'writerAmountCpw');
                    $writerPricePerWord = PriceCalculator::normalizePrice(
                        ($baseCost * $writerPercentageCpw) / 100
                    );
                }

                $editorCostPerWordType = ArrayUtils::get($defaultOrder, 'editorCpwType');
                if ($editorCostPerWordType == "fixed") {
                    $editorPricePerWord = (float) ArrayUtils::get($defaultOrder, 'editorAmountCpw');
                } else {
                    $editorPercentageCpw = (float) ArrayUtils::get($defaultOrder, 'editorAmountCpw');
                    $editorPricePerWord = PriceCalculator::normalizePrice(
                        ($baseCost * $editorPercentageCpw) / 100
                    );
                }
                
                $writerWordsCost = PriceCalculator::normalizePrice($this->words * $writerPricePerWord);
                $editorWordsCost = PriceCalculator::normalizePrice($this->words * $editorPricePerWord);

                return array(
                    'servicesById' => $servicesById,
                    'totalCost' => (float) PriceCalculator::normalizePrice($baseCost - $discount),
                    'writerCost' => (float) $writerWordsCost,
                    'editorCost' => (float) $editorWordsCost,
                );
                break;
                
            default:
                $pagesPriceMultiplicator = $this->spacing == "single" ? 2 : 1;
                $isComplexAssigment = $this->is_complex_assignment;
                $price = 0;

                $price = ($this->pages * $pagesPriceMultiplicator * $this->price_per_page) +
                        ($this->slides / 2) * $this->price_per_page +
                        ($this->excel_sheets / 2) * $this->price_per_page +
                        ($this->charts / 2) * $this->price_per_page;

                $complexCost = 0;
                if ($isComplexAssigment) {
                    $complexCost = ($price / 100) * 20;
                }

                $writerCost = 0;
                if($this->requested_writer_id) {
                    $writerCost = ($price / 100) * $this->writer_percent;
                }
                    
                $discountPrice = 0;
                if( $this->discount ) {
                    if( $this->discount_type == Coupon::PERCENT ) {
                        $discountPrice = ($price * $this->discount) / 100;
                    } else {
                        $discountPrice = $this->discount;
                    }
                }

                $price += $complexCost + $writerCost;
                $totalCost = $price - $discountPrice;

                $pricesCost = array(
                    'rawCost' => $price,
                    'totalCost' => $totalCost
                );

                return $pricesCost;
                break;
        }
    }

    public function defaultOrder() {
        $discount = null;
        if($this->discount) {
            $discount = array(
                'type_id' => $this->discount_type,
                'value' => $this->discount
            );
        }

        $writerCppType = Configuration::get('WRITER_CPP_TYPE', PROX_SITE_ID, "percentage");
        if($writerCppType == 'percentage') {
            $writerAmountCpp = Configuration::get('WRITER_PERCENTAGE_CPP', PROX_SITE_ID, 0);
        } else {
            $writerAmountCpp = Configuration::get('WRITER_AMOUNT_CPP', PROX_SITE_ID, 0);
        }

        $editorCppType = Configuration::get('EDITOR_CPP_TYPE', PROX_SITE_ID, "percentage");
        if($editorCppType == 'percentage') {
            $editorAmountCpp = Configuration::get('EDITOR_PERCENTAGE_CPP', PROX_SITE_ID, 0);
        } else {
            $editorAmountCpp = Configuration::get('EDITOR_AMOUNT_CPP', PROX_SITE_ID, 0);
        }

        $orderManagerCppType = Configuration::get('ORDER_MANAGER_CPP_TYPE', PROX_SITE_ID, "percentage");
        if($orderManagerCppType == 'percentage') {
            $orderManagerAmountCpp = Configuration::get('ORDER_MANAGER_PERCENTAGE_CPP', PROX_SITE_ID, 0);
        } else {
            $orderManagerAmountCpp = Configuration::get('ORDER_MANAGER_AMOUNT_CPP', PROX_SITE_ID, 0);
        }

        $writerCpwType = Configuration::get('WRITER_CPW_TYPE', PROX_SITE_ID, "percentage");
        if($writerCpwType == 'percentage') {
            $writerAmountCpw = Configuration::get('WRITER_PERCENTAGE_CPW', PROX_SITE_ID, 0);
        } else {
            $writerAmountCpw = Configuration::get('WRITER_AMOUNT_CPW', PROX_SITE_ID, 0);
        }

        $editorCpwType = Configuration::get('EDITOR_CPW_TYPE', PROX_SITE_ID, "percentage");
        if($editorCpwType == 'percentage') {
            $editorAmountCpw = Configuration::get('EDITOR_PERCENTAGE_CPW', PROX_SITE_ID, 0);
        } else {
            $editorAmountCpw = Configuration::get('EDITOR_AMOUNT_CPW', PROX_SITE_ID, 0);
        }

        $defaultOrder = array (
            'pages' =>                              (float) $this->pages,
            'slides' =>                             (int) $this->slides,
            'words' =>                              (int) $this->words,
            'excel_sheets' =>                       (int) $this->excel_sheets,
            'charts' =>                             (int) $this->charts,
            'discount' =>                           $discount,
            'winbackCoupons' =>                     $this->winbackCoupons,
            'writerCategoryId' =>                   $this->writer_category_id,
            'tariffPricePerPage' =>                 (float) $this->price_per_page,
            'tariffHrs' =>                          $this->hrs_customer,
            'spacing' =>                            $this->spacing,
            'getSamplesOn' =>                       $this->writer_samples,
            'getProgressiveDeliveryOn' =>           $this->progressive_delivery,
            'getUsedSourcesOn' =>                   $this->used_sources,
            'complexAssignmentDiscipline' =>        (bool) $this->is_complex_assignment,
            'writerPercent' =>                      (int) $this->writer_percent,
            'expertProofreading' =>                 (bool) $this->expert_proofreading,
            'vipSupport' =>                         (bool) $this->vip_support,
            'plagiarismReport' =>                   (bool) $this->plagiarism_report,
            'draftOutline' =>                       (bool) $this->draft_outline,
            'writerCppType' =>                      $writerCppType,
            'writerAmountCpp' =>                    (float) $writerAmountCpp,
            'editorCppType' =>                      $editorCppType,
            'editorAmountCpp' =>                    (float) $editorAmountCpp,
            'orderManagerCppType' =>                $orderManagerCppType,
            'orderManagerAmountCpp' =>              (float) $orderManagerAmountCpp,
            'writerCpwType' =>                      $writerCpwType,
            'writerAmountCpw' =>                    (float) $writerAmountCpw,
            'editorCpwType' =>                      $editorCpwType,
            'editorAmountCpw' =>                    (float) $editorAmountCpw
        );

        return $defaultOrder;
    }

    /**
     * used to cache additional payments.
     */
    protected $additionalServices = array();

    public function getAdditionalServices() {
        $options = array();
        $currency = $this->currency;

        if(Validate::isLoadedObject($currency)) {
            $options['currency'] = array(
                'symbol' => $currency->symbol,
                'rate' => (float) $this->currency_to_usd_rate ? (float) $this->currency_to_usd_rate : (float) $currency->conversion_rate
            );
        }
        
        if (empty($this->additionalServices)) {
            $calculatedCosts = $this->calculate();
            $servicesById = ArrayUtils::get($calculatedCosts, 'servicesById');

            $additionalServices = array();

            if($this->writer_samples) {
                $writerSamplesCost = ArrayUtils::get($servicesById, self::PROVIDE_ME_SAMPLES);
                $writerSamplesCost_price = (float) ArrayUtils::get($writerSamplesCost, 'price');

                $additionalServices[] = array(
                    "title" => "Order writer’s samples", 
                    "typeId" => self::PROVIDE_ME_SAMPLES, 
                    "coupon_data" => null, 
                    "description" => formatPrice($writerSamplesCost_price, $options), 
                );
            }

            if($this->used_sources) {
                $usedSourcesCost = ArrayUtils::get($servicesById, self::USED_SOURCES);
                $usedSourcesCost_price = (float) ArrayUtils::get($usedSourcesCost, 'price');

                $additionalServices[] = array(
                    "title" => "Copy of sources used", 
                    "typeId" => self::USED_SOURCES, 
                    "coupon_data" => null, 
                    "description" => formatPrice($usedSourcesCost_price, $options),
                );
            }


            if($this->expert_proofreading) {
                $expertProofreadingCost = ArrayUtils::get($servicesById, self::EXPERT_PROOFREADING);
                $expertProofreadingCost_price = (float) ArrayUtils::get($expertProofreadingCost, 'price');

                $additionalServices[] = array(
                    "title" => "Expert Proofreading", 
                    "typeId" => self::EXPERT_PROOFREADING, 
                    "coupon_data" => null, 
                    "description" => formatPrice($expertProofreadingCost_price, $options)
                );
            }

            if($this->vip_support) {
                $vipSupportCost = ArrayUtils::get($servicesById, self::VIP_SUPPORT);
                $vipSupportCost_price = (float) ArrayUtils::get($vipSupportCost, 'price');

                $additionalServices[] = array(
                    "title" => "VIP Support", 
                    "typeId" => self::VIP_SUPPORT, 
                    "coupon_data" => null, 
                    "description" => formatPrice($vipSupportCost_price, $options)
                );
            }

            if($this->plagiarism_report) {
                $plagiarismReportCost = ArrayUtils::get($servicesById, self::PLAGIARISM_REPORT);
                $plagiarismReportCost_price = (float) ArrayUtils::get($plagiarismReportCost, 'price');

                $additionalServices[] = array(
                    "title" => "Plagiarism Report", 
                    "typeId" => self::PLAGIARISM_REPORT, 
                    "coupon_data" => null, 
                    "description" => formatPrice($plagiarismReportCost_price, $options)
                );
            }

            if($this->draft_outline) {
                $draftOutlineCost = ArrayUtils::get($servicesById, self::DRAFT_OUTLINE);
                $draftOutlineCost_price = (float) ArrayUtils::get($draftOutlineCost, 'price');

                $additionalServices[] = array(
                    "title" => "Draft/Outline", 
                    "typeId" => self::DRAFT_OUTLINE, 
                    "coupon_data" => null, 
                    "description" => formatPrice($draftOutlineCost_price, $options)
                );
            }

            if($this->progressive_delivery) {
                $progressiveDeliveryCost = ArrayUtils::get($servicesById, self::PROGRESSIVE_DELIVERY);
                $progressiveDeliveryCost_price = (float) ArrayUtils::get($progressiveDeliveryCost, 'price');

                $additionalServices[] = array(
                    "title" => "Progressive delivery", 
                    "typeId" => self::PROGRESSIVE_DELIVERY, 
                    "coupon_data" => null, 
                    "description" => formatPrice($progressiveDeliveryCost_price, $options) 
                );
            }

            $this->additionalServices = $additionalServices;
        }

        return $this->additionalServices;
    }

    /**
     * used to cache additional payments.
     */
    protected $additionalPayments = array();

    public function getAdditionalPayments() {
        if (empty($this->additionalPayments)) {

            $additionalPayments = array();

            $sql = new DbQuery();
	        $sql->select('*');
	        $sql->from('additional_service');
			$sql->where('order_id = ' . (int) $this->id);
	        $sql->where('is_paid = 1');
            $result = Db::getInstance()->executeS($sql);
            
            foreach( $result as $additionalPayment ) {
                $amount =  ArrayUtils::get($additionalPayment, 'total');

                $additionalPayments[] = [
                    'id' => (int) ArrayUtils::get($additionalPayment, 'additional_service_id'),
                    'title' => "Additional Payment of " . formatPrice($amount),
                    'price' => (float) $amount,
                    'paidAt' => ArrayUtils::get($additionalPayment, 'paid_at'),
                ];
            }

            $this->additionalPayments = $additionalPayments;
        }

        return $this->additionalPayments;
    }

    /**
     * @since 1.5.0.4
     *
     * @return OrderState or null if Order haven't a state
     */
    public function getCurrentOrderState()
    {
        if ($this->current_state) {
            return new OrderState($this->current_state);
        }

        return null;
    }

    /**
     * Get order history.
     *
     * @param int $order_state_id Filter a specific order status
     * @param int $no_hidden Filter no hidden status
     * @param int $filters Flag to use specific field filter
     *
     * @return array History entries ordered by date DESC
     */
    public function getHistory($order_state_id = false, $no_hidden = false, $filters = 0)
    {
        if (!$order_state_id) {
            $order_state_id = 0;
        }

        $logable = false;
        $delivery = false;
        $paid = false;
        $shipped = false;
        if ($filters > 0) {
            if ($filters & OrderState::FLAG_NO_HIDDEN) {
                $no_hidden = true;
            }
            if ($filters & OrderState::FLAG_DELIVERY) {
                $delivery = true;
            }
            if ($filters & OrderState::FLAG_LOGABLE) {
                $logable = true;
            }
            if ($filters & OrderState::FLAG_PAID) {
                $paid = true;
            }
            if ($filters & OrderState::FLAG_SHIPPED) {
                $shipped = true;
            }
        }

        if (!isset(self::$_historyCache[$this->id . '_' . $order_state_id . '_' . $filters]) || $no_hidden) {
            $result = Db::getInstance()->executeS('
            SELECT os.*, oh.*, e.`first_name` as employee_firstname, e.`last_name` as employee_lastname, osl.`name` as ostate_name
            FROM ' . Db::prefix('order') . ' o
            LEFT JOIN ' . Db::prefix('order_history') . ' oh ON o.`id_order` = oh.`id_order`
            LEFT JOIN ' . Db::prefix('order_state') . ' os ON os.`order_state_id` = oh.`order_state_id`
            LEFT JOIN ' . Db::prefix('employee') . ' e ON e.`employee_id` = oh.`employee_id`
            WHERE oh.id_order = ' . (int) $this->id . '
            ' . ($no_hidden ? ' AND os.hidden = 0' : '') . '
            ' . ($logable ? ' AND os.logable = 1' : '') . '
            ' . ($delivery ? ' AND os.delivery = 1' : '') . '
            ' . ($paid ? ' AND os.paid = 1' : '') . '
            ' . ($shipped ? ' AND os.shipped = 1' : '') . '
            ' . ((int) $order_state_id ? ' AND oh.`order_state_id` = ' . (int) $order_state_id : '') . '
            ORDER BY oh.date_add DESC, oh.id_order_history DESC');

            if ($no_hidden) {
                return $result;
            }
            
            self::$_historyCache[$this->id . '_' . $order_state_id . '_' . $filters] = $result;
        }

        return self::$_historyCache[$this->id . '_' . $order_state_id . '_' . $filters];
    }

    /**
     * Clean static history cache, must be called when an OrderHistory is added as it changes
     * the order history and may change its value for isPaid/isDelivered/... This way calls to
     * getHistory will be up to date.
     */
    public static function cleanHistoryCache()
    {
        self::$_historyCache = [];
    }

    /**
     * used to cache order reviews.
     */
    protected $orderReviews = array();

    public function getOrderReviews() {
        if (empty($this->orderReviews)) {

            $orderReviews = array();

            $sql = new DbQuery();
	        $sql->select('orating.rating_id');
	        $sql->from('order_rating', 'orating');
			$sql->where('orating.order_id = ' . (int) $this->id );
            $result = Db::getInstance()->executeS($sql);

            foreach( $result as $orderReview ) {
                $orderReview = new Rating( (int) $orderReview['rating_id'] );
                if(Validate::isLoadedObject($orderReview)) {
                    $orderReviews[] = $orderReview->present();
                }
            }

            $this->orderReviews = $orderReviews;
        }

        return $this->orderReviews;
    }

    /**
     * used to cache order files.
     */
    protected $orderFiles = null;

    public function getFiles() {
        if (is_null($this->orderFiles)) {
            $sql = new DbQuery();
            $sql->select('f.file_id');
            $sql->from('order_file', 'f');
            $sql->where('f.`order_id` = ' . (int) $this->id );
            $sql->orderBy('f.file_id DESC');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }

            $this->orderFiles = $result;
        }

		return $this->orderFiles;
    }
    
    /**
     * used to cache order messages.
     */
    protected $orderMessages = null;

    public function getMessages() {
        $app = Application::getInstance();
        $user = $app->user;

        if (is_null($this->orderMessages)) {
            $sql = new DbQuery();
            $sql->select('m.message_id');
            $sql->from('order_message', 'm');
            $sql->where('m.`order_id` = ' . (int) $this->id );
            
            if(!$user->is_admin && !$user->is_sub_admin) {
                $sql->where(' (m.`from_department` = ' . (int) DEPARTMENT_WRITER . ') OR ( m.`to_department` = ' . (int) DEPARTMENT_WRITER . ' AND is_verified = 1 ) ');
            } 

            $sql->orderBy('m.message_id DESC');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }

            $messages = array();
            foreach( $result as $message ) {
                $message = new Message( (int) $message['message_id'] );
                if(Validate::isLoadedObject($message)) {
                    $messages[] = $message;
                }
            }

            $this->orderMessages = $messages;
        }

		return $this->orderMessages;
    }

    /**
     * used to cache order writer.
     */
    protected $cacheWriter = null;

    /**
     * Get order writer.
     *
     * @return Employee $writer
     */
    public function getWriter()
    {
        if (is_null($this->cacheWriter)) {
            $this->cacheWriter = new Employee((int) $this->writer_id);
        }

        return $this->cacheWriter;
    }
    
    /**
     * used to cache order editor.
     */
    protected $cacheEditor = null;

    /**
     * Get order editor.
     *
     * @return Employee $editor
     */
    public function getEditor()
    {
        if (is_null($this->cacheEditor)) {
            $this->cacheEditor = new Employee((int) $this->editor_id);
        }

        return $this->cacheEditor;
	}

    /**
     * used to cache order manager.
     */
    protected $cacheOrderManager = null;

    /**
     * Get order editor.
     *
     * @return Employee $orderManager
     */
    public function getOrderManager()
    {
        if (is_null($this->cacheOrderManager)) {
            $this->cacheOrderManager = new Employee((int) $this->ordermanager_id);
        }

        return $this->cacheOrderManager;
	}

	/**
     * used to cache order customer.
     */
    protected $cacheCustomer = null;

    /**
     * Get order customer.
     *
     * @return User $customer
     */
    public function getCustomer()
    {
        if (is_null($this->cacheCustomer)) {
            $this->cacheCustomer = new Customer((int) $this->customer_id);
        }

        return $this->cacheCustomer;
    }

    public function markAsPaid() {
        if(!$this->is_paid) {
            $this->is_paid = 1;
            $this->total_paid = $this->total;
            $this->paid_at = DateUtils::now();
        }

        // $writer_deadline = time() + ($this->hrs_writer*60*60);
        // $this->writer_deadline = DateUtils::date($writer_deadline);
    }

    public function getOrderManagers() {
        $order_statuses = array(
            Order::WRITER_ASSIGNED,
            Order::DELIVERED,
            Order::DONE,
            Order::REVISION
        );
        $statuses = implode(',', $order_statuses);

        $sql = new DbQuery();
        $sql->select('e.employee_id, SUM(o.pages) AS pages, o.slides, o.charts, o.ordermanager_assigned_at');
        $sql->from('employee', 'e');
        $sql->leftJoin('order', 'o', 'o.ordermanager_id = e.employee_id');
        $sql->where('o.status_id IN (' . $statuses . ')');
        $sql->where('e.employee_group = ' . (int) Employee::GROUP_SUB_ADMIN );
        $sql->where('e.is_started = 1' );
        
        $sql->orderBy('o.ordermanager_assigned_at DESC');
        $sql->groupBy('e.employee_id');
        $results = Db::getInstance()->executeS($sql);

        $orderManagers = array();
        foreach($results as $ordermanager ) {
            $employee = new Employee( (int) $ordermanager['employee_id'] );
            if(Validate::isLoadedObject($employee)) {
                $active_pages = $ordermanager['pages'] + $ordermanager['slides'] + $ordermanager['charts'];
                $orderManagers[] = array(
                    'employee_id' => (int) $employee->id,
                    'active_pages' => $active_pages
                );
            }
        }

        usort($orderManagers, function ($emp1, $emp2) {
            return $emp1['active_pages'] <=> $emp2['active_pages'];
        });

        return $orderManagers;
    }

    public function duplicate() {
        if($this->id == null) {
            return false;
        }

        $newOrder = new Order();
        $newOrder->force_id = false;
        $newOrder->service_type = $this->service_type;
        $newOrder->form_type = $this->form_type;
        $newOrder->paper_format_id = $this->paper_format_id;
        $newOrder->paper_format_option = $this->paper_format_option;
        $newOrder->academic_level_id = $this->academic_level_id;
        $newOrder->tariff_id = $this->tariff_id;
        $newOrder->price_per_page = $this->price_per_page;
        $newOrder->hrs_customer = $this->hrs_customer;
        $newOrder->hrs_writer = $this->hrs_writer;
        $newOrder->customer_deadline = DateUtils::date(time() + ($this->hrs_customer*60*60));
        $newOrder->writer_deadline = DateUtils::date(time() + ($this->hrs_writer*60*60));
        $newOrder->customer_id = $this->customer_id;
        $newOrder->site_id = $this->site_id;

        $newOrder->title = $this->title;
        $newOrder->paper_type_id = $this->paper_type_id;
        $newOrder->paper_type_option = $this->paper_type_option;
        $newOrder->topic_category_id = $this->topic_category_id;
        $newOrder->topic_category_option = $this->topic_category_option;
        $newOrder->is_complex_assignment = $this->is_complex_assignment;

        $newOrder->writer_category_id = $this->writer_category_id;
        $newOrder->writer_percent = $this->writer_percent;

        $newOrder->sources = $this->sources;
        $newOrder->pages = $this->pages;
        $newOrder->words = $this->words;
        $newOrder->charts = $this->charts;
        $newOrder->slides = $this->slides;
        $newOrder->excel_sheets = $this->excel_sheets;
        $newOrder->spacing = $this->spacing;
        $newOrder->paper_details = $this->paper_details;

        $newOrder->software = $this->software;
        $newOrder->keywords = $this->keywords;
        $newOrder->english_type_id = $this->english_type_id;
        $newOrder->target_audience_id = $this->target_audience_id;
        $newOrder->writing_tone_id = $this->writing_tone_id;
        $newOrder->writing_style_id = $this->writing_style_id;
        $newOrder->content_feel_id = $this->content_feel_id;

        $newOrder->used_sources = $this->used_sources;
        $newOrder->writer_samples = $this->writer_samples;
        $newOrder->expert_proofreading = $this->expert_proofreading;
        $newOrder->progressive_delivery = $this->progressive_delivery;

        $newOrder->raw_cost = $this->raw_cost;
        $newOrder->total = $this->total; 
        $newOrder->writer_pay = $this->writer_pay;
        $newOrder->editor_pay = $this->editor_pay;

        $newOrder->writer_samples = $this->writer_samples;
        $newOrder->used_sources = $this->used_sources;
        $newOrder->is_complex_assignment = $this->is_complex_assignment;
        $newOrder->expert_proofreading = $this->expert_proofreading;
        $newOrder->progressive_delivery = $this->progressive_delivery;
        $newOrder->add();

        return $newOrder;
    }

    public function deliverOrderToCustomer() {
        $app = Application::getInstance();
        $smarty = $app->smarty;

        // get order site details
        $site = new Site( (int) $this->site_id );

        $this->status_id = Order::DELIVERED;
        $this->finished_at = DateUtils::now();
        $this->delivered_at = DateUtils::now();
        $this->update();


        $customer = $this->getCustomer();
        if(Validate::isLoadedObject($customer)) {
            if($customer->is_anonymous) return true;
            
            $order_files = $this->getFiles();

            $file_attachments = array();
            if(Configuration::get('ATTACH_COMPLETED_FILES')) {
                foreach( $order_files as $file ) {
                    $file = new File( (int) $file['file_id']); 
                    if( Validate::isLoadedObject($file) ) {
                        if ( $file->uploader_id != $customer->id ) {
                            $file_attachments[] = $file->attach();
                        }
                    } 
                } 
            }
            
            $smarty->assign([
                'site_url' => $site->domain,
                'order_id' => $this->id,
                'username' => $customer->name,
                'requestRefundAllowed' => (bool) Configuration::get('REQUEST_REFUND_ALLOWED', $customer->site_id)
            ]);

            $order_complete_body = $smarty->fetch('messages/order_complete.tpl');

            $message = new Message();
            $message->order_id = $this->id;
            $message->sender_id = Employee::START_ID;
            $message->from_department = DEPARTMENT_SUPPORT;
            $message->to_department = DEPARTMENT_CUSTOMER;
            $message->subject = "Please check the completed order";
            $message->body = $order_complete_body;
            $message->is_verified = 1;
            $message->verified_at = DateUtils::now();
            $message->add();

            try { 
                $mailParams = array(
                    'order_domain' => $site->domain,
                    'order_url' => $site->domain . '/dashboard/order/' . $this->id,
                    'order_id' => $this->id,
                    'order_title' => $this->title,
                    'username' => $customer->name
                );

                Mail::send(
                    MailTemplate::TEMPLATE_ORDER_COMPLETE,
                    $message->subject,
                    $mailParams,
                    $customer->email,
                    null,
                    null,
                    null,
                    $file_attachments,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    $customer->site_id
                );  
            } catch( \Exception $e ) {
                
            }
        }
    }

    /** Set current order writer
     * @param int $order_employee_id
     */
    public function setCurrentWriter($employee_id, $autofill = false)
    {
        if (empty($employee_id)) {
            return false;
        }

        $writer = new OrderEmployee();
        $writer->order_id = (int) $this->id;
        $writer->employee_type = 'writer';
        $writer->employee_id = (int) $employee_id;

        if($autofill) {
            $writer->hrs_employee = $this->hrs_writer;
            $writer->employee_deadline = $this->writer_deadline;
            $writer->employee_pay = $this->writer_pay;
            $writer->is_employee_paid = $this->is_writer_paid;
            $writer->employee_paid_at = $this->writer_paid_at;
            $writer->is_employee_assigned = $this->writer_assigned;
            $writer->employee_assigned_at = $this->writer_assigned_at;
            $writer->is_employee_confirmed = $this->is_writer_confirmed;
            if($this->is_writer_confirmed) {
                $writer->employee_confirmed_at = $this->writer_assigned_at;
            }
            $writer->is_completed = Validate::isDate($this->completed_at) ? true : false;
            $writer->is_delivered = Validate::isDate($this->delivered_at) ? true : false;
            $writer->is_approved = Validate::isDate($this->approved_at) ? true : false;
            $writer->is_cancelled = Validate::isDate($this->cancelled_at) ? true : false;
            $writer->is_revised = ($this->status_id == Order::REVISION) ? true : false;
            $writer->is_disputed = ($this->status_id == Order::DISPUTE) ? true : false;
            $writer->is_refunded = ($this->status_id == Order::DISPUTE) ? true : false;
        }

        $writer->add();

        $this->current_writer_id = $writer->id;
        $this->writer_id = $employee_id;
        $this->update();

        return $writer->id;
    }

    /**
     * Get current writer stats
     *
     * @return array Order employee details
     */
    public function getCurrentWriterFull()
    {
        return Db::getInstance()->getRow('SELECT oe.* FROM ' . Db::prefix('order_employee') . ' oe WHERE oe.`order_employee_id` = ' . (int) $this->current_writer_id );
    }
    
    /**
     * This method return the ID of the previous order.
     *
     * @since 1.5.0.1
     *
     * @return int
     */
    public function getPreviousOrderId( $customerId = null )
    {
        return Db::getInstance()->getValue('
            SELECT order_id
            FROM ' . DB_PREFIX . 'order 
			WHERE order_id < ' . (int) $this->id . 
			($customerId ? ' AND customer_id = '. (int) $customerId : '') . '
            ORDER BY order_id DESC');
    }
	
	/**
     * This method return the ID of the next order.
     *
     * @since 1.0.0
     *
     * @return int
     */
    public function getNextOrderId( $customerId = null )
    {
        return Db::getInstance()->getValue('
            SELECT order_id
            FROM ' . DB_PREFIX . 'order
			WHERE order_id > ' . (int) $this->id . 
			($customerId ? ' AND customer_id = '. (int) $customerId : '') . '
            ORDER BY order_id ASC');
    }
}