<?php 
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Proxim\Database\DbQuery;
use Proxim\Cache\Cache;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;

/**
 * Class Notification.
 */
class Notification extends ObjectModel
{
    CONST ACTION_ORDER = 'order';
    CONST ACTION_MESSAGE = 'message';
    CONST ACTION_REPLY = 'reply';
    CONST ACTION_FILE = 'file';

    const TYPE_NEW_ORDER = 'new_order'; 
    const TYPE_NEW_INQUIRY = 'new_inquiry';

    const TYPE_PAYMENT = 'new_payment'; 

    const TYPE_MESSAGE = 'new_message'; 
    const TYPE_REPLY = 'new_reply'; 

    const TYPE_DELETE_FILE = 'delete_file'; 

    const TYPE_BID_REQUEST = 'new_bid_request'; 

    /** @var $id Notification ID */
    public $id;
    
    /** @var string to_department */
    public $to_department;
    
    /** @var int to_user_id */
    public $to_user_id;
    
    /** @var string from_department */
    public $from_department;
    
    /** @var int from_user_id */
    public $from_user_id;

    /** @var string action */
    public $action;

    /** @var string node_type */
    public $node_type;

    /** @var string node_url */
    public $node_url;

    /** @var string message */
    public $message;

    /** @var bool is_seen */
    public $is_seen = 0;

    /** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
    public $date_add;
    
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'notification',
        'primary' => 'notification_id',
        'fields' => array(
            'to_department' => array('type' => self::TYPE_STRING, 'size' => 255, 'required' => true),
            'to_user_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'from_department' => array('type' => self::TYPE_STRING, 'size' => 255, 'required' => true),
            'from_user_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'action' => array('type' => self::TYPE_STRING, 'size' => 255, 'required' => true),
            'node_type' => array('type' => self::TYPE_STRING, 'size' => 255, 'required' => true),
            'node_url' => array('type' => self::TYPE_STRING, 'size' => 255, 'required' => true),
            'message' => array('type' => self::TYPE_STRING),
            'is_seen' => array('type' => self::TYPE_BOOL, 'validate' => 'isBool'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
        ),
    );

    public function __construct($id = null)
    {
        parent::__construct($id);
    }

    public function send($autodate = true, $null_values = true) {
        $app = Application::getInstance();
        $user = $app->user;

        /* if the viewer is the target */
        if( $user->id == $this->to_user_id && $this->to_department == DEPARTMENT_CUSTOMER) {
            return;
        }

        /* get receiver user */
        $receiver = new Employee( (int) $this->to_user_id );
        if (!validate::isLoadedObject($receiver)) {
            return;
        }

        /* insert notification */
        $success = parent::add($autodate, $null_values);

        /* update notifications counter +1 */
        $receiver->live_notifications_counter = $receiver->live_notifications_counter+1;
        $receiver->update();

        switch ($this->action) {

            case '':

                break;
        }

        return $success;
    }

    public function present() {
        $app = Application::getInstance();
        $user = $app->user;

        $notification = array(
            "id" => $this->id,
            "seen" => (bool) $this->is_seen,
            "date" => DateUtils::convertUtcDateTimeToTimeZone($this->date_add, $user->timezone)
        );

        $notification['state'] = "info";

        if ( $this->from_department == DEPARTMENT_CUSTOMER ) {
            $sender = new Customer( (int) $this->from_user_id );
            $notification['from_id'] = $sender->id;
            $notification['from_name'] = "Customer " . $sender->id;
        } else {
            $sender = new Employee( (int) $this->from_user_id );
            $notification['from_id'] = $sender->id;
            $notification['from_name'] = "Writer ".$sender->id;
        }

        switch( $this->action ) {

            // order
            case self::ACTION_ORDER:

                switch( $this->node_type ) {

                    case Notification::TYPE_NEW_ORDER:
                        $notification['icon'] = "fa-shopping-basket";
                        $notification['url'] = $app->base_uri . '/order/' . $this->node_url;
                        $notification['message'] = "created a new order";
                        $notification['state'] = "success";
                        break;

                    case Notification::TYPE_BID_REQUEST:
                        $notification['icon'] = "fa-rss";
                        $notification['url'] = $app->base_uri . '/order/' . $this->node_url . '/assign';
                        $notification['message'] = "placed a bid";
                        $notification['state'] = "warning";
                        break;
                }

                break;

            // message
            case self::ACTION_MESSAGE:

                switch( $this->node_type ) {

                    case Notification::TYPE_MESSAGE:
                        $notification['icon'] = "fa-comments";
                        $notification['url'] = $app->base_uri . '/order/' . $this->node_url . "/messages";
                        $notification['message'] = "sent a new message";
                        $notification['state'] = "info";
                        break;
                }

                break;

            // reply
            case self::ACTION_REPLY:

                switch( $this->node_type ) {
                    
                    case Notification::TYPE_REPLY:
                        $notification['icon'] = "fa-comments";
                        $notification['url'] = $app->base_uri . '/order/' . $this->node_url . '/messages';
                        $notification['message'] = "replied to your message";
                        $notification['state'] = "info";
                        break;
                }

                break;
        }

        /* prepare message */
        $notification['full_message'] = $notification['from_name'] . " ". $notification['message'];

        return $notification;
    }
}