<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Proxim\Cache\Cache;
use Proxim\Database\DbQuery;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;

/**
 * MailTemplate
 */
class MailTemplate extends ObjectModel
{ 

    const TEMPLATE_ORDER_PAYMENT = 'order_payment';
    const TEMPLATE_ORDER_NEW_AVAILABLE = 'new_order_available';
    const TEMPLATE_ORDER_FILES = 'order_files';
    const TEMPLATE_ORDER_ASSIGN = 'order_assign'; 
    const TEMPLATE_ORDER_REASSIGN = 'order_reassign'; 
    const TEMPLATE_ORDER_WRITER_SUBMIT = 'writer_submitted_order';
    const TEMPLATE_ORDER_NEW_EDITOR = 'editor_new_order';
    const TEMPLATE_ORDER_COMPLETE = 'order_complete';
    const TEMPLATE_ORDER_STATUS_CHANGE = 'order_status_change';
    const TEMPLATE_ORDER_REVISION = 'order_revision_request';
    const TEMPLATE_ORDER_MESSAGE = 'order_message';
    const TEMPLATE_ORDER_FEEDBACK_REPLY = 'feedback_reply';
    const TEMPLATE_ORDER_PAY_REMINDER = 'pay_order_reminder';
    const TEMPLATE_ORDER_FILE_DELETE = 'order_file_delete';
    const TEMPLATE_ORDER_REFUND_REQUEST = 'order_refund_request';
    
    const TEMPLATE_EMPLOYEE_PASSWORD_CHANGE = 'employee_password_change';
    const TEMPLATE_EMPLOYEE_TWO_FACTOR = 'employee_two_factor_email';

    const TEMPLATE_CUSTOMER_PASSWORD_CHANGE = 'customer_password_change';

	/** @var $id MailTemplate ID */
	public $id;
	
	/** @var string template */
	public $template;

	/** @var string subject */
    public $subject;

    /** @var string from_name */
    public $from_name;

    /** @var string message */
    public $message;
    
    /** @var bool plaintext */
    public $plaintext = 0;
    
    /** @var bool is_disabled */
    public $is_disabled = 0;

	/** @var string date_upd */
	public $date_upd;
	
	/** @var string date_add */
	public $date_add;

    /** @var string Template Name */
	public $name;
	
	/**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'mail_template',
        'primary' => 'mail_template_id',
        'fields' => array(
			'template' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true),
			'subject' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
            'from_name' => array('type' => self::TYPE_STRING, 'validate' => 'isString'),
			'message' => array('type' => self::TYPE_HTML),
            'plaintext' => array('type' => self::TYPE_INT, 'validate' => 'isBool'),
            'is_disabled' => array('type' => self::TYPE_INT, 'validate' => 'isBool'),
			'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
        ),
    );

    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null) {
        parent::__construct($id);
        if($this->id) {
            $this->name = MailTemplate::getTitleByTemplateName($this->template); 
        }
	}

    public function add($autodate = true, $null_values = true) {
		return parent::add($autodate, $null_values);
    }

    public static function getIdByTemplateName($template) {
        $cacheId = 'MailTemplate::getIdByTemplateName_' . pSQL($template);
        if (!Cache::isStored($cacheId)) {
            $query = new DbQuery();
            $query->select('mt.mail_template_id');
            $query->from('mail_template', 'mt');
            $query->where('template = \'' . pSQL($template) . '\'');
            $result = (int) Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($query);
            Cache::store($cacheId, $result);

            return $result;
        }

        return Cache::retrieve($cacheId);
    }

    public static function getMailTemplates() {
        $mailTemplates = array(
            self::TEMPLATE_ORDER_PAYMENT => [
                'template' => self::TEMPLATE_ORDER_PAYMENT,
                'title' => 'Successful Order Payment (Sent to Customer)',
            ],
            self::TEMPLATE_ORDER_NEW_AVAILABLE => [
                'template' => self::TEMPLATE_ORDER_NEW_AVAILABLE,
                'title' => 'New Order Available (Sent to Support & Writers)'
            ],
            self::TEMPLATE_ORDER_FILES => [
                'template' => self::TEMPLATE_ORDER_FILES,
                'title' => 'Order Files Uploaded (Sent to Support & Writers)'
            ],
            self::TEMPLATE_ORDER_ASSIGN => [
                'template' => self::TEMPLATE_ORDER_ASSIGN,
                'title' => 'New Order Assigned (Sent to Writers & Editors)'
            ],
            self::TEMPLATE_ORDER_WRITER_SUBMIT => [
                'template' => self::TEMPLATE_ORDER_WRITER_SUBMIT,
                'title' => 'Writer Submitted Order (Sent to Support)'
            ],
            self::TEMPLATE_ORDER_NEW_EDITOR => [
                'template' => self::TEMPLATE_ORDER_NEW_EDITOR,
                'title' => 'New Order Available (Sent to Editor)'
            ],
            self::TEMPLATE_ORDER_COMPLETE => [
                'template' => self::TEMPLATE_ORDER_COMPLETE,
                'title' => 'Order Complete (Sent to Customer)'
            ],
            self::TEMPLATE_ORDER_STATUS_CHANGE => [
                'template' => self::TEMPLATE_ORDER_STATUS_CHANGE,
                'title' => 'Order Status Changed (Sent to Customer)'
            ],
            self::TEMPLATE_ORDER_REVISION => [
                'template' => self::TEMPLATE_ORDER_REVISION,
                'title' => 'Order Revision Request (Sent to Support)'
            ],
            self::TEMPLATE_ORDER_MESSAGE => [
                'template' => self::TEMPLATE_ORDER_MESSAGE,
                'title' => 'New Message (Sent to Customer & Writers)'
            ],
            self::TEMPLATE_ORDER_FEEDBACK_REPLY => [
                'template' => self::TEMPLATE_ORDER_FEEDBACK_REPLY,
                'title' => 'Feedback Reply  (Sent to Customer)'
            ],
            self::TEMPLATE_ORDER_PAY_REMINDER => [
                'template' => self::TEMPLATE_ORDER_PAY_REMINDER,
                'title' => 'Pay Order Reminder (Sent to Customer)'
            ],
            self::TEMPLATE_ORDER_FILE_DELETE => [
                'template' => self::TEMPLATE_ORDER_FILE_DELETE,
                'title' => 'Order File Delete Request (Sent to Support)'
            ],
            self::TEMPLATE_ORDER_REFUND_REQUEST => [
                'template' => self::TEMPLATE_ORDER_REFUND_REQUEST,
                'title' => 'Order Refund Request (Sent to Support)'
            ],
            self::TEMPLATE_EMPLOYEE_PASSWORD_CHANGE => [
                'template' => self::TEMPLATE_EMPLOYEE_PASSWORD_CHANGE,
                'title' => 'Password Reset - Confirmation'
            ],
            self::TEMPLATE_EMPLOYEE_TWO_FACTOR => [
                'template' => self::TEMPLATE_EMPLOYEE_TWO_FACTOR,
                'title' => 'Two Factor Authentication'
            ],
            self::TEMPLATE_CUSTOMER_PASSWORD_CHANGE => [
                'template' => self::TEMPLATE_CUSTOMER_PASSWORD_CHANGE,
                'title' => 'Password Reset - Confirmation'
            ]
        );

        return $mailTemplates;
    }

    public static function getTitleByTemplateName($template) {
        $mailTemplates = self::getMailTemplates();

        $mailTemplate = ArrayUtils::get($mailTemplates, $template);
        return ArrayUtils::get($mailTemplate, 'title');
    }
}