<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Proxim\Application;
use Proxim\ObjectModel;
use Proxim\Configuration;
use Proxim\Site\Site;
use Proxim\Mail;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;

class ErrorLog extends ObjectModel
{
    /** @var $id Log ID */
	public $id;
	
	/** @var string site_domain */
	public $site_domain;
	
	/** @var string message */
	public $message;

    /** @var string date_upd */
	public $date_upd;

	/** @var string date_add */
    public $date_add;
    
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'error_log',
        'primary' => 'log_id',
        'fields' => array(
            'site_domain' => array('type' => self::TYPE_STRING),
            'message' => array('type' => self::TYPE_STRING),
			'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
			'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
        ),
	);

	public function __construct($id = null)
    {
		parent::__construct($id);
    }

    public static function handleError(\Exception $exception ) {
        $site_url = Application::getInstance()->base_uri;
        $message = $exception->getMessage() . ' in ' . $exception->getFile() . ' on line ' . $exception->getLine();
        
        $errorLog = new ErrorLog();
        $errorLog->site_domain = $site_url;
        $errorLog->message = $message;
        $errorLog->add();

        $mailParams = array(
            'message' => $errorLog->message,
            'site_domain' => $errorLog->site_domain,
        );

        $mailTo = array(
            PROX_DEVELOPER_MAIL
        );

        if(Configuration::get('SITE_MAIL')) {
            $mailTo[] = Configuration::get('SITE_MAIL');
        }

        Mail::Send(
            'error_log',
            'An Error Occured at ' . $site_url,
            $mailParams,
            $mailTo
        );
    }

    /**
     * @param $idMail Mail ID
     *
     * @return bool Whether removal succeeded
     */
    public static function eraseLog($logId)
    {
        return Db::getInstance()->delete('error_log', 'log_id = ' . (int) $logId);
    }

    /**
     * @return bool
     */
    public static function eraseAllLogs()
    {
        return Db::getInstance()->execute('TRUNCATE TABLE ' . DB_PREFIX . 'error_log');
    }
}