<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim;

use Db;
use Exception;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;

/**
 * Class Configuration
 */
class Configuration extends ObjectModel
{
    public $id;

    /** @var int site_id */
    public $site_id;

    /** @var string key */
    public $name;

    /** @var string value */
    public $value;

    /** @var string Object creation date */
    public $date_add;

    /** @var string Object last modification date */
    public $date_upd;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'configuration',
        'primary' => 'configuration_id',
        'fields' => array(
            'name' => array('type' => self::TYPE_STRING, 'validate' => 'isConfigName', 'required' => true, 'size' => 254),
            'site_id' => array('type' => self::TYPE_INT),
            'value' => array('type' => self::TYPE_STRING),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        ),
    );

    /** @var array Configuration cache (kept for backward compat) */
    protected static $_cache = null;
    protected static $_new_cache_global = null;

    /** @var array Configuration cache with optimised key order */
    protected static $_new_cache_site = null;
    protected static $_initialized = false;

    /** @var array Vars types */
    protected static $types = array();

    /**
     * Return ID a configuration key.
     *
     * @param string $key
     *
     * @return int Configuration key ID
     */
    public static function getIdByName($key, $siteId = null)
    {
        $sql = 'SELECT `' . bqSQL(self::$definition['primary']) . '`
                FROM ' . Db::prefix( bqSQL(self::$definition['table']) ) . '
                WHERE name = \'' . pSQL($key) . '\'
                ' . Configuration::sqlRestriction($siteId);

        return (int) Db::getInstance()->getValue($sql);
    }

    /**
     * Is the configuration loaded.
     *
     * @return bool `true` if configuration is loaded
     */
    public static function configurationIsLoaded()
    {
        return self::$_initialized;
    }

    /** 
     * Load all configuration data.
     */
    public static function loadConfiguration()
    {
        $sql = 'SELECT `site_id`, `name`, `value` FROM ' . Db::prefix( bqSQL(self::$definition['table']) );
        $db = Db::getInstance();

        $results = $db->executeS($sql);

        if ($results) {
            foreach ($results as $row) {
                if (!isset(self::$_cache[self::$definition['table']])) {
                    self::$_cache[self::$definition['table']] = array(
                        'global' => array()
                    );
                }

                if ($row['value'] === null) {
                    $row['value'] = '';
                }

                if ($row['site_id']) {
                    self::$_cache[self::$definition['table']][$row['site_id']][$row['name']] = $row['value'];
                    self::$_new_cache_site[$row['name']][$row['site_id']] = $row['value'];
                } else {
                    self::$_cache[self::$definition['table']]['global'][$row['name']] = $row['value'];
                    self::$_new_cache_global[$row['name']] = $row['value'];
                }
            }

            self::$_initialized = true;
        }
    }

    /**
     * Get a single configuration value.
     *
     * @param string $key Key wanted
     *
     * @return string Value
     */
    public static function get($key, $siteId = null, $default = false)
    {
        // Init the cache on demand
        if (!self::$_initialized) {
            Configuration::loadConfiguration();
        }

        if (self::$_new_cache_site === null) {
            $siteId = 0;
        } else {
            if($siteId === null) $siteId = PROX_SITE_ID;
        }

        if ($siteId && Configuration::hasKey($key, $siteId)) {
            return self::$_new_cache_site[$key][$siteId];
        } elseif (Configuration::hasKey($key)) {
            return self::$_new_cache_global[$key];
        }

        return $default;
    }

    /**
     * Get global value.
     *
     * @param string $key Configuration key
     *
     * @return string
     */
    public static function getGlobalValue($key)
    {
        return Configuration::get($key);
    }

    /**
     * Check if key exists in configuration.
     *
     * @param string $key
     *
     * @return bool
     */
    public static function has($key)
    {
        if (!is_int($key) && !is_string($key)) {
            return false;
        }

        return isset(self::$_new_cache_global[$key]);
    }

    /**
     * Get several configuration values (in one language only).
     *
     * @throws ProximException
     *
     * @param array $keys Keys wanted
     *
     * @return array Values
     */
    public static function getMultiple($keys, $siteId = null)
    {
        if (!is_array($keys)) {
            throw new Exception('keys var is not an array');
        }

        $results = array();
        foreach ($keys as $key) {
            $results[$key] = Configuration::get($key, $siteId);
        }

        return $results;
    }

    /**
     * Check if key exists in configuration.
     *
     * @param string $key
     *
     * @return bool
     */
    public static function hasKey($key, $siteId = null)
    {
        if (!is_int($key) && !is_string($key)) {
            return false;
        }

        if ( $siteId ) {
            return isset(self::$_new_cache_site[$key][$siteId]);
        }

        return isset(self::$_new_cache_global[$key]);
    }

    /**
     * Set TEMPORARY a single configuration value (in one language only).
     *
     * @param string $key Configuration key
     * @param mixed $values `$values` is an array if the configuration is multilingual, a single string else
     */
    public static function set($key, $values, $siteId = null)
    {
        if (!Validate::isConfigName($key)) {
            die(sprintf('[%s] is not a valid configuration key', Tools::htmlentitiesUTF8($key)) );
        }

        if ($siteId === null) {
            $siteId = PROX_SITE_ID;
        }

        if (!is_array($values)) {
            $values = array($values);
        }

        foreach ($values as $value) {
            if ($siteId) {
                self::$_new_cache_site[$key][$siteId] = $value;
                self::$_cache[self::$definition['table']]['site'][$siteId][$key] = $value;
            } else {
                self::$_new_cache_global[$key] = $value;
                self::$_cache[self::$definition['table']]['global'][$key] = $value;
            }
        }
    }

    /**
     * Update configuration key for global context only.
     *
     * @param string $key
     * @param mixed $values
     * @param bool $html
     *
     * @return bool
     */
    public static function updateGlobalValue($key, $values, $html = false)
    {
        return Configuration::updateValue($key, $values, $html, 0);
    }

    /**
     * Update configuration key and value into database (automatically insert if key does not exist).
     *
     * Values are inserted/updated directly using SQL, because using (Configuration) ObjectModel
     * may not insert values correctly (for example, HTML is escaped, when it should not be).
     *
     * @TODO Fix saving HTML values in Configuration model
     *
     * @param string $key Configuration key
     * @param mixed $values $values is an array if the configuration is multilingual, a single string else
     * @param bool $html Specify if html is authorized in value
     *
     * @return bool Update result
     */
    public static function updateValue( $key, $values, $html = false, $siteId = null )
    {
        if (!Validate::isConfigName($key)) {
            die(sprintf('[%s] is not a valid configuration key', Tools::htmlentitiesUTF8($key)) );
        }

        if (!is_array($values)) {
            $values = array($values);
        }

        if ($siteId === null) {
            $siteId = PROX_SITE_ID;
        }

        if ($html) {
            foreach ($values as &$value) {
                $value = Tools::purifyHTML($value);
            }
            unset($value);
        }

        $result = true;
        foreach ($values as $value) {
            $storedValue = Configuration::get($key, $siteId);
            // if there isn't a $stored_value, we must insert $value
            if (
                (!is_numeric($value) && $value === $storedValue) || 
                (is_numeric($value) && $value == $storedValue && 
                Configuration::hasKey($key, $siteId))) 
            {
                continue;
            }

            // If key already exists, update value
            if (Configuration::hasKey($key, $siteId)) {
                $result &= Db::getInstance()->update(self::$definition['table'], array(
                    'value' => pSQL($value, $html),
                    'date_upd' => date('Y-m-d H:i:s'),
                ), '`name` = \'' . pSQL($key) . '\'' . Configuration::sqlRestriction($siteId), 1, true);
            } else {
                // If key does not exists, create it
                if (!$configID = Configuration::getIdByName($key, $siteId)) {
                    $now = date('Y-m-d H:i:s');
                    $data = array(
                        'name' => pSQL($key),
                        'site_id' => $siteId ? (int) $siteId : null,
                        'value' => pSQL($value, $html),
                        'date_add' => $now,
                        'date_upd' => $now,
                    );
                    $result &= Db::getInstance()->insert(self::$definition['table'], $data, true);
                    $configID = Db::getInstance()->Insert_ID();
                }
            }
        }

        Configuration::set($key, $values, $siteId);

        return $result;
    }

    /**
     * Add SQL restriction on shops for configuration table.
     *
     * @param int $siteId
     *
     * @return string
     */
    protected static function sqlRestriction($siteId)
    {
        if ($siteId) {
            return ' AND site_id = ' . (int) $siteId;
        } else {
            return ' AND (site_id IS NULL OR site_id = 0)';
        }
    }

    /**
     * Delete a configuration key in database (with or without language management).
     *
     * @param string $key Key to delete
     *
     * @return bool Deletion result
     */
    public static function deleteByName($key)
    {
        if (!Validate::isConfigName($key)) {
            return false;
        }

        $result = Db::getInstance()->execute('
            DELETE FROM `' . DB_PREFIX . bqSQL(self::$definition['table']) . '`
            WHERE `name` = "' . pSQL($key) . '"'
        );

        self::$_cache = null;
        self::$_new_cache_global = null;
        self::$_initialized = false;

        return $result;
    }
}
