<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

namespace Proxim\Addon\Theme;

use Exception;
use Proxim\Application;
use Proxim\Configuration;
use Proxim\FileSystem as ProximFileSystem;
use Proxim\Module\Module;
use Proxim\Tools;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Translation\TranslatorInterface;
use Symfony\Component\Yaml\Parser;

class ThemeManager {
    /**
     * @var Application
     */
    private $app;

    /**
     * @var ThemeValidator
     */
    private $themeValidator;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @var Finder
     */
    private $finder;

    /**
     * @var string|null
     */
    public $sandbox;

    /**
     * @var ThemeRepository
     */
    private $themeRepository;

    /**
     * @param Application $app
     * @param ThemeValidator $themeValidator
     * @param Filesystem $filesystem
     * @param Finder $finder
     * @param ThemeRepository $themeRepository
     */
    public function __construct(
        Application $app,
        ThemeValidator $themeValidator,
        Filesystem $filesystem,
        Finder $finder,
        ThemeRepository $themeRepository
    ) {
        $this->app = $app;
        $this->themeValidator = $themeValidator;
        $this->filesystem = $filesystem;
        $this->finder = $finder;
        $this->themeRepository = $themeRepository;
    }

    /**
     * Add new theme from zipball. This will unzip the file and move the content
     * to the right locations.
     * A theme can bundle modules, resources, documentation, email templates and so on.
     *
     * @param string $source The source can be a module name (installed from either local disk or addons.prestaapp.com).
     *                       or a location (url or path to the zip file)
     *
     * @return bool true for success
     */
    public function install($source)
    {
        if ((filter_var($source, FILTER_VALIDATE_URL))) {
            $source = Tools::createFileFromUrl($source);
        }
        if (preg_match('/\.zip$/', $source)) {
            $this->installFromZip($source);
        }

        return true;
    }

    /**
     * Remove all theme files, resources, documentation and specific modules.
     *
     * @param string $name The source can be a module name (installed from either local disk or addons.prestaapp.com).
     *                     or a location (url or path to the zip file)
     *
     * @return bool true for success
     */
    public function uninstall($name)
    {
        /** @var Theme $theme */
        $theme = $this->themeRepository->getInstanceByName($name);
        $theme->onUninstall();

        $this->filesystem->remove($theme->getDirectory());

        return true;
    }

    /**
     * Download new files from source, backup old files, replace files with new ones
     * and execute all necessary migration scripts form current version to the new one.
     *
     * @param string $name
     * @param string $version the version you want to up upgrade to
     * @param string $source if the upgrade is not coming from addons, you need to specify the path to the zipball
     *
     * @return bool true for success
     */
    public function upgrade($name, $version, $source = null)
    {
        return true;
    }

    /**
     * Actions to perform when switching from another theme to this one.
     * Example:
     *    - update configuration
     *    - enable/disable modules.
     *
     * @param string $name The theme name to enable
     *
     * @return bool True for success
     */
    public function enable($name)
    {
        $app = $this->app;

        /* if file exits, remove it and use YAML configuration file instead */
        @unlink(PROX_DIR_THEMES . $name . '/theme.json');

        /** @var Theme $theme */
        $theme = $this->themeRepository->getInstanceByName($name);

        if($theme->getName() != 'phantom') {
            if (!$this->themeValidator->isValid($theme)) {
                return false;
            }

            $licenseLink = Module::$purchase_link . "/verify.php?" . http_build_query([
                'domain' => $app->request->getHost(),
                'module' => $theme->getName(),
                'activation_key' => $theme->get('license_key')
            ]);

            $ch = curl_init();
            curl_setopt ($ch, CURLOPT_URL, $licenseLink);
            curl_setopt ($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch,  CURLOPT_USERAGENT , "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1)");
            curl_setopt ($ch, CURLOPT_CONNECTTIMEOUT, 0);
            $result = curl_exec($ch);

            if (curl_errno($ch)) {
                return false;
            }
 
            curl_close($ch);
            $result = Tools::jsonDecode($result, true);

            if(isset($result['error'])) {
                return false;
            }
        }

        $this->disable($this->app->theme_name);

        $theme->onEnable();

        Configuration::updateValue('PROX_ACTIVE_THEME', $theme->getName());

        $this->saveTheme($theme);

        return true;
    }

    /**
     * Actions to perform when switching from this theme to another one.
     *
     * @param string $name The theme name to enable
     *
     * @return bool True for success
     */
    public function disable($name)
    {
        /** @var Theme $theme */
        $theme = $this->themeRepository->getInstanceByName($name);

        @unlink(PROX_DIR_THEMES . $name . '/theme.json');

        return true;
    }

    /**
     * Actions to perform to restore default settings.
     *
     * @param string $themeName The theme name to reset
     *
     * @return bool True for success
     */
    public function reset($themeName)
    {
        return $this->disable($themeName) && $this->enable($themeName);
    }

    /**
     * @param array $configuration
     *
     * @return self
     */
    private function doApplyConfiguration(array $configuration): self
    {
        foreach ($configuration as $key => $value) {
            $this->appConfiguration->set($key, $value);
        }

        return $this;
    }

    private function getSandboxPath()
    {
        if (!isset($this->sandbox)) {
            $this->sandbox = PROX_DIR_CACHE . 'sandbox/' . uniqid() . '/';
            $this->filesystem->mkdir($this->sandbox, ProximFileSystem::DEFAULT_MODE_FOLDER);
        }

        return $this->sandbox;
    }

    /**
     * @param string $source
     *
     * @throws Exception
     */
    private function installFromZip($source)
    {
        /** @var Finder $finderClass */
        $finderClass = get_class($this->finder);
        $this->finder = $finderClass::create();

        $sandboxPath = $this->getSandboxPath();
        Tools::ZipExtract($source, $sandboxPath);

        $themeConfigurationFile = $sandboxPath . '/config/theme.yml';

        if (!file_exists($themeConfigurationFile)) {
            throw new Exception('Missing theme configuration file which should be in located in /config/theme.yml');
        }

        $theme_data = (new Parser())->parse(file_get_contents($themeConfigurationFile));

        $theme_data['directory'] = $sandboxPath;

        try {
            $theme = new Theme($theme_data);
        } catch (Exception $exception) {
            throw new Exception(sprintf('Theme data %s is not valid', var_export($theme_data, true)));
        }

        if (!$this->themeValidator->isValid($theme)) {
            $this->filesystem->remove($sandboxPath);
            $this->themeValidator->getErrors($theme->getName());
            throw new Exception(sprintf('Theme configuration file is not valid - %s', var_export($this->themeValidator->getErrors($theme->getName()), true)));
        }

        $module_root_dir = PROX_DIR_MODULE;

        $themePath = PROX_DIR_THEMES . $theme->getName();
        if ($this->filesystem->exists($themePath)) {
            throw new Exception('There is already a theme named ' . $theme->getName() . ' in your themes/ folder. Remove it if you want to continue.');
        }

        $this->filesystem->mkdir($themePath);
        $this->filesystem->mirror($sandboxPath, $themePath);

        $this->filesystem->remove($sandboxPath);
    }

    /**
     * @param Theme $theme
     */
    public function saveTheme($theme)
    {
        $jsonConfigFolder = PROX_DIR_THEMES . $theme->getName();
        if (!$this->filesystem->exists($jsonConfigFolder) && !is_dir($jsonConfigFolder)) {
            mkdir($jsonConfigFolder, ProximFileSystem::DEFAULT_MODE_FOLDER, true);
        }

        file_put_contents(
            $jsonConfigFolder . '/theme.json',
            json_encode($theme->get(null))
        );
    }
}