
var {
    api,
    site_title,
    api_path,
    uploads_path,
    secret,
    notifications_sound,
    static_path,
    theme_uri,
    eventEmitter
} = proxim;

api['data/upload'] = api_path + "/data/upload";
api['data/load'] = api_path + "/data/load";
api['user/picture'] = api_path + "/user/remove-picture";

function formatFileSize(bytes) {
    if (typeof bytes !== 'number') {
        return 'processing...';
    }
    if (bytes >= 1000000000) {
        return (bytes / 1000000000).toFixed(2) + ' GB';
    }
    if (bytes >= 1000000) {
        return (bytes / 1000000).toFixed(2) + ' MB';
    }
    return (bytes / 1000).toFixed(2) + ' KB';
}

function uploadFilesTransport(data) {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        // Add any event handlers here...
        xhr.open('POST', `${api_path}/files/upload`, true);

        xhr.onload = function () {
            if (this.status >= 200 && this.status < 300) {
                resolve(xhr.response);
            } else {
                reject({
                    status: this.status,
                    statusText: xhr.statusText
                });
            }
        };

        xhr.setRequestHeader('Cache-Control',
            'max-age=0');
        xhr.setRequestHeader('If-Modified-Since',
            'Sat, 01 Jan 2000 00:00:00 GMT');
        xhr.setRequestHeader('Accept',
            'application/json');

        xhr.addEventListener("progress",
            progress => uploadProgress({
                progress: progress,
                instanceId: data.get('instanceId')
            }));
        xhr.upload.addEventListener("progress",
            progress => uploadProgress({
                progress: progress,
                instanceId: data.get('instanceId')
            }));

        xhr.send(data);
    })
}

/** File Upload Progress */
function uploadProgress({ progress, instanceId }) {
    const { total, loaded } = progress;
    const percent = Math.min(100, (loaded / total) * 100) || 0;

    const formFile = $(`[data-id='${instanceId}']`);
    formFile.find(".FormFile__upload-progress-bar").width(`${percent}%`);
}

function intlTelInputgetNumber(e) {
    return e.intlTelInput("getNumber");
}

// initialize the modal (plugins)
function initialize_modal() {
    // run uploader 
    initialize_uploader();

    initialize_dropzone();

    initialize_stepper();
}

function initialize_stepper() {
    $(".incr-btn").unbind('click').click(function (e) {
        e.preventDefault();
        var $button = $(this);
        var oldValue = $button.parent().find('.quantity').val();
        
        if (!oldValue.length) {
            oldValue = 0;
        }

        $button.parent().find('.incr-btn[data-action="decrease"]').removeClass('inactive');
        if ($button.data('action') == "increase") {
            var newVal = parseFloat(oldValue) + 1;
        } else {
            // Don't allow decrementing below 1
            if (oldValue > 1) {
                var newVal = parseFloat(oldValue) - 1;
            } else {
                newVal = $button.parent().find('.quantity').data('minvalue');
                $button.addClass('inactive');
            }
        }
        $button.parent().find('.quantity').val(newVal);
        e.preventDefault();
    });
}

function initialize_dropzone() {
    var dropArea = document.getElementsByClassName('js_dropzone');

    if(dropArea.length) {
        dropArea = dropArea[0];
        dropArea.addEventListener('dragenter', dropzoneHandler, false);
        dropArea.addEventListener('dragleave', dropzoneHandler, false);
        dropArea.addEventListener('dragover', dropzoneHandler, false);
        dropArea.addEventListener('drop', dropzoneHandler, false);
    }
    
}

function dropzoneHandler(e) {
    e.preventDefault();
    const files = Array.from(e.dataTransfer ? e.dataTransfer.files : e.target.files);

    const files_control = $(this);
    const publisher = files_control.parents('.js_files-uploader');
    const uploads_area = files_control.find('.files-control--files');
    const dragEnteredElements = [];
    const fileElement = files_control.find('[type="file"]');
    const handle = fileElement.data('handle') || 'order';

    Promise.all(
        files.map(blob => new Promise((resolve, reject) => {
            const instanceId = Math.round(Math.random() * Number.MAX_SAFE_INTEGER);
            const file = {
                is_order: handle == 'order' ? true : false,
                filename: blob.name,
                size: formatFileSize(blob.size),
                extension: blob.name.match(/[a-z]+$/),
                id: instanceId,
                isUploaded: false
            };

            this.progressTotal = 0;
            this.progressLoaded = 0;

            uploads_area.append(render_template("#form-file", file));

            const data = new window.FormData();
            
            data.append('files[]', blob);
            data.append('handle', handle);
            data.append('instanceId', instanceId);

            if(fileElement.data('id') !== undefined) {
                data.append('id', fileElement.data('id'));
            }

            var formFile = $(`[data-id='${instanceId}']`);
            var error = formFile.find('.alert.alert-danger');

            uploadFilesTransport(data)
                .then((response) => {
                    response = JSON.parse(response);
                    formFile.find("[name='uploadTypeId']").attr('disabled', false);
                    formFile.find('.FormFile__remove-button').attr('disabled', false);
                    formFile.find(".FormFile__upload-progress").remove();

                    /* handle response */
                    if (response.error) {
                        error.html(response.message).slideDown();
                    } else {
                        const uploadedFile = response.file;
                        formFile.attr('data-id', uploadedFile.id);
                        formFile.data('id', uploadedFile.id);

                        if (!publisher.data('files')) {
                            publisher.data('files', {});
                        }

                        let files = publisher.data('files');
                        files[uploadedFile.id] = uploadedFile;
                        publisher.data('files', files);

                        if (Object.keys(files).length > 0) {
                            publisher.find('button[type="submit"]').attr('disabled', false);
                        }
                    }
                })
        })
            .catch(() => { })
        ));
}

// initialize uploader
function initialize_uploader() {
    $('.js_x-uploader').each(function (index) {
        /* return if the plugin already running  */
        if ($(this).parents('form.x-uploader').length > 0) {
            return;
        }

        var data = {};
        data['url'] = api['data/upload'];
        data['secret'] = secret;

        if($(this).data('multiple') !== undefined) {
            data['multiple'] = true;
        }
        $(this).before(render_template("#x-uploader", data));
        $(this).prev().append($(this));
    });

    $(document).on('change', '.FormFile__upload-type-control [name="uploadTypeId"]', function (e) {
        e.preventDefault();
        const _this = $(this);
        const uploadTypeId = _this.val();
        const formFile = _this.closest('.FormFile');
        const fileId = formFile.data('id');
        const publisher = $('.js_files-uploader');
        if (!publisher.data('files')) {
            publisher.data('files', {});
        }

        let files = publisher.data('files');

        if (Object.keys(files).length == 0) {
            return;
        }

        files[fileId].uploadTypeId = uploadTypeId;
        publisher.data('files', files);
    });

    $(document).on('click', '.FormFile__remove-button', function (e) {
        e.preventDefault();
        const _this = $(this);
        const formFile = _this.closest('.FormFile');
        const fileId = formFile.data('id');
        var handle = formFile.data('handle');
        var isUploaded = formFile.data('uploaded');
        const publisher = $('.js_files-uploader');

        if (!publisher.data('files')) {
            publisher.data('files', {});
        }
        let files = publisher.data('files');

        if (files[fileId] !== undefined) {
            delete files[fileId];
        }

        if( isUploaded ) {
            confirm('Delete', 'Are you sure you want to delete this?', function() {
                $.post(api_path+'/admin/delete', {'handle': handle, 'id': fileId}, function(response) {
                    formFile.remove();
                    /* check the response */
                    if(response.callback) {
                        eval(response.callback);
                    } else {
                        $('#modal').modal('hide');
                    }
                }, 'json')
                .fail(function() {
                    modal('#modal-message', {title: 'Error', message: 'There is something that went wrong!'});
                });
            });
        } else {
            formFile.remove();
        }

        publisher.data('files', files);
        if ( files !== undefined ) {
            if(Object.keys(files).length == 0) {
                publisher.removeData('files');
            }
        }
    });
}

// browser notification
function browser_notification(icon, body, url, tag) {
    /* check if the browser supports notifications */
    if (!("Notification" in window)) {
        return;
    }
    /* check whether notification permissions have alredy been granted */
    if (Notification.permission !== "granted") {
        /* request permission */
        Notification.requestPermission();
    } else {
        /* send notification */
        var notification = new Notification(site_title, {
            icon: icon,
            body: body,
            tag: tag
        });
        notification.onclick = function () {
            window.open(url);
            notification.close();
        };
    }
}

// notification highlighter
function notification_highlighter() {
    try {
        var search_params = new URLSearchParams(window.location.search);
        var notify_id = search_params.get("notify_id");
    } catch (err) {
        var notify_id = getParameterByName("notify_id");
    }
    if (notify_id) {
        var _elem = $('#' + notify_id);
        if (_elem.length > 0) {
            $('html, body').animate({
                scrollTop: _elem.offset().top
            }, 1000);
            _elem.find('.js_notifier-flasher:first').addClass("x-notifier");
            setTimeout(function () {
                _elem.find('.js_notifier-flasher:first').removeClass("x-notifier");
            }, '2500');
        }
    }
}

// data heartbeat
function data_heartbeat() {
    var data = {};
    data['last_message'] = $(".js_live-messages").find(".js_scroller li:first").data('id') || 0;
    data['last_notification'] = $(".js_live-notifications").find(".js_scroller li:first").data('id') || 0;

    if($(".js_live-messenger").length) {
        data['last_messenger'] = $(".js_live-messenger").find(".js_scroller li:first").data('last-message') || 0;
    }

    $.post( `${api_path}/data/live`, data, function (response) {
        if (response.callback) {
            eval(response.callback);
        } else {
            if (response.messages) {
                if ($(".js_live-messages").find(".js_scroller ul").length > 0) {
                    $(".js_live-messages").find(".js_scroller ul").prepend(response.messages);
                } else {
                    $(".js_live-messages").find(".js_scroller").html("<ul>" + response.messages + "</ul>");
                }

                var messages = parseInt($(".js_live-messages").find("span.label").text()) + response.messages_count;
                $(".js_live-messages").find("span.label").text(messages).removeClass("hidden");
                $("#chat_sound")[0].play();
            }
            
            if (response.notifications) {
                $.each(response.notifications_json, function (index, element) {
                    /* send browser notifications */
                    browser_notification(element.icon, element.message, element.url, element.notification_id);
                });
                
                if ($(".js_live-notifications").find(".js_scroller ul").length > 0) {
                    $(".js_live-notifications").find(".js_scroller ul").prepend(response.notifications);
                } else {
                    $(".js_live-notifications").find(".js_scroller").html("<ul>" + response.notifications + "</ul>");
                }

                var notifications = parseInt($(".js_live-notifications").find("span.label").text()) + response.notifications_count;
                $(".js_live-notifications").find("span.label").text(notifications).removeClass("hidden");
                if (notifications_sound && $("#notification_sound").length ) {
                    $("#notification_sound")[0].play();
                }
            }
 
            eventEmitter.emit('actionAfterDataHeartBeat', response );

            setTimeout('data_heartbeat();', 8000);
        }
    }, 'json');
}

// load more
function load_more(element) {
    if (element.hasClass('done') || element.hasClass('loading')) return;
    var _this = element;
    var loading = _this.find('.loader');
    var text = _this.find('span');
    var remove = _this.data('remove') || false;
    var stream = _this.parent().find('ul:first');
    /* prepare data object */
    var data = {};
    data['get'] = _this.data('get');
    if (_this.data('filter') !== undefined) {
        data['filter'] = _this.data('filter');
    }
    if (_this.data('type') !== undefined) {
        data['type'] = _this.data('type');
    }
    if (_this.data('uid') !== undefined) {
        data['uid'] = _this.data('uid');
    }
    if (_this.data('id') !== undefined) {
        data['id'] = _this.data('id');
    }
    data['offset'] = _this.data('offset') || 1; /* we start from iteration 1 because 0 already loaded */
    /* show loader & hide text */
    _this.addClass('loading');
    text.hide();
    loading.removeClass('x-hidden');
    /* get & load data */
    $.post(api['data/load'], data, function (response) {
        _this.removeClass('loading');
        text.show();
        loading.addClass('x-hidden');
        /* check the response */
        if (response.callback) {
            eval(response.callback);
        } else {
            if (response.data) {
                data['offset']++;
                if (response.append) {
                    stream.append(response.data);
                } else {
                    stream.prepend(response.data);
                }
            } else {
                if (remove) {
                    _this.remove();
                } else {
                    _this.addClass('done');
                    text.text('There is no more data to show');
                }
            }
        }
        _this.data('offset', data['offset']);
    }, 'json')
        .fail(function () {
            _this.removeClass('loading');
            text.show();
            loading.addClass('x-hidden');
            modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
        });
}

$(function () {

    proxim.eventEmitter.on('initializeApp', function() {

        // run colorpicker plugin
        $('.js_colorpicker').colorpicker({
            format: 'hex'
        });
        
        // run metisMenu
        $(".js_metisMenu").length && $(".js_metisMenu").metisMenu();

        // run bootstrap tooltip
        $('[data-toggle="tooltip"]').length && $('[data-toggle="tooltip"]').tooltip();

        $('[data-toggle="popover"]').length && $('[data-toggle="popover"]').popover();

        // run autosize (expand textarea) plugin
        autosize($('.js_autosize'));

        $('.select_search').each(function (i, elm) {
            if (!$(elm).data('select2')) {
                $(elm).select2();
            }
        });

        // run slimScroll plugin
        $('.js_scroller').each(function(){
            var _this = $(this);
            /* return if the plugin already running  */
            if(_this.parent('.slimScrollDiv').length > 0) {
                return;
            }
            /* run if not */
            _this.slimScroll({
                height: _this.attr('data-slimScroll-height') || '280px',
                start: _this.attr('data-slimScroll-start') || 'top',
                distance: '2px'
            })
        });

        // init datetimepicker plugin
        if($(".js_datetimepicker").length > 0) {
            $('.js_datetimepicker').datetimepicker({
                format: "DD-MM-YYYY HH:mm",
                locale: $('html').attr('lang').split("_", 1).toString() || 'en'
            });
        }

        // Timezone handler
        $(document).on("change", "#timezone-region", function(){
            var value = $(this).val(),
                $timezone_combo = $("#timezone-combo-"+value);
            $timezone_combo.find("option:first").prop("selected", true);
            $("[data-content=timezone]").val($timezone_combo.val()).change();
        });

        $(document).on("change", "[id^=timezone-combo-]", function(){
            var value = $(this).val();
            $("[data-content=timezone]").val(value).change();
        });

        // Co-combo breaker
        $(document).on("change", "select[data-combo]", function(){
            var $combo = $("#"+$(this).data("combo"));

            if($combo.length) {
                $combo.children(".switch-combo").hide();
            }
            var $combo_container = $("#" + $(this).closest("select").data("combo")),
                $combo_target = $("[data-combo-value~=" + $("option:selected", this).attr("value") + "]", $combo_container);

            if($combo_target.length){
                $combo_target
                    .show()
                    .find("[data-required]")
                    .each(function() {
                        $(this).attr("required", "required"); // re-enable any disabled required
                    });
            }

            $('.select_search').each(function (i, elm) {
                if (!$(elm).data('select2')) {
                    $(elm).select2();
                }
            });

            // Disable [required] in hidden combos
            $(".switch-combo", $combo_container).each(function() {
                if($(this).is(":visible")) return;
                $("[required]", this).attr("data-required", true).removeAttr("required");
            });
        });

        // Co-combo breaker
        $(document).on("change", "input[data-combo]", function(){
            var $combo = $("#"+$(this).data("combo"));

            if($combo.length) {
                $combo.children(".switch-combo").hide();
            }

            var $combo_container = $("#" + $(this).data("combo")),
                $combo_target = $combo_container.find("[data-combo-value~=" + $(this).val() + "]").first();

            if($combo_target.length){
                $combo_target
                    .show()
                    .find("[data-required]")
                    .each(function() {
                        $(this).attr("required", "required"); // re-enable any disabled required
                    });
            }

            $('.select_search').each(function (i, elm) {
                if (!$(elm).data('select2')) {
                    $(elm).select2();
                }
            });

            // Disable [required] in hidden combos
            $(".switch-combo", $combo_container).each(function() {
                if($(this).is(":visible")) return;
                $("[required]", this).attr("data-required", true).removeAttr("required");
            });
        });

        // run moment plugin
        $(".js_moment").each(function(){
            var _this = $(this);
            var time = _this.data('time');

            var time = new Date( time );
            if ( Number.isNaN(Date.parse(time)) ) {
                _this.text('N/A');
                return;
            }

            _this.text( dateFormat(time, 'mmm d, yyyy \'at\' h:MM TT') );
        });

        // init fastlink plugin
        FastClick.attach(document.body);

        // run Sngine audio files
        $("audio.js_audio").on("play", function() {
            $("audio").not(this).each(function(index, audio) {
                audio.pause();
            });
        });

        if ($('.js_phone').length > 0) {
            $(".js_phone").intlTelInput({
                // allowDropdown: false,
                autoHideDialCode: false,
                autoPlaceholder: "off",
                // dropdownContainer: "body",
                // excludeCountries: ["us"],
                formatOnDisplay: false,
                initialCountry: "auto",
                geoIpLookup: function(callback) {
                    $.get('https://ipinfo.io', function() {}, "jsonp").always(function(resp) {
                        var countryCode = (resp && resp.country) ? resp.country : "";
                        callback(countryCode);
                    });
                },
                // hiddenInput: "full_number", // initialCountry: "auto",
                nationalMode: true,
                // onlyCountries: ['ke', 'ug', 'tz'],
                // placeholderNumberType: "MOBILE",
                preferredCountries: ['ke', 'us'],
                separateDialCode: true,
                utilsScript: `${theme_uri}/js/plugins/intl.telinput/utils.js`
            });
    
            $(document).on('input', '.js_phone', function(e){
                var _this = $(this);
                var phone_input = $('.js_phone_number');
                phone_input.val(intlTelInputgetNumber(_this));
            }); 
        
            $(document).on('click', '.intl-tel-input', function(e){
                var _this = $('.js_phone');
                var p_input = $('.js_phone_number');
                p_input.val(intlTelInputgetNumber(_this));
            });
    
        } 

        // init bootstrap selectpicker plugin
        $('.selectpicker').selectpicker({
            style: 'btn-option'
        });

        /*Summernote editor*/
        if ($(".js_summernote").length) {
            $(".js_summernote").each(function () {
                var $height = $(this).data('height') || 200;
                $(this).summernote({
                    height: $height,
                    tabsize: 2,
                    toolbar: [
                        ['style', ['bold', 'italic', 'underline', 'clear']],
                        ['para', ['ul', 'ol', 'paragraph']],
                        ['table', ['table']],
                        ['insert', ['link']],
                    ]
                });
            });
        }

        /* Ckeditor*/
        if ($("#ckeditor").length) {
            $ckeditor = document.querySelector( '#ckeditor' );
            var $height = $ckeditor.getAttribute('height') || 200;

            ClassicEditor
                .create( $ckeditor, {
                    toolbar: [ 'heading', '|', 'bold', 'italic', 'link', 'bulletedList', 'numberedList', 'indent', 'outdent', 'insertTable', 'undo', 'redo' ]
                }).then( editor => {
                    // Set custom height via jQuery by appending a scoped style
                    $(`<style type="text/css" scoped>.ck-editor .ck-editor__editable_inline {min-height: ${$height}px !important;}</style>`).insertAfter($ckeditor);
            
                });
        }

        // run bootstrap selectpicker plugin
        $('.selectpicker').selectpicker({
            style: 'btn-option'
        });

        // init datetimepicker plugin
        if ($(".js_datetimepicker").length > 0) {
            $('.js_datetimepicker').datetimepicker({
                format: "DD-MM-YYYY HH:mm",
                locale: $('html').attr('lang').split("_", 1).toString() || 'en'
            });
        } 

        // run autosize (expand textarea) plugin
        autosize($('.js_autosize'));

        $(document).on('propertychange input', '.force-integer', function (e) {
            var $input = $(this);
            $input.val($input.val().replace(/[^\d\.]+/g,''));
        });

    });

    // init browser notifications
    if (("Notification" in window)) {
        if (Notification.permission !== "granted") {
            Notification.requestPermission();
        }
    }

    // run notification highlighter
    notification_highlighter();

    // run data heartbeat
    data_heartbeat();

    // run the stepper
    initialize_stepper();

    // run x-uploader
    /* initialize the uplodaer */
    initialize_uploader();
    $(document).ajaxComplete(function () {
        initialize_uploader();
    });

    // init offcanvas-sidebar
    $('[data-toggle=offcanvas]').click(function() {
        $('.offcanvas').toggleClass('active');
    });

    // run autocomplete tags
    /* focus the input */
    $(document).on('click', '.js_autocomplete-tags', function() {
        var input = $(this).find('input').focus();
    });
    /* show and get the results if any */
    $(document).on('keyup', '.js_autocomplete-tags input', function() {
        var _this = $(this);
        var query = _this.val();
        var type = _this.data('type');
        var parent = _this.parents('.js_autocomplete-tags');
        /* change the width of typehead input */
        prev_length = _this.data('length') || 0;
        new_length = query.length;
        if(new_length > prev_length && _this.width() < 250) {
            _this.width(_this.width()+6);
        } else if(new_length < prev_length) {
            _this.width(_this.width()-6);
        }
        _this.data('length', query.length);
        /* check maximum number of tags */
        if(parent.find('ul.tags li').length > 9) {
            return;
        }
        /* check the query string */
        if(query != '') {
            /* check if results dropdown-menu not exist */
            if(_this.next('.dropdown-menu').length == 0) {
                /* construct a new one */
                var offset = _this.offset();
                var posX = offset.left - $(window).scrollLeft();
                if($(window).width() - posX < 180) {
                    _this.after('<div class="dropdown-menu auto-complete tl"></div>');
                } else {
                    _this.after('<div class="dropdown-menu auto-complete"></div>');
                }
            }
            /* get skipped ids */
            var skipped_ids = [];
            $.each(parent.find('ul.tags li'), function(i,tag) {
                skipped_ids.push($(tag).data('uid'));
            });

            $.post(`${api_path}/data/autocomplete`, {'type': type, 'query': query, 'skipped_ids': JSON.stringify(skipped_ids)}, function(response) {
                if(response.callback) {
                    eval(response.callback);
                } else if(response.autocomplete) {
                    _this.next('.dropdown-menu').show().html(response.autocomplete);
                }
            }, 'json');
        } else {
            /* check if results dropdown-menu already exist */
            if(_this.next('.dropdown-menu').length > 0) {
                _this.next('.dropdown-menu').hide();
            }
        }
    });
    /* show previous results dropdown-menu when the input is clicked */
    $(document).on('click focus', '.js_autocomplete-tags input', function() {
        /* check maximum number of tags */
        if($(this).parents('.js_autocomplete-tags').find('ul.tags li').length > 9) {
            return;
        }
        /* only show again if the input & dropdown-menu are not empty */
        if($(this).val() != '' && $(this).next('.dropdown-menu').find('li').length > 0) {
            $(this).next('.dropdown-menu').show();
        }
    });
    /* hide the results dropdown-menu when clicked outside the input */
    $(document).on('click', function(e) {
        if(!$(e.target).is(".js_autocomplete-tags")) {
            $('.js_autocomplete-tags .dropdown-menu').hide();
        }
    });
    /* add a tag */
    $(document).on('click', '.js_tag-add', function() {
        var uid = $(this).data('uid');
        var name = $(this).data('name');
        var parent = $(this).parents('.js_autocomplete-tags');
        var tag = '<li data-uid="'+uid+'">'+name+'<button type="button" class="close js_tag-remove" title="Remove"><span>&times;</span></button></li>'
        parent.find('.tags').append(tag);
        parent.find('input').val('').focus();
        /* check if there is chat-form next to js_autocomplete-tags */
        if(parent.siblings('.chat-form').length > 0) {
            if(parent.find('ul.tags li').length == 0) {
                if(!parent.siblings('.chat-form').hasClass('invisible')) {
                    parent.siblings('.chat-form').addClass('invisible');
                }
            } else {
                parent.siblings('.chat-form').removeClass('invisible');
            }
        }
    });
    /* remove a tag */
    $(document).on('click', '.js_tag-remove', function() {
        var tag = $(this).parents('li');
        var parent = $(this).parents('.js_autocomplete-tags');
        tag.remove();
        /* check if there is chat-form next to js_autocomplete-tags */
        if(parent.siblings('.chat-form').length > 0) {
            if(parent.find('ul.tags li').length == 0) {
                if(!parent.siblings('.chat-form').hasClass('invisible')) {
                    parent.siblings('.chat-form').addClass('invisible');
                }
            } else {
                parent.siblings('.chat-form').removeClass('invisible');
            }
        }
        return false;
    });

    /* stop propagation */
    $(document).on('click', '.x-uploader', function (e) {
        /* get type */
        var type = $(this).find('.js_x-uploader').data('type') || "photos";
        if (type == "photos") {
            e.stopPropagation();
        }
    });

    /* initialize uploading */
    $(document).on('change', '.x-uploader input[type="file"]', function () {
        $(this).parent('.x-uploader').submit();
    });

    /* uploading */
    $(document).on('submit', '.x-uploader', function (e) {
        var _this = $(this);
        e.preventDefault;
        /* initialize AJAX options */
        var options = {
            dataType: "json",
            uploadProgress: _handle_progress,
            success: _handle_success,
            error: _handle_error,
            resetForm: true
        };
        options['data'] = {};
        /* get uploader input */
        var uploader = $(this).find('input[type="file"]');
        /* get type */
        var type = $(this).find('.js_x-uploader').data('type') || "photos";
        options['data']['type'] = type;
        /* get handle */
        var handle = $(this).find('.js_x-uploader').data('handle');
        if (handle === undefined) {
            return false;
        }
        options['data']['handle'] = handle;
        /* get multiple */
        if($(this).find('.js_x-uploader').data('multiple') !== undefined) {
            options['data']['multiple'] = true;
        }
        /* get id */
        var id = $(this).find('.js_x-uploader').data('id');
        if (id !== undefined) {
            options['data']['id'] = id;
        }
        /* check type */
        if (type == "photos") {
            /* check handle */
            if (handle == "picture-user") {
                var loader = $('.profile-avatar-change-loader');
                loader.show();

            } else if (handle == "x-image") {
                var parent = $(this).parents('.x-image');
                var loader = parent.find('.loader');
                loader.show();
            } else if(handle == "publisher" || handle == "prepaper-preview") {
                var publisher = $('.publisher');
                var files_num = uploader.get(0).files.length;
                
                /* check if there is already uploading process */
                if(!publisher.data('photos')) {
                    publisher.data('photos', {});
                }

                var attachments = publisher.find('.publisher-attachments');
                var loader = $('<ul></ul>').appendTo(attachments);
                attachments.show();
                for (var i = 0; i < files_num; ++i) {
                    $('<li class="loading"><div class="loader loader_small"></div></li>').appendTo(loader).show();
                }

            }
        } else if (type == "video" || type == "audio" || type == "file") {
            /* check handle */
            if (handle == "publisher" || handle == "prepaper-preview") {
                /* show upload loader */
                var publisher = $('.publisher');
                /* check if there is current (uploading|scrabing|video|audio) process */
                if (publisher.data('photos') || publisher.data('scrabing') || publisher.data('video') || publisher.data('audio') || publisher.data('file')) {
                    return false;
                }
                publisher.data(type, {});
                var attachments = $('.publisher-attachments');
                var loader = $('<ul></ul>').appendTo(attachments);
                attachments.show();
                $('<li class="loading"><div class="loader loader_small"></div></li>').appendTo(loader).show();
            }
        }

        eventEmitter.emit('actionBeforeXUploader', {
            '_this': _this,
            'type': type,
            'handle': handle
        });

        /* handle progress */
        function _handle_progress(e) {
            /* disable uploader input during uploading */
            uploader.prop('disabled', true);
            /* get progress percent */
            var percent = parseInt((e.loaded / e.total * 100));
            /* add percent to loader progress bar */
            if(loader) loader.find('.progress-bar').css('width', percent+'%').attr('aria-valuenow', percent);
        }

        /* handle success */
        function _handle_success(response) {
            /* enable uploader input */
            uploader.prop('disabled', false);
            /* hide upload loader */
            if(loader) loader.hide();
            
            /* handle the response */
            if (response.callback) {
                eval(response.callback);
            } else {
                /* check type */
                if (type == "photos") {
                    /* check the handle */
                    if (handle == "picture-user") {
                        /* update (user|page|group) picture */
                        var image_path = uploads_path + '/' + response.file;
                        $('.profile-avatar-wrapper .avatar').css("background-image", 'url(' + image_path + ')');

                    } else if (handle == "x-image") {
                        /* update x-image picture */
                        var image_path = uploads_path + '/' + response.file;
                        parent.css("background-image", 'url(' + image_path + ')');
                        /* add the image to input */
                        parent.find('.js_x-image-input').val(response.file);
                        /* show the remover */
                        parent.find('button').show();
                    } else if(handle == "publisher" || handle == "prepaper-preview") {
                        /* remove upload loader */
                        if(loader) loader.remove();
                        /* add the attachment to publisher data */
                        var files = publisher.data('photos');
                        for(var i in response.files) {
                            file = response.files[i];
                            files[file.source] = file;
                            /* add publisher-attachments */
                            var image_path = uploads_path+'/'+file.source;
                            attachments.find('ul').append(render_template("#publisher-attachments-item", {'id': file.id, 'image_path':image_path, 'handle' : handle}));
                        }

                        publisher.data('photos', files);
                    }
                }
            }

            eventEmitter.emit('actionAfterXUploader', {
                '_this': _this,
                'type': type,
                'handle': handle,
                'response': response 
            });
        }

        /* handle error */
        function _handle_error() {
            /* enable uploader input */
            uploader.prop('disabled', false);
            /* hide upload loader */
            if (loader) loader.hide();
            /* check the handle */
            if (handle == "publisher" || handle == "prepaper-preview") {
                /* hide the attachment from publisher */
                if ((type == "photos" && jQuery.isEmptyObject(publisher.data('photos'))) || type != "photos") {
                    attachments.hide();
                }
                /* remove upload loader */
                if (loader) loader.remove();
            }
            modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
        }
        /* submit the form */
        $(this).ajaxSubmit(options);
        return false;
    });

    /* publisher attachment remover */
    $(document).on('click', '.js_publisher-attachment-remover', function() {
        var item = $(this).parents('li.item');
        var id = item.data('id');
        var handle = item.data('handle');

        confirm('Delete', 'Are you sure you want to delete this?', function() {
            $.post(api_path+'/admin/delete', {'handle': handle, 'id': id}, function(response) {
                /* remove the attachment item */
                item.remove();
                $('#modal').modal('hide');
            }, 'json')
            .fail(function() {
                modal('#modal-message', {title: 'Error', message: 'There is something that went wrong!'});
            });
        });
        
    });

    /* handle profile (picture) remover */
    $(document).on('click', '.js_delete-picture', function (e) {
        e.stopPropagation();
        var id = $(this).data('id');
        var handle = $(this).data('handle');
        var remove = ($(this).hasClass('js_delete-cover')) ? 'cover' : 'picture';

        var wrapper = $('.profile-avatar-wrapper');

        confirm('Delete Picture', 'Are you sure you want to remove your profile picture?', function () {
            $.post(api['user/picture'], { 'handle': handle, 'id': id }, function (response) {
                /* check the response */
                if (response.callback) {
                    eval(response.callback);
                } else {
                    /* hide delete btn  */
                    wrapper.find('.profile-avatar-delete').hide();
                    /* update (user) picture with default picture */
                    wrapper.find('img').css("background-image", 'url(' + response.file + ')');
                    $('#modal').modal('hide');
                }
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    /* handle x-image remover */
    $(document).on('click', '.js_x-image-remover', function () {
        var _this = $(this);
        var parent = _this.parents('.x-image');
        confirm('Delete', 'Are you sure you want to delete this?', function () {
            /* remove x-image image */
            parent.attr('style', '');
            /* add the image to input */
            parent.find('.js_x-image-input').val('');
            /* hide the remover */
            _this.hide();
            /* hide the confimation */
            $('#modal').modal('hide');
        });
    });

    // run load-more
    /* load more data by click */
    $(document).on('click', '.js_see-more', function () {
        load_more($(this));
    });
    
    /* load more data by scroll */
    $('.js_see-more-infinite').bind('inview', function (event, visible) {
        if (visible == true) {
            load_more($(this));
        }
    });

    // handle data reseter
    $(document).on('show.bs.dropdown', '.js_live-messages, .js_live-notifications', function (e) {
        var _this = $(this);
        var counter = parseInt(_this.find("span.label").text()) || 0;
        if (!$(this).hasClass('open') && counter > 0) {
            /* reset the client counter & hide it */
            _this.find("span.label").addClass('hidden').text('0');
            /* get the reset target */
            if (_this.hasClass('js_live-messages')) {
                var data = { 'reset': 'messages' };
            } else if (_this.hasClass('js_live-notifications')) {
                var data = { 'reset': 'notifications' };
            }
            /* reset the server counter */
            $.post(api_path + '/data/reset', data, function (response) {
                /* check the response */
                if (!response) return;
                /* check if there is a callback */
                if (response.callback) {
                    eval(response.callback);
                }
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        }
    });

    // handle notifications sound
    $(document).on('click', '.js_notifications-sound-toggle', function () {
        notifications_sound = $(this).is(":checked");
        $.post(api_path + '/user/settings?edit=notifications_sound', { 'notifications_sound': (notifications_sound) ? 1 : 0 }, function (response) {
            /* check the response */
            if (!response) return;
            /* check if there is a callback */
            if (response.callback) {
                eval(response.callback);
            }
        }, 'json')
            .fail(function () {
                modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
            });
    });

    /* assign user */
    $(document).on('click', '.js_assign-order', function (e) {
        e.preventDefault();
        var uid = $(this).data('uid');
        var oid = $(this).data('oid');
        var handle = $(this).data('handle');

        confirm('Assign Order', 'Are you sure you want to assign this order to U' + uid + '?', function () {
            $.post(api_path + '/orders/' + oid + '/assign', { 'do': 'assign', 'handle': handle, 'uid': uid }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    /* complete order */
    $(document).on('click', '.js_complete-order', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');

        confirm('Mark as Complete', 'If all the requirements of this order are met, please, proceed with submitting your paper.', function () {
            $.post(api_path + '/orders/' + oid + '/complete', { 'do': 'complete' }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    /* deliver order */
    $(document).on('click', '.js_deliver-order', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');

        confirm('Deliver order to Customer', 'Are you sure you want to send this paper to the customer?', function () {
            $.post(api_path + '/orders/' + oid + '/deliver', { 'do': 'deliver' }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    /* take order */
    $(document).on('click', '.js_take-order', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');

        confirm('Take order', 'Are you sure you want to take and work on this order?', function () {
            $.post(api_path + '/orders/' + oid + '/take', { 'do': 'take' }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    // place bid
    $(document).on('click', '.js_place-bid', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');

        confirm('Apply for this order', 'Are you sure you want to apply for this order?', function () {
            $.post(api_path + '/orders/' + oid + '/apply', { 'do': 'apply' }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    // remove bid
    $(document).on('click', '.js_remove-bid', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');
        var bid = $(this).data('bid');

        confirm('Delete my Bid', 'Are you sure you want to delete your bid for this order?', function () {
            $.post(api_path + '/orders/' + oid + '/remove_bid', { 'do': 'remove_bid', 'bid_id': bid }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    // confirm order
    $(document).on('click', '.js_confirm-order', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid'),
            action = $(this).data('action');

        let title = description = "";

        if (action == "accept") {
            title = 'Accept Order Request';
            description = 'Support team assigned you this order. Are you sure you want to confirm this order request?';
        } else {
            title = 'Reject Order Request';
            description = 'Support team assigned you this order. Are you sure you want to decline this order request?';
        }

        confirm(title, description, function () {
            $.post(api_path + '/orders/' + oid + '/confirm', { 'do': action }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    // request payment
    $(document).on('click', '.js_request-payment', function (e) {
        e.preventDefault();

        confirm('Request Payment', 'Are you sure you want to send this request?', function () {
            $.post(api_path + '/user/request-payment', function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    // show message
    $(document).on('click', '.js_order-message', function (e) {
        e.stopPropagation();
        var _this = $(this);
        var mid = _this.data('id');
        var is_read = _this.data('read');

        if (!is_read) {
            $.post(api_path + '/messages/' + mid + '/read', { 'message_id': mid }, function (response) {
                _this.removeClass('new');
            }, 'json');
        }
    });

    $(document).on('click', '.js_reorder', function (e) {
        e.preventDefault();
        var oid = $(this).data('oid');

        confirm('Re-order', 'Are you sure you want to re-order this order?', function () {
            $.post(api_path + '/orders/' + oid + '/reorder', { 'do': 'reorder' }, function (response) {
                if (response.callback) {
                    eval(response.callback);
                }

                window.location.reload;
            }, 'json')
                .fail(function () {
                    modal('#modal-message', { title: 'Error', message: 'There is something that went wrong!' });
                });
        });
    });

    $(document).on('click', '.files-control .browse-button', function (e) {
        e.preventDefault();
        $(document).find(document).removeClass('dragging-file');
        const files_control = $(this).closest('.files-control');
        const publisher = files_control.parents('.js_files-uploader');
        const uploads_area = files_control.find('.files-control--files');
        const dragEnteredElements = [];
        const fileElement = files_control.find('[type="file"]');
        const handle = fileElement.data('handle') || 'order';
        fileElement.click();

        fileElement.on('change', function (e) {
            const files = Array.from(e.dataTransfer ? e.dataTransfer.files : e.target.files);
            e.target.value = '';

            Promise.all(
                files.map(blob => new Promise((resolve, reject) => {
                    const instanceId = Math.round(Math.random() * Number.MAX_SAFE_INTEGER);
                    const file = {
                        is_order: handle == 'order' ? true : false,
                        filename: blob.name,
                        size: formatFileSize(blob.size),
                        extension: blob.name.match(/[a-z]+$/),
                        id: instanceId,
                        isUploaded: false
                    };

                    this.progressTotal = 0;
                    this.progressLoaded = 0;

                    uploads_area.append(render_template("#form-file", file));

                    const data = new window.FormData();
                    data.append('files[]', blob);
                    data.append('instanceId', instanceId);
                    data.append('handle', handle);

                    if(fileElement.data('id') !== undefined) {
                        data.append('id', fileElement.data('id'));
                    }
                    
                    var formFile = $(`[data-id='${instanceId}']`);
                    var error = formFile.find('.alert.alert-danger');

                    uploadFilesTransport(data)
                        .then((response) => {
                            response = JSON.parse(response);
                            if(handle == 'order') {
                                formFile.find("[name='uploadTypeId']").attr('disabled', false);
                            }

                            formFile.find('.FormFile__remove-button').attr('disabled', false);
                            formFile.find(".FormFile__upload-progress").remove();

                            /* handle response */
                            if (response.error) {
                                error.html(response.message).slideDown();
                            } else {
                                const uploadedFile = response.file;
                                formFile.attr('data-id', uploadedFile.id);
                                formFile.data('id', uploadedFile.id);

                                if (!publisher.data('files')) {
                                    publisher.data('files', {});
                                }

                                let files = publisher.data('files');
                                files[uploadedFile.id] = uploadedFile;
                                publisher.data('files', files);

                                if (Object.keys(files).length > 0) {
                                    publisher.find('button[type="submit"]').attr('disabled', false);
                                }
                            }
                        })
                })
                    .catch(() => { })
                ));
        });
    });

    $(document).on('submit', '.js_files-uploader', function (e) {
        e.preventDefault();
        var publisher = $(this);
        var error = publisher.find(".alert.alert-danger");
        var handle = publisher.data('handle') || 'order';
        var files = publisher.data('files');

        if (files === undefined || Object.keys(files).length == 0) {
            return;
        }

        var actionURL;

        if(handle == 'order') {
            var order_id = publisher.data('oid');
            var upload_for = publisher.find('[name="upload_for"]:checked').val() || "me";
            if (order_id === undefined) {
                return;
            }

            order_id = parseInt(order_id);
            actionURL = `${api_path}/orders/${order_id}/upload_files?upload_for=${upload_for}`;
        } else if( handle == 'plugin' ) {
            actionURL = `${api_path}/modules/upload`;
        } else {
            if(publisher.data('actionUrl') !== undefined) {
                var actionUrl = publisher.data('actionUrl');
                actionURL = `${api_path}/${actionUrl}`;
            }
        }

        for (var file in files) {
            var formFile = publisher.find(`[data-id='${files[file].id}']`);
            var fileError = formFile.find('.alert.alert-danger');

            if(handle == 'order') {
                if (!files[file].uploadTypeId) {
                    fileError.html("Select the file type").slideDown();
                    return;
                } else {
                    fileError.slideUp();
                }
            }
        }

        $.post(actionURL, { files: files }, 'json').then((response) => {
            /* handle response */
            if (response.error) {
                error.html(response.message).slideDown();
            } else {
                eval(response.callback);
            }
        }).fail((resp) => {
            error.html("There was a problem uploading the files. Please try again!").slideDown();
        });
    });

    $('body').on('click', '.js_session-deleter', function() {
        var id = $(this).data('id');
        confirm('Delete', 'Are you sure you want to delete this?', function() {
            $.post(`${api_path}/user/session`, {
                'handle': 'session',
                'id': id
            }, function(response) {
                if (response.callback) {
                    eval(response.callback);
                } else {
                    window.location.reload();
                }
            }, 'json').fail(function() {
                modal('#modal-message', {
                    title: 'Error',
                    message: 'There is something that went wrong!'
                });
            });
        });
    });

    /* delete all sessions */
    $(document).on('click', '.js_session-delete-all', function (event) {
        event.preventDefault();

        const data = {'handle': 'sessions'};
        if($(this).data('uid')) {
            data['uid'] = $(this).data('uid');
        } 

        confirm('Delete', 'Are you sure you want to delete this?', function() {
            $.post( `${api_path}/user/session`, data, function(response) {
                /* check the response */
                if(response.callback) {
                    eval(response.callback);
                } else {
                    window.location.reload();
                }
            }, 'json')
            .fail(function() {
                modal('#modal-message', {title: 'Error', message: 'There is something that went wrong!'});
            });
        });
    });

});