<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Currency;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class Wallet extends Module
{
    public function __construct()
    {
        $this->name = 'wallet';
        $this->icon = 'fa fa-wallet';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->tab = 'customer_profile';
        $this->displayName = 'Wallet';
        $this->description = 'Allow customers to checkout from their Wallet, using funds that they have deposited in their account.';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createWalletTables()) {
            return false;
        } 

        $this->registerHook('displayAdminEditCustomerNavbar') &&
        $this->registerHook('displayAdminEditCustomerNavContent');

        Configuration::updateValue('WALLET_ENABLED', true);
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
        
        Configuration::updateValue('WALLET_ENABLED', false);
    }

    public function getWalletByCustomerId( $customerId ) {
        $sql = new DbQuery();
        $sql->select('cw.*');
        $sql->from('customer', 'cw');
        $sql->where('cw.`customer_id` = ' . (int) $customerId );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);
        return $result;
   }

    public function createWalletTables()
    {
        try {
            $tablesUpdated = false;
            $result = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('wallet_transaction'));
            if(!isset($result['customer_id'])) {
                $tablesUpdated = true;
    
                if (!Db::getInstance()->Execute("ALTER TABLE " . Db::prefix('wallet_transaction') . " CHANGE wallet_id customer_id BIGINT(20);")) {
                    return false;
                }
            }
        } catch(\Exception $e) {

        }
    
        $result = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('customer'));
        if(!isset($result['wallet_balance'])) {
            $tablesUpdated = true;

            if (!Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('customer') . " ADD (
                    `wallet_balance` FLOAT(14,2) NOT NULL DEFAULT 0.00
                );
            ")) {
                return false;
            }
        }
    
        try {
            if($tablesUpdated) {
                $walletTransactions = Db::getInstance()->executeS("SELECT * FROM ". Db::prefix('wallet_transaction'));
    
                foreach($walletTransactions as $walletTransaction) {
                    $transactionWallet = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('customer_wallet') . " WHERE wallet_id = " . (int) $walletTransaction['customer_id']);
                    // update transaction customer_id
                    Db::getInstance()->update(
                        'wallet_transaction',
                        array(
                            'customer_id' => $transactionWallet['customer_id'],
                        ),
                        'wallet_transaction_id = ' . (int) $walletTransaction['wallet_transaction_id']
                    );
    
                    // update balance
                    Db::getInstance()->update(
                        'customer',
                        array(
                            'wallet_balance' => $transactionWallet['wallet_balance'],
                        ),
                        'customer_id = ' . (int) $transactionWallet['customer_id']
                    );
                }
            }
        } catch(\Exception $e) {

        }

        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('wallet_transaction') . " (
                `wallet_transaction_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `customer_id` BIGINT(20) UNSIGNED NOT NULL,
                `currency_id` BIGINT(20) UNSIGNED DEFAULT NULL,
                `transaction_id` VARCHAR(255) DEFAULT NULL,
                `source` VARCHAR(255) NOT NULL,
                `payment_method` VARCHAR(255) DEFAULT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`wallet_transaction_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminEditCustomerNavbar() {
        return $this->showTemplate('nav_item');
    }

    public function hookDisplayAdminEditCustomerNavContent( $params ) {
        $smarty = $this->smarty;
        $customerId = ArrayUtils::get($params, 'customerId');

        $wallet = $this->getWalletByCustomerId( $customerId );

        $wallet_balance = $wallet['wallet_balance'];
        $transactions = $options = array();

        $sql = new DbQuery();
        $sql->select('*');
        $sql->from('wallet_transaction');
        $sql->where('customer_id = ' . (int) $wallet['customer_id']);
        $sql->orderBy('wallet_transaction_id DESC');
        $sql->limit('20');
        $result = Db::getInstance()->executeS($sql);

        foreach($result as $transaction) {
            $currency = new Currency( (int) $transaction['currency_id'] );
            if(Validate::isLoadedObject($currency)) {
                $options['currency'] = array(
                    'symbol' => $currency->symbol
                );
            } else {
                $options['currency'] = array(
                    'symbol' => 'USD'
                );
            }

            $transactions[] = array(
                'wallet_in' => ($transaction['source'] == "deposit" || $transaction['source'] == "refund") ? true : false,
                'source' => $transaction['source'],
                'payment_method' => $transaction['payment_method'],
                'amount' => $transaction['amount'],
                'amountFormatted' => formatPrice($transaction['amount'], $options),
                'date' => DateUtils::convertToISOFormat($transaction['date_add'])
            );
        }

        $smarty->assign('module_name', $this->name);
        $smarty->assign('customer_id', $wallet['customer_id']);
        $smarty->assign('transactions', $transactions);
        $smarty->assign('wallet_balance', $wallet_balance);

        return $this->showTemplate('nav_content');
    }

    public function loadWallet() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        if(!$user->is_admin) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Not Authorized"
            ]);
        }

        $amount = ArrayUtils::get($payload, 'amount');
        $customer_id = (int) ArrayUtils::get($payload, 'customer_id');

        $wallet = $this->getWalletByCustomerId( $customer_id );
        if(!$wallet) {
            return $app->sendResponse([
                "error" => true
            ]);
        }

        if(!$amount || !Validate::isPrice($amount)) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter a valid amount"
            ]);
        }

        $currencyId = Currency::getIdByIsoCode('USD');
        $currency = new Currency( $currencyId );

        $success = Db::getInstance()->insert(
            'wallet_transaction',
            array(
                'customer_id' => (int) $customer_id,
                'currency_id' => $currency->id,
                'transaction_id' => Tools::randomGen(10),
                'source' => 'deposit',
                'amount' => $amount,
                'date_add' => DateUtils::now()
            )
        );

        if(!$success) {
            return $app->sendResponse([
                "error" => true
            ]);
        }

        $success = Db::getInstance()->update(
            'customer',
            array(
                'wallet_balance' => $wallet['wallet_balance'] + $amount,
            ),
            'customer_id = ' . (int) $customer_id
        );

        Hook::exec('activitylog', [
            'object' => 'wallet',
            'object_id' => $wallet['customer_id'],
            'event' => 'wallet_load_balance'
        ]);

        return $app->controller->modal("SUCCESS", "Your account has been successfully credited", formatPrice($amount) . " has been credited to your account."); 
    }

    public function editWalletBalance() {
        $app = $this->application;
        $payload = $app->request->post();

        $amount = ArrayUtils::get($payload, 'amount');
        $customer_id = (int) ArrayUtils::get($payload, 'customer_id');

        $wallet = $this->getWalletByCustomerId( $customer_id );
        if(!$wallet) {
            return $app->sendResponse([
                "error" => true
            ]);
        }

        if(!Validate::isFloat($amount) || !Validate::isPrice($amount)) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter a valid amount"
            ]);
        }

        $success = Db::getInstance()->update(
            'customer',
            array(
                'wallet_balance' => $amount,
            ),
            'customer_id = ' . (int) $customer_id
        );

        Hook::exec('activitylog', [
            'object' => 'wallet',
            'object_id' => $wallet['customer_id'],
            'event' => 'wallet_edit_balance'
        ]);

        return $app->sendResponse([
            "success" => true,
            "message" => "Wallet balance updated"
        ]);
    }
}