<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Hook;
use Proxim\Mail;
use Proxim\MailTemplate;
use Proxim\Module\Module;
use Proxim\Tools;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class Two_Factor_Auth extends Module
{
    const TWO_FACTOR_ENABLED = 'TWO_FACTOR_ENABLED';
    const TWO_FACTOR_TYPE = 'TWO_FACTOR_TYPE';

    public function __construct()
    { 
        $this->name = 'two_factor_auth';
        $this->icon = 'fas fa-fingerprint';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Two-Factor Authentication';
        $this->description = 'Protect against users using weak passwords, password guessing and brute force attacks.';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        try {
            Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('employee') . " ADD (
                    `two_factor_enabled` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                    `two_factor_type` ENUM('email','sms','google') DEFAULT NULL,
                    `two_factor_key` VARCHAR(64) DEFAULT NULL,
                    `two_factor_gsecret` VARCHAR(64) DEFAULT NULL
                );
            ");
        } catch(Exception $exception) {
        }

        return $this->registerHook([
            'displayFooterAfter',
            'actionAuthentication',
            'displayProfileNavTitle',
            'displayProfileNavbar',
            'displayProfileNavContent'
        ]);
    } 

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    /**
     * return_json
     * 
     * @param array $response
     * @return json
     */
    public function returnJson($response = array()) {
        header('Content-Type: application/json');
        exit(json_encode($response));
    }

    /**
     * reload
     * 
     * @return void
     */
    public function reload() {
        header("Refresh:0");
        exit;
    }

    public function disableTwoFactorAuthentication( $user_id ) {
        $employee = new Employee( (int) $user_id );
        if(Validate::isLoadedObject($employee)) {
            Db::getInstance()->update(
                'employee', [
                    'two_factor_enabled' => '0',
                    'two_factor_type' => null,
                    'two_factor_key' => null,
                    'two_factor_gsecret' => null
                ],
                'employee_id = ' . (int) $employee->id
            );
        }
    }

    public function hookDisplayFooterAfter() {
        $app = $this->application;
        $user = $app->user;

        if( !$user->isLogged() ) {
            return $this->showTemplate('two_factor_modal');
        } 
    } 

    public function getContent()
    {
        $twoFactorConfiguration = Configuration::getMultiple([
            self::TWO_FACTOR_ENABLED,
            self::TWO_FACTOR_TYPE
        ]);

        $sms_enabled = Module::isInstalled('smsnotifications');
        $this->smarty->assign([
            'sms_enabled' => (bool) $sms_enabled,
            'twoFactorConfiguration' => $twoFactorConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplayProfileNavTitle() {
        if(Configuration::get(self::TWO_FACTOR_ENABLED)) {
            return $this->showTemplate('profile.nav.title');
        }
    }

    public function hookDisplayProfileNavbar() {
        if(Configuration::get(self::TWO_FACTOR_ENABLED)) {
            return $this->showTemplate('profile.nav.item');
        }
    }

    public function hookDisplayProfileNavContent() {
        $app = $this->application;
        $controller = $app->controller;
        $user = $app->user;
        $smarty = $app->smarty;

        if(!Configuration::get(self::TWO_FACTOR_ENABLED)) {
            $this->disableTwoFactorAuthentication($user->id);
        }

        $userData = Db::getInstance()->getRow('SELECT two_factor_enabled, two_factor_type, two_factor_key, two_factor_gsecret FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $user->id );

        /* check user two-factor */
        if( $userData['two_factor_enabled'] ) {
            /* enabled */
            /* system two-factor method != user two-factor method */
            if(Configuration::get(self::TWO_FACTOR_TYPE) != $userData['two_factor_type']) {
                $this->disableTwoFactorAuthentication( $user->id );
                $this->reload();
            }
        } else {
            /* disabled */
            if( Configuration::get(self::TWO_FACTOR_TYPE) == "google" ) {
                $ga = new PHPGangsta_GoogleAuthenticator();
                /* get user gsecret */
                if(!$userData['two_factor_gsecret']) {
                    /* create new gsecret */
                    $two_factor_gsecret = $ga->createSecret();
                    /* update user gsecret */
                    Db::getInstance()->update(
                        'employee', [
                            'two_factor_gsecret' => $two_factor_gsecret
                        ],
                        'employee_id = ' . (int) $user->id
                    );
                } else {
                    $two_factor_gsecret = $userData['two_factor_gsecret'];
                }
                /* get QR */
                $two_factor_QR = $ga->getQRCodeGoogleUrl(
                    $user->email, 
                    $two_factor_gsecret, 
                    Configuration::get('SITE_NAME')
                );
                /* assign variables */
                $smarty->assign([
                    'two_factor_QR' => $two_factor_QR,
                    'two_factor_gsecret' => $two_factor_gsecret
                ]);
            }
        }

        $smarty->assign([
            'userData' => $userData,
            'two_factor_type' => Configuration::get(self::TWO_FACTOR_TYPE)
        ]);

        return $this->showTemplate('profile.tab.pane');
    }

    public function hookActionAuthentication( $payload ) {
        $app = $this->application;
        $controller = $app->controller;
        $employee = ArrayUtils::get($payload, 'employee');

        /* two-factor authentication */
        $userData = Db::getInstance()->getRow('SELECT two_factor_enabled, two_factor_type, two_factor_key, two_factor_gsecret FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $employee->id );
        $two_factor_type = Configuration::get(self::TWO_FACTOR_TYPE);

        if($userData['two_factor_enabled']) {
            /* system two-factor disabled */
            if(!Configuration::get(self::TWO_FACTOR_ENABLED)) {
                $this->disableTwoFactorAuthentication($employee->id);
                return;
            }

            /* system two-factor method != user two-factor method */
            if($two_factor_type != $userData['two_factor_type']) {
                $this->disableTwoFactorAuthentication($employee->id);
                return;
            }

            switch ($two_factor_type) {
                case 'email':
                    /* generate two-factor key */
                    $two_factor_key = Tools::randomGen(6, 'NUMERIC');

                    /* update user two factor key */
                    Db::getInstance()->update(
                        'employee', [
                            'two_factor_key' => $two_factor_key
                        ],
                        'employee_id = ' . (int) $employee->id
                    );

                    /* prepare method name */
                    $method = "Email";
                    /* prepare activation email */
                    $subject = Configuration::get('SITE_NAME') . " Two-Factor Authentication Token";

                    /* send email */
                    if(!Mail::send(
                        MailTemplate::TEMPLATE_EMPLOYEE_TWO_FACTOR,
                        $subject,
                        array(
                            "username" => Tools::ucfirst($employee->first_name), 
                            "two_factor_key" => $two_factor_key
                        ),
                        $employee->email,
                        $employee->first_name . ' ' . $employee->last_name,
                        null,
                        null,
                        null,
                        null,
                        $this->_path . 'mails/'
                    )) {
                        $this->returnJson([
                            "error" => true,
                            "message" => "Two-factor authentication email could not be sent"
                        ]);
                    }
                    break;
                
                case 'sms':
                    /* system two-factor method = sms but not user phone not verified */
                    if(!$employee->phone) {
                        $this->disableTwoFactorAuthentication($employee->id);
                        return;
                    }

                    /* generate two-factor key */
                    $two_factor_key = Tools::randomGen(6, 'NUMERIC');
                    /* update user two factor key */
                    Db::getInstance()->update(
                        'employee', [
                            'two_factor_key' => $two_factor_key
                        ],
                        'employee_id = ' . (int) $employee->id
                    );

                    /* prepare method name */
                    $method = "Phone";
                    /* prepare activation SMS */
                    $message  = Configuration::get('SITE_NAME') . " Two-factor authentication key: " . $two_factor_key;

                    /* send SMS */
                    $smsnotifications = Module::getInstanceByName('smsnotifications');
                    if(!$smsnotifications->sendSMS($employee->phone, $message, PROX_SITE_ID)) {
                        $this->returnJson([
                            "error" => true,
                            "message" => "Two-factor authentication SMS could not be sent"
                        ]);
                    }
                    break;

                case 'google':
                    /* prepare method name */
                    $method = "Google Authenticator app";
                    break;
            }

            $this->returnJson([
                "callback" => "modal('#two-factor-authentication', { 'employee_id':'" . $employee->id . "', 'method': '".$method."'})"
            ]);
        }
    }

    public function twoFactorAuthentication() {
        $app = $this->application;
        $user = $app->user;
        $payload = $app->request->post();

        $employee_id = ArrayUtils::get($payload, 'employee_id');
        $two_factor_key = ArrayUtils::get($payload, 'two_factor_key');

        if($user->isLogged()) {
            return $app->sendResponse( array('callback' => 'window.location.reload();') );
        }

        if(!$two_factor_key) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Please enter the 6-digit code"
            ]);
        }

        $employee = new Employee( (int) $employee_id );
        if(!Validate::isLoadedObject($employee)) {
            return $app->sendResponse([
                "error" => true
            ]);
        }

        if(Configuration::get(self::TWO_FACTOR_TYPE) == "google") {
            /* get user */
            $two_factor_gsecret = Db::getInstance()->getValue('SELECT two_factor_gsecret FROM '. Db::prefix('employee') . ' WHERE employee_id = ' . (int) $employee->id );
            if(!$two_factor_gsecret) {
                return $app->sendResponse([
                    "error" => true
                ]);
            }
            
            /* Google Authenticator */
            $ga = new PHPGangsta_GoogleAuthenticator();
            /* verify code */
            if(!$ga->verifyCode($two_factor_gsecret, $two_factor_key)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Invalid code, please try again"
                ]);
            }
        } else {
            /* check two-factor key */
            $check_key = Db::getInstance()->getRow('SELECT employee_id FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $employee->id . ' AND two_factor_key = \'' . pSQL($two_factor_key) . '\'' );
            if(!$check_key) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Invalid code, please try again"
                ]);
            }
        }

        $employee->failed_login_count = 0;
        $employee->last_login = DateUtils::now();
		$employee->update();

		Hook::exec('activitylog', [
            'object' => 'employee',
            'object_id' => $employee->id,
            'event' => 'signin'
        ]);

        $app->updateUser( $employee );

        return $app->sendResponse( array('callback' => 'window.location.reload();') );
    }

    public function updateProfileSettings() {
        $app = $this->application;
        $user = $app->user;
        $payload = $app->request->post();

        $type = ArrayUtils::get($payload, 'type');
        $two_factor_enabled = ArrayUtils::has($payload, 'two_factor_enabled');
        $two_factor_type = Configuration::get(self::TWO_FACTOR_TYPE);

        $userData = Db::getInstance()->getRow('SELECT two_factor_enabled, two_factor_type, two_factor_key, two_factor_gsecret FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $user->id );

        if( $two_factor_type != $type) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Select a valid authentication type"
            ]);
        }

        switch ($two_factor_type) {
            case 'sms':
                if($two_factor_enabled && !$user->phone) {
                    return $app->sendResponse([
                        "error" => true,
                        "message" => "Two-Factor Authentication can't be enabled till you enter your phone number"
                    ]);
                }
                break;

            case 'google':

                if( ArrayUtils::has($payload, 'gcode') ) {
                    $gcode = ArrayUtils::get($payload, 'gcode');

                    /* Google Authenticator */
                    $ga = new PHPGangsta_GoogleAuthenticator();
                    /* verify code */
                    if(!$ga->verifyCode($userData['two_factor_gsecret'], $gcode)) {
                        return $app->sendResponse([
                            "error" => true,
                            "message" => "Invalid code, Try again or try to scan your QR code again"
                        ]);
                    }

                    $two_factor_enabled = true;
                }
                break;
        }

        /* update user */
        Db::getInstance()->update(
            'employee', [
                'two_factor_enabled' => (bool) $two_factor_enabled,
                'two_factor_type' => $two_factor_type,
            ],
            'employee_id = ' . (int) $user->id
        );

        return $app->sendResponse([
            "success" => true,
			"message" => "Your two-factor authentication settings have been updated"
        ]);
    }

    public function updateTwoFactorSettings() {
        $app = $this->application;
        $payload = $app->request->post();
        
        Configuration::updateValue(self::TWO_FACTOR_ENABLED, ArrayUtils::has($payload, 'two_factor_enabled') ? true : false);
        Configuration::updateValue(self::TWO_FACTOR_TYPE, ArrayUtils::get($payload, 'two_factor_type'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}