<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class Stripe extends PaymentModule
{
    /**
    * Stripe configuration
    */
    const STRIPE_ENABLED = 'STRIPE_ENABLED';
    const STRIPE_MODE = 'STRIPE_MODE';
    const STRIPE_LIVE_KEY = 'STRIPE_LIVE_KEY';
    const STRIPE_LIVE_SECRET = 'STRIPE_LIVE_SECRET';
    const STRIPE_TEST_KEY = 'STRIPE_TEST_KEY';
    const STRIPE_TEST_SECRET = 'STRIPE_TEST_SECRET';

    public function __construct()
    {
        $this->name = 'stripe';
        $this->icon = 'fab fa-stripe';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Stripe';
        $this->description = 'Set up Stripe as a payment method and enable users to pay without leaving your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $stripeConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::STRIPE_ENABLED,
                self::STRIPE_MODE,
                self::STRIPE_LIVE_KEY,
                self::STRIPE_LIVE_SECRET,
                self::STRIPE_TEST_KEY,
                self::STRIPE_TEST_SECRET,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'stripeConfiguration' => $stripeConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateStripeSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::STRIPE_ENABLED, (ArrayUtils::get($payload, 'stripe_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue(self::STRIPE_MODE, ArrayUtils::get($payload, 'stripe_mode'), false, $site_id);
        Configuration::updateValue(self::STRIPE_LIVE_KEY, ArrayUtils::get($payload, 'stripe_live_key'), false, $site_id);
        Configuration::updateValue(self::STRIPE_LIVE_SECRET, ArrayUtils::get($payload, 'stripe_live_secret'), false, $site_id);
        Configuration::updateValue(self::STRIPE_TEST_KEY, ArrayUtils::get($payload, 'stripe_test_key'), false, $site_id);
        Configuration::updateValue(self::STRIPE_TEST_SECRET, ArrayUtils::get($payload, 'stripe_test_secret'), false, $site_id);
        
        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}