<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\MailTemplate;
use Proxim\Module\Module;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;
use Twilio\Rest\Client;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class SMSNotifications extends Module
{
    const SMS_NOTIFICATIONS_SMS_SITE_PHONE = 'SMS_NOTIFICATIONS_SMS_SITE_PHONE';
    const SMS_NOTIFICATIONS_SMS_PROVIDER = 'SMS_NOTIFICATIONS_SMS_PROVIDER';
    const SMS_NOTIFICATIONS_TWILIO_SID = 'SMS_NOTIFICATIONS_TWILIO_SID';
    const SMS_NOTIFICATIONS_TWILIO_TOKEN = 'SMS_NOTIFICATIONS_TWILIO_TOKEN';
    const SMS_NOTIFICATIONS_TWILIO_PHONE = 'SMS_NOTIFICATIONS_TWILIO_PHONE';
    const SMS_NOTIFICATIONS_BULKSMS_USERNAME = 'SMS_NOTIFICATIONS_BULKSMS_USERNAME';
    const SMS_NOTIFICATIONS_BULKSMS_PASSWORD = 'SMS_NOTIFICATIONS_BULKSMS_PASSWORD';
    const SMS_NOTIFICATIONS_INFOBIP_USERNAME = 'SMS_NOTIFICATIONS_INFOBIP_USERNAME';
    const SMS_NOTIFICATIONS_INFOBIP_PASSWORD = 'SMS_NOTIFICATIONS_INFOBIP_PASSWORD';

    const SMS_NOTIFICATIONS_ORDER_PAYMENT = 'SMS_NOTIFICATIONS_ORDER_PAYMENT';
    const SMS_NOTIFICATIONS_PASSWORD_CHANGE = 'SMS_NOTIFICATIONS_PASSWORD_CHANGE';
    const SMS_NOTIFICATIONS_PAY_ORDER = 'SMS_NOTIFICATIONS_PAY_ORDER';
    const SMS_NOTIFICATIONS_ORDER_REFUND = 'SMS_NOTIFICATIONS_ORDER_REFUND';
    const SMS_NOTIFICATIONS_EDITOR_NEW_ORDER = 'SMS_NOTIFICATIONS_EDITOR_NEW_ORDER';
    const SMS_NOTIFICATIONS_WRITER_NEW_ORDER = 'SMS_NOTIFICATIONS_WRITER_NEW_ORDER';
    const SMS_NOTIFICATIONS_ORDER_ASSIGN = 'SMS_NOTIFICATIONS_ORDER_ASSIGN';
    const SMS_NOTIFICATIONS_ORDER_REASSIGN = 'SMS_NOTIFICATIONS_ORDER_REASSIGN';
    const SMS_NOTIFICATIONS_WRITER_SUBMITTED_ORDER = 'SMS_NOTIFICATIONS_WRITER_SUBMITTED_ORDER';
    const SMS_NOTIFICATIONS_ORDER_COMPLETE = 'SMS_NOTIFICATIONS_ORDER_COMPLETE';
    const SMS_NOTIFICATIONS_ORDER_STATUS = 'SMS_NOTIFICATIONS_ORDER_STATUS';
    const SMS_NOTIFICATIONS_ORDER_REVISION = 'SMS_NOTIFICATIONS_ORDER_REVISION';
    const SMS_NOTIFICATIONS_ORDER_MESSAGE = 'SMS_NOTIFICATIONS_ORDER_MESSAGE';
    const SMS_NOTIFICATIONS_ORDER_FILES = 'SMS_NOTIFICATIONS_ORDER_FILES';
    const SMS_NOTIFICATIONS_FILE_DELETE_REQUEST = 'SMS_NOTIFICATIONS_FILE_DELETE_REQUEST';
    const SMS_NOTIFICATIONS_FEEDBACK_REPLY = 'SMS_NOTIFICATIONS_FEEDBACK_REPLY';

    public function __construct()
    {
        $this->name = 'smsnotifications';
        $this->icon = 'fas fa-comment-alt';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'SMS Notifications';
        $this->description = 'Send SMS Messages, Password Resets, Alerts, Order Notifications With Confidence';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'actionEmailSendBefore',
            'actionMailAlterMessageBeforeSend'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $smsnotificationsConfiguration = Configuration::getMultiple([
            self::SMS_NOTIFICATIONS_SMS_SITE_PHONE,
            self::SMS_NOTIFICATIONS_SMS_PROVIDER,
            self::SMS_NOTIFICATIONS_TWILIO_SID,
            self::SMS_NOTIFICATIONS_TWILIO_TOKEN,
            self::SMS_NOTIFICATIONS_TWILIO_PHONE,
            self::SMS_NOTIFICATIONS_BULKSMS_USERNAME,
            self::SMS_NOTIFICATIONS_BULKSMS_PASSWORD,
            self::SMS_NOTIFICATIONS_INFOBIP_USERNAME,
            self::SMS_NOTIFICATIONS_INFOBIP_PASSWORD,
            self::SMS_NOTIFICATIONS_PASSWORD_CHANGE,
            self::SMS_NOTIFICATIONS_EDITOR_NEW_ORDER,
            self::SMS_NOTIFICATIONS_PAY_ORDER,
            self::SMS_NOTIFICATIONS_ORDER_FILES,
            self::SMS_NOTIFICATIONS_ORDER_ASSIGN,
            self::SMS_NOTIFICATIONS_ORDER_REASSIGN,
            self::SMS_NOTIFICATIONS_WRITER_SUBMITTED_ORDER,
            self::SMS_NOTIFICATIONS_ORDER_COMPLETE,
            self::SMS_NOTIFICATIONS_ORDER_STATUS,
            self::SMS_NOTIFICATIONS_ORDER_REVISION,
            self::SMS_NOTIFICATIONS_ORDER_MESSAGE,
            self::SMS_NOTIFICATIONS_FEEDBACK_REPLY,
            self::SMS_NOTIFICATIONS_ORDER_PAYMENT,
            self::SMS_NOTIFICATIONS_FILE_DELETE_REQUEST,
            self::SMS_NOTIFICATIONS_WRITER_NEW_ORDER,
            self::SMS_NOTIFICATIONS_ORDER_REFUND
        ]);

        $this->smarty->assign([
            'svg_path' => $this->_url_path . 'images/svg',
            'module_nav_tabs' => $this->getTemplateContent('nav'),
            'smsnotificationsConfiguration' => $smsnotificationsConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    /**
     * sendSMS
     *
     * @param string $phone
     * @param string $message
     * @return boolean
     */
    public function sendSMS($phone, $message, $siteId) {
        $smsProvider = Configuration::get(self::SMS_NOTIFICATIONS_SMS_PROVIDER);

        switch($smsProvider) {
            case 'twilio':
                try {
                    $client = new Client(
                        Configuration::get(self::SMS_NOTIFICATIONS_TWILIO_SID),
                        Configuration::get(self::SMS_NOTIFICATIONS_TWILIO_TOKEN)
                    );
    
                    $message = $client->account->messages->create(
                        $phone,
                        array(
                            'from' => Configuration::get(self::SMS_NOTIFICATIONS_TWILIO_PHONE),
                            'body' => $message
                        )
                    );

                    if(!$message->sid) {
                        return false;
                    }
                } catch(Exception $e) {
                    return false;
                }

                break;
            
            case 'bulksms':
                $username = Configuration::get('SMS_NOTIFICATIONS_BULKSMS_USERNAME');
                $password = Configuration::get('SMS_NOTIFICATIONS_BULKSMS_PASSWORD');
                $messages = array(
                    array(
                        'to' => $phone,
                        'body' => $message
                    )
                );

                try {
                    $ch = curl_init();
                    $headers = array(
                        'Content-Type:application/json',
                        'Authorization:Basic '. base64_encode("$username:$password")
                    );
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                    curl_setopt ( $ch, CURLOPT_URL, "https://api.bulksms.com/v1/messages?auto-unicode=true&longMessageMaxParts=30" );
                    curl_setopt ( $ch, CURLOPT_POST, 1 );
                    curl_setopt ( $ch, CURLOPT_RETURNTRANSFER, 1 );
                    curl_setopt ( $ch, CURLOPT_POSTFIELDS, json_encode($messages) );
                    /* allow cUrl functions 20 seconds to execute */
                    curl_setopt ( $ch, CURLOPT_TIMEOUT, 20 );
                    /* wait 10 seconds while trying to connect */
                    curl_setopt ( $ch, CURLOPT_CONNECTTIMEOUT, 10 );
                    $output = array();
                    $output['server_response'] = curl_exec( $ch );
                    $curl_info = curl_getinfo( $ch );
                    $output['http_status'] = $curl_info[ 'http_code' ];
                    $output['error'] = curl_error($ch);
                    curl_close( $ch );
                    if($output['http_status'] != 201) {
                        return false;
                    }
                } catch(Exception $e) {
                    return false;
                }
                break;
    
            case 'infobip':
                $infobip_username = Configuration::get('SMS_NOTIFICATIONS_INFOBIP_USERNAME');
                $infobip_password = Configuration::get('SMS_NOTIFICATIONS_INFOBIP_PASSWORD');

                $sms = array(
                    "from" => Configuration::get('SITE_NAME', $siteId),
                    "to" => $phone,
                    "text" => $message
                );

                try {
                    $ch = curl_init();
                    $header = array("Content-Type:application/json", "Accept:application/json");
                    curl_setopt($ch, CURLOPT_URL, "https://api.infobip.com/sms/1/text/single");
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
                    curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                    curl_setopt($ch, CURLOPT_USERPWD, $infobip_username . ":" . $infobip_password );
                    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, TRUE);
                    curl_setopt($ch, CURLOPT_MAXREDIRS, 2);
                    curl_setopt($ch, CURLOPT_POST, 1);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($sms));
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    $response = curl_exec($ch);
                    $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    curl_close($ch);
                    if(!($http_status >= 200 && $http_status < 300)) {
                        return false;
                    }

                } catch(Exception $e) {
                    return false;
                }

                break;
        }

        return true;
    }

    public function hookActionEmailSendBefore($payload) {
        $to = ArrayUtils::get($payload, 'to');
        $templateVars = ArrayUtils::get($payload, 'templateVars');
        $template = ArrayUtils::get($payload, 'template');
        $siteId = ArrayUtils::get($payload, 'siteId');

        $site_name = Configuration::get('SITE_NAME', $siteId);
        $site_url = Configuration::get('SITE_DOMAIN', $siteId);
        $site_email = Configuration::get('SITE_EMAIL', $siteId);

        switch($template) {
            case MailTemplate::TEMPLATE_ORDER_PAY_REMINDER:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_title = ArrayUtils::get($templateVars, 'order_title');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $customer_name = ArrayUtils::get($templateVars, 'name', 'Customer');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_PAY_ORDER);
                if(!$smsMessage) return;

                $customer = new Customer();
                $customer->getByEmail($to, null, $siteId);
                if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{customer_name}', '{order_url}', '{order_title}'), array($order_id, $customer_name, $order_url, $order_title), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($customer->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_FILES:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $customer_name = ArrayUtils::get($templateVars, 'name', 'Customer');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_FILES);
                if(!$smsMessage) return;

                $customer = new Customer();
                $customer->getByEmail($to, null, $siteId);
                if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{customer_name}', '{order_url}'), array($order_id, $customer_name, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($customer->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_STATUS_CHANGE:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $customer_name = ArrayUtils::get($templateVars, 'customer_name', 'Customer');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_STATUS);
                if(!$smsMessage) return;

                $customer = new Customer();
                $customer->getByEmail($to, null, $siteId);
                if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{customer_name}'), array($order_id, $customer_name), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($customer->phone, $smsMessage, $siteId);
                }

                break;

            case MailTemplate::TEMPLATE_ORDER_ASSIGN:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $username = ArrayUtils::get($templateVars, 'username');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_ASSIGN);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{username}', '{order_url}'), array($order_id, $username, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }

                break;

            case MailTemplate::TEMPLATE_ORDER_REASSIGN:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $username = ArrayUtils::get($templateVars, 'username');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_REASSIGN);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{username}', '{order_url}'), array($order_id, $username, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_WRITER_SUBMIT:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_WRITER_SUBMITTED_ORDER);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{order_url}'), array($order_id, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_NEW_EDITOR:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_EDITOR_NEW_ORDER);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{order_url}'), array($order_id, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_COMPLETE:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $customer_name = ArrayUtils::get($templateVars, 'name', 'Customer');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_COMPLETE);
                if(!$smsMessage) return;

                $customer = new Customer();
                $customer->getByEmail($to, null, $siteId);
                if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{order_url}', '{customer_name}'), array($order_id, $order_url, $customer_name), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($customer->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_REVISION:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $order_url = ArrayUtils::get($templateVars, 'order_url');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_REVISION);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{order_url}'), array($order_id, $order_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_MESSAGE:
                $send_sms = ArrayUtils::get($templateVars, 'send_sms');
                if(!$send_sms) return;

                $to_department = ArrayUtils::get($templateVars, 'to_department');
                $order_id = ArrayUtils::get($templateVars, 'orderId');
                $subject = ArrayUtils::get($templateVars, 'subject');
                $text_message = ArrayUtils::get($templateVars, 'text_message');
                $username = ArrayUtils::get($templateVars, 'name');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_ORDER_MESSAGE);
                if(!$smsMessage) return;

                if($to_department == DEPARTMENT_CUSTOMER) {
                    $customer = new Customer();
                    $customer = $customer->getByEmail($to, null, $siteId);
                    if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                        $smsMessage = str_replace(
                            array('{order_id}', '{subject}', '{text_message}', '{username}'), 
                            array($order_id, $subject, $text_message, $username), 
                            $smsMessage
                        );
                        $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                        $this->sendSMS($customer->phone, $smsMessage, $siteId);
                    }
                } else {
                    $employee = new Employee();
                    $employee->getByEmail($to);
                    if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                        $smsMessage = str_replace(array('{order_id}', '{subject}', '{text_message}', '{username}'), array($order_id, $subject, $text_message, $username), $smsMessage);
                        $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                        $this->sendSMS($employee->phone, $smsMessage, $siteId);
                    }
                }
                break;

            case MailTemplate::TEMPLATE_ORDER_FEEDBACK_REPLY:
                $order_id = ArrayUtils::get($templateVars, 'order_id');
                $reply = ArrayUtils::get($templateVars, 'reply');
                $customer_name = ArrayUtils::get($templateVars, 'name');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_FEEDBACK_REPLY);
                if(!$smsMessage) return;

                $customer = new Customer();
                $customer->getByEmail($to, null, $siteId);
                if(Validate::isLoadedObject($customer) && Validate::isPhoneNumber($customer->phone)) {
                    $smsMessage = str_replace(array('{order_id}', '{reply}', '{customer_name}'), array($order_id, $reply, $customer_name), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($customer->phone, $smsMessage, $siteId);
                }
                break;

            case MailTemplate::TEMPLATE_EMPLOYEE_PASSWORD_CHANGE:
                $username = ArrayUtils::get($templateVars, 'username');
                $confirm_url = ArrayUtils::get($templateVars, 'url');
                $smsMessage = Configuration::get(self::SMS_NOTIFICATIONS_PASSWORD_CHANGE);
                if(!$smsMessage) return;

                $employee = new Employee();
                $employee->getByEmail($to);
                if(Validate::isLoadedObject($employee) && Validate::isPhoneNumber($employee->phone)) {
                    $smsMessage = str_replace(array('{username}', '{confirm_url}'), array($username, $confirm_url), $smsMessage);
                    $smsMessage = str_replace(array('{site_name}', '{site_url}', '{site_email}'), array($site_name, $site_url, $site_email), $smsMessage);
                    $this->sendSMS($employee->phone, $smsMessage, $siteId);
                }

                break;
        }
    } 

    public function updateSMSNotificationsSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        foreach($payload as $key => $value) {
            Configuration::updateValue($key, $value);
        }

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }

    public function updateSMSNotificationsTemplates() {
        $app = $this->application;
        $payload = $app->request->post();

        foreach($payload as $key => $value) {
            Configuration::updateValue($key, $value);
        }

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}
