<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;
use Sendpulse\RestApi\ApiClient;
use Sendpulse\RestApi\Storage\FileStorage;

require_once dirname(__FILE__) . '/vendor/autoload.php';

class SendPulse extends Module
{
    const SENDPULSE_ENABLED = 'SENDPULSE_ENABLED';
    const SENDPULSE_API_USER_ID = 'SENDPULSE_API_USER_ID';
    const SENDPULSE_API_SECRET = 'SENDPULSE_API_SECRET';

    public function __construct()
    {
        $this->name = 'sendpulse';
        $this->icon = 'fa fa-envelope';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'SendPulse';
        $this->description = 'Send Email Newsletters, Password Resets, Promotional Emails, Shipping Notifications With Confidence';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'displayAdminEditSiteGeneralTabContent',
            'actionAfterSiteSettingsGeneralUpdate',
            'actionEmailSendBefore',
            'actionMailAlterMessageBeforeSend'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminEditSiteGeneralTabContent() {
        $application = $this->application;
        $smarty = $this->smarty;

        $sendpulseConfiguration = Configuration::getMultiple([
                self::SENDPULSE_ENABLED
            ],
            $application->site->id
        );
        $smarty->assign('sendpulseConfiguration', $sendpulseConfiguration);
        
        return $this->showTemplate('tab_content');
    }

    public function getContent()
    {
        $sendpulseConfiguration = Configuration::getMultiple([
                self::SENDPULSE_API_USER_ID,
                self::SENDPULSE_ENABLED,
                self::SENDPULSE_API_SECRET
            ]
        );

        $this->smarty->assign([
            'sendpulseConfiguration' => $sendpulseConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookActionMailAlterMessageBeforeSend($payload) {
        if(!Configuration::get(self::SENDPULSE_ENABLED)) {
            return true;
        }

        $message = ArrayUtils::get($payload, 'message');
        $fileAttachment = ArrayUtils::get($payload, 'fileAttachment');
        $htmlMessage = ArrayUtils::get($payload, 'htmlMessage');

        $from = $message->getFrom();
        $to = $message->getTo();

        $fromAddress = $toAddress = $attachment = array();
        if (is_array($from)) {
            foreach ($from as $addr => $addrName) {
                $addr = trim($addr);
                $fromAddress = array(
                    'name' => $addrName,
                    'email' => $addr
                );
            }
        }

        if (is_array($to)) {
            foreach ($to as $addr => $addrName) {
                $addr = trim($addr);
                $toAddress[] = array(
                    'name' => $addrName,
                    'email' => $addr
                );
            }
        }

        if ($fileAttachment && !empty($fileAttachment)) {
            // Multiple attachments?
            if (!is_array(current($fileAttachment))) {
                $fileAttachment = array($fileAttachment);
            }

            foreach ($fileAttachment as $att) {
                $attachment[$att['name']] = file_get_contents($att['file_path']);
            }
        }

        $email = array(
            'html' => $htmlMessage,
            'subject' => $message->getSubject(),
            'from' => $fromAddress,
            'to' => $toAddress,
            'attachments' => $attachment
        );

        try {
            $sendpulseClient = new ApiClient(
                Configuration::get(self::SENDPULSE_API_USER_ID), 
                Configuration::get(self::SENDPULSE_API_SECRET), 
                new FileStorage()
            );
            $sendpulseClient->smtpSendMail($email);
        } catch (Exception $e) {
        }

        return false;
    }

    public function updateSendPulseSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::SENDPULSE_ENABLED, ArrayUtils::has($payload, 'sendpulse_enabled') ? true : false);
        Configuration::updateValue(self::SENDPULSE_API_USER_ID, ArrayUtils::get($payload, 'sendpulse_api_user_id'));
        Configuration::updateValue(self::SENDPULSE_API_SECRET, ArrayUtils::get($payload, 'sendpulse_api_secret'));

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    } 

    public function hookActionAfterSiteSettingsGeneralUpdate( $params ) {
        $site_id = ArrayUtils::get($params, 'site_id');
        $payload = ArrayUtils::get($params, 'payload');

        $site = new Site( (int) $site_id );
        if(Validate::isLoadedObject($site)) {
            Configuration::updateValue(self::SENDPULSE_ENABLED, ArrayUtils::has($payload, 'sendpulse_enabled') ? true : false, false, $site->id);
        }
    }
}
