<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Pager;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Util\ArrayUtils;
use Proxim\Util\Formatting;
use Proxim\Util\StringUtils;
use Proxim\Validate;

define('CURRENT_SAMPLES_MODULE_DIR', realpath(dirname(__FILE__)));

require_once(CURRENT_SAMPLES_MODULE_DIR . '/classes/Sample.php');
require_once(CURRENT_SAMPLES_MODULE_DIR . '/classes/SampleFile.php');

class Samples extends Module
{
    public function __construct()
    {
        $this->name = 'samples';
        $this->icon = 'fa fa-book';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Samples';
        $this->description = 'Display a list of your writer samples to your customers to increase conversion';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        }

        $this->registerHook([
            'displayAdminNavBarBeforeEnd',
            'actionBeforeAdminDelete',
            'actionAfterFileUpload'
        ]);
    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('sample') . " (
                `sample_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `title` TEXT NOT NULL,
                `instructions` LONGTEXT DEFAULT NULL,
                `academic_level_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_type_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_type_option` TEXT DEFAULT NULL,
                `topic_category_id` BIGINT(20) UNSIGNED NOT NULL,
                `topic_category_option` TEXT DEFAULT NULL,
                `is_complex_assignment` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                `paper_format_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_format_option` TEXT DEFAULT NULL,
                `pages` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `slides` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `sources` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `spacing` enum('single','double') NOT NULL DEFAULT 'double',
                `views` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`sample_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('sample_file') . " (
                `sample_file_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `sample_id` BIGINT(20) UNSIGNED NOT NULL,
                `name` VARCHAR(255) NOT NULL,
                `size` INT(10) NOT NULL DEFAULT 0,
                `source` TEXT NOT NULL,
                `extension` VARCHAR(20) DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`sample_file_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminNavBarBeforeEnd() {
        if($this->checkAccess()) {
            return $this->showTemplate('nav_item');
        }
    }

    public function assignVairiables() {
        global $globals;
        $smarty = $this->smarty;

        $academicLevels = array();

        $objectPresenter = new ObjectPresenter();

        $disciplineGroups = Discipline::getDisciplinesByGroup();
        $paperFormats = PaperFormat::getPaperFormats();
        $paperTypesPreferences = PaperType::getPaperTypes();

        $paperTypes = array();
        foreach($paperTypesPreferences as $paperType) {
            $paperType = new PaperType( $paperType['paper_type_id'] );
            if(Validate::isLoadedObject($paperType)) {
                $paperTypes[] = $objectPresenter->present($paperType);
            }
        }

        foreach ( $globals['academicLevels'] as $academicLevelId => $academicLevel ) {
            $academicLevel['id'] = $academicLevelId;
            $academicLevels[] = $academicLevel;
        }

        $smarty->assign([
            'academicLevels' => $academicLevels,
            'paperTypes' => $paperTypes,
            'paperFormats' => $paperFormats,
            'disciplineGroups' => $disciplineGroups,
        ]);
    }

    public function listSamples() {
        global $globals;

        $app = $this->application;
        $smarty = $this->smarty;
        $params = $app->request->get();

        $is_search = ArrayUtils::has($params, 'search') ? true : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
        $sample_id = ArrayUtils::has($params, 'sample_id');
        $corslevel = ArrayUtils::has($params, 'corslevel');
        $discipline_ids = ArrayUtils::has($params, 'discipline_ids');
        $title = ArrayUtils::has($params, 'title');

        $sql = new DbQuery();
        $sql->select('DISTINCT p.*');
        $sql->from('sample', 'p');

        if ($is_search) {
            if ($sample_id) {
                $sql->where('p.sample_id LIKE \'%' . pSQL($sample_id) . '%\'');
            }

            if($corslevel > 0 ) {
                $sql->where( 'p.academic_level_id = ' . (int) $corslevel );
            }

            if(is_array($discipline_ids) && !empty($discipline_ids)) {
                $disciplineQ = implode(',', $discipline_ids);
                $sql->where( "p.topic_category_id IN ($disciplineQ)" );
            }

            if ($title) {
                $sql->where('p.title LIKE \'%' . pSQL($title) . '%\'');
            }
        }

        $sql->orderBy('p.sample_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_samples = count($result);

        $samples = array();

        if( $total_samples > 0) {
            $params['total_items'] = $total_samples;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/samples/listSamples?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            $objectPresenter = new ObjectPresenter();
            foreach( $result as $sample ) {
                $sample = new Sample( (int) $sample['sample_id'] );
                if( Validate::isLoadedObject($sample) ) {
                    $samplePresented = $objectPresenter->present( $sample );

                    $paper_type = new PaperType( (int) $sample->paper_type_id );
                    $samplePresented['paper_type'] = array(
                        'id' => (int) $paper_type->id,
                        'title' => $paper_type->title
                    );

                    $discipline = new Discipline( (int) $sample->topic_category_id );
                    $samplePresented['discipline'] = array(
                        'id' => (int) $discipline->id,
                        'title' => $discipline->title
                    );

                    $samplePresented['academic_level'] = ArrayUtils::get($globals['academicLevels'],  $sample->academic_level_id );

                    $samples[] = $samplePresented;
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        } 


        $smarty->assign([
            'view' => 'samples',
            'sub_view' => 'listSamples',
            'samples' => $samples,
            'is_search' => $is_search
        ]);

        return $this->showTemplate('samples');
    }

    public function addNew() {
        $app = $this->application;
        $smarty = $this->smarty;

        $this->assignVairiables();

        $smarty->assign([
            'view' => 'samples',
            'sub_view' => 'addSample'
        ]);

        return $this->showTemplate('samples');
    }

    public function viewSample() {
        global $globals;
        $app = $this->application;
        $smarty = $this->smarty;
        $sampleId = $app->request->get('sampleId');

        $sample = new Sample( (int) $sampleId );
        if(!Validate::isLoadedObject($sample)) {
            $app->setTemplate('404');
            $app->display();
        }

        $objectPresenter = new ObjectPresenter();

        $files = $sample->getFiles();

        $sampleFiles = array();
        foreach($files as $sampleFile) {
            $sampleFile = new SampleFile( (int) $sampleFile['sample_file_id'] );
            if(Validate::isLoadedObject($sampleFile)) {
                $sampleFile = $objectPresenter->present($sampleFile);
                $sampleFile['icon'] = $sampleFile['extension'];
                $sampleFile['size_formatted'] = formatBytes($sampleFile['size']);
                $sampleFile['download_link'] =  $this->application->base_uri . "/samples/downloadSample?fileId=" . (int) $sampleFile['id'];
                $sampleFiles[] = $sampleFile;
            }
        }

        $this->assignVairiables();

        $samplePresented = $objectPresenter->present($sample);

        $samplePresented['files'] = $sampleFiles;

        $paper_type = new PaperType( (int) $sample->paper_type_id );
        $samplePresented['paper_type'] = array(
            'id' => (int) $paper_type->id,
            'title' => $paper_type->title
        );

        $discipline = new Discipline( (int) $sample->topic_category_id );
        $samplePresented['discipline'] = array(
            'id' => (int) $discipline->id,
            'title' => $discipline->title
        );

        $samplePresented['academic_level'] = ArrayUtils::get($globals['academicLevels'],  $sample->academic_level_id );

        $smarty->assign([
            'view' => 'samples',
            'sub_view' => 'viewSample',
            'sample' => $samplePresented
        ]);

        return $this->showTemplate('samples');
    }

    public function downloadSample() {
        $app = $this->application;
        $fileId = $app->request->get('fileId');

        $sampleFile = new SampleFile( (int) $fileId );
        if(!Validate::isLoadedObject($sampleFile)) {
            $app->setTemplate('404');
            $app->display();
        }

        $file_path = PROX_DIR_UPLOADS . $sampleFile->source;

		$downloadFileName = $sampleFile->name;
		if($sampleFile->extension && StringUtils::endsWith($downloadFileName, $sampleFile->extension)) {
			$downloadFileName .= "." . $sampleFile->extension;
		}

		$downloadFile = @fopen($file_path,"rb");
		if (!$downloadFile) {
			$app->setTemplate('404');
        	$app->display();
		}

        $downloadFileName = $sampleFile->name;
        
		if($sampleFile->extension && !StringUtils::endsWith($downloadFileName, $sampleFile->extension)) {
			$downloadFileName .= "." . $sampleFile->extension;
		}

        $downloadFileSize = filesize($file_path);
        
        $mime_types = array(
	        "htm" => "text/html",
	        "exe" => "application/octet-stream",
	        "zip" => "application/zip",
	        "doc" => "application/msword",
	        "docx" => "application/msword",
	        "jpg" => "image/jpg",
	        "php" => "text/plain",
	        "xls" => "application/vnd.ms-excel",
	        "ppt" => "application/vnd.ms-powerpoint",
	        "gif" => "image/gif",
	        "pdf" => "application/pdf",
	        "txt" => "text/plain",
	        "html"=> "text/html",
	        "png" => "image/png",
	        "jpeg"=> "image/jpg",
			"js" => "application/x-javascript"
	    );

		if( isset($sampleFile->extension, $mime_types) ) {
            $mime_type = $mime_types[$sampleFile->extension];
        } else {
            $mime_type = "application/force-download";
        }

        @ob_end_clean();
        if( ini_get('zlib.output_compression') ) {
        	ini_set('zlib.output_compression', 'Off');
        }

        header('Pragma: public');
        header("Expires: 0");
        header('Connection: Keep-Alive');
        header("Cache-Control: public, must-revalidate, post-check=0, pre-check=0");
	    header('Content-Type: ' . $mime_type);
	    header('Content-Disposition: attachment; filename="'.$downloadFileName.'"');
	    header("Content-Transfer-Encoding: binary");
	    header('Accept-Ranges: bytes');

	    if( isset($_SERVER['HTTP_RANGE']) ) {
	        list($a, $range) = explode("=",$_SERVER['HTTP_RANGE'],2);
	        list($range) = explode(",",$range,2);
	        list($range, $range_end) = explode("-", $range);
	        $range=intval($range);
	        if(!$range_end) {
	            $range_end=$downloadFileSize-1;
	        } else {
	            $range_end=intval($range_end);
	        }

	        $new_length = $range_end-$range+1;
	        header("HTTP/1.1 206 Partial Content");
	        header("Content-Length: $new_length");
	        header("Content-Range: bytes $range-$range_end/$downloadFileSize");
	    } else {
	        $new_length = $downloadFileSize;
	        header("Content-Length: " . $downloadFileSize);
	    }

	    $chunksize = 1*(1024*1024);
	    $bytes_sent = 0;

        if(isset($_SERVER['HTTP_RANGE'])) {
	        fseek($downloadFile, $range);
	    }

        while(
        	!feof($downloadFile) && 
        	(!connection_aborted()) && 
        	($bytes_sent < $new_length) 
        ) {
            $buffer = fread($downloadFile, $chunksize);
            echo($buffer);
            flush();
            $bytes_sent += strlen($buffer);
        }

	    @fclose($downloadFile);
	    return;
    }

    public function apiAddSample() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        $new_sample = true;
        if(ArrayUtils::has($payload, 'sample_id')) {
            $sample_id = ArrayUtils::get($payload, 'sample_id');
            $sample = new Sample( (int) $sample_id );
            if(!Validate::isLoadedObject($sample)) {
                return $app->sendResponse([
                    'error' => true,
                    'message' => "This sample does not exist"
                ]);
            }
            $new_sample = false;
        } else {
            $sample = new Sample();
        }

        $title = ArrayUtils::get($payload, 'title');
        if (!$title) {
			return $app->sendResponse([
                "error" => true,
                "message" => "Enter the title for your sample"
            ]);
        }

        $instructions = ArrayUtils::get($payload, 'instructions');
        if(!$instructions) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter the instructions of this sample"
            ]);
        }

        /** Pages / Charts / Slides */
        $pages = (int) ArrayUtils::get($payload, 'pages');
        $sources = (int) ArrayUtils::get($payload, 'sources');

        // @todo both pages, charts, and slides can't be zero
		if ( $pages == 0) {
			return $app->sendResponse([
                "error" => true,
                "message" => "You need to add at least 1 page"
            ]);
        }

        /** Spacing */
        $spacing = ArrayUtils::get($payload, 'spacing');
        if ( !$spacing || !in_array($spacing, array(Sample::SPACING_SINGLE, Sample::SPACING_DOUBLE)) ) {
			return $app->sendResponse([
                "error" => true,
                "message" => "Select the spacing for this sample"
            ]);
        }

        /** Topic category */
        $topic_cat_id = (int) ArrayUtils::get($payload, 'topicCatId');
        $discipline = new Discipline( $topic_cat_id );
        if(Validate::isLoadedObject($discipline)) {
            $topic_cat_option = ArrayUtils::get($payload, 'topcatOption', $discipline->title);
			$isComplexAssignment = $discipline->is_complex_assignment;
        }

        /** Paper Type */
        $paper_type_id = (int) ArrayUtils::get($payload, 'paperTypeId');
        $paperType = new PaperType( $paper_type_id );
        if(Validate::isLoadedObject($paperType)) {
            $paper_type_option = ArrayUtils::get($payload, 'paperTypeOption', $paperType->title);
        }

        /** Paper Format */
        $paper_format_id = (int) ArrayUtils::get($payload, 'paperFormat');

        $paperFormat = new PaperFormat( $paper_format_id  );
        if(Validate::isLoadedObject($paperFormat)) {
            $paper_format_option = $paperFormat->title;
            $paper_format_option = ArrayUtils::get($payload, 'paperFormatOption', $paperFormat->title);
        }

        $academic_level_id = ArrayUtils::get($payload, 'academicLevelId');

        $slug = ArrayUtils::get($payload, 'slug');
        if($slug) {
            $sample->slug = Formatting::sanitize_title_with_dashes( $slug , null, 'save');
        } else {
            $sample->slug = Formatting::sanitize_title_with_dashes($title , null, 'save');
        }

        $sample->title = $title;
        $sample->instructions = $instructions;
        $sample->academic_level_id = $academic_level_id;
        $sample->paper_type_id = $paper_type_id;
        $sample->paper_type_option = $paper_type_option;
        $sample->topic_category_id = $topic_cat_id;
        $sample->topic_category_option = $topic_cat_option;
        $sample->is_complex_assignment = $isComplexAssignment;
        $sample->paper_format_id = $paper_format_id;
        $sample->paper_format_option = $paper_format_option;
        $sample->sources = $sources;
        $sample->pages = $pages;
        $sample->spacing = $spacing;

        if($new_sample) {
            $sample->add();

            return $app->sendResponse([
                'callback' => 'window.location.href="/samples/viewSample?sampleId='.$sample->id.'";'
            ]);
        } else {
            $sample->update();

            return $app->sendResponse([
                'success' => true,
                'message' => 'Sample info has been updated'
            ]);
        }
    }

    public function hookActionAfterFileUpload($payload) {
        $handle = ArrayUtils::get($payload, 'handle');
        $id = ArrayUtils::get($payload, 'id');
        $file = ArrayUtils::get($payload, 'file');

        switch($handle) {
            case 'sample':
                $sample = new Sample( (int) $id);
                if(Validate::isLoadedObject($sample)) {
                    $sampleFile = new SampleFile();
                    $sampleFile->sample_id = $sample->id;
                    $sampleFile->name = $file['name'];
                    $sampleFile->size = $file['size'];
                    $sampleFile->source = $file['source'];
                    $sampleFile->extension = $file['extension'];
                    $sampleFile->add();
                }
                break;
        }
    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        switch($handle) {
            case "sample":
                $sample = new Sample( (int) $id );
                if(Validate::isLoadedObject($sample)) {
                    $sample->delete();
                }
                break;

            case "sample-file":
                $sampleFile = new SampleFile( (int) $id );
                if(Validate::isLoadedObject($sampleFile)) {
                    $sampleFile->delete();
                }
                break;
        }
    }
}
