<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Database\DbQuery;
use Proxim\Application;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\Formatting;

/**
 * Sample
 */
class Sample extends ObjectModel
{
    const START_RESOURCE_ID = 3422;

    const SPACING_SINGLE = 'single';
	const SPACING_DOUBLE = 'double';

    /** @var $id Sample ID */
    public $id;

    /** @var string title */
    public $title;

    /** @var string instructions */
    public $instructions;

    /** @var int academic_level_id */
    public $academic_level_id;

    /** @var int paper_type_id */
    public $paper_type_id;

    /** @var string paper_type_option */
    public $paper_type_option;

    /** @var int topic_category_id */
    public $topic_category_id;

    /** @var string topic_category_option */
    public $topic_category_option;

    /** @var bool is_complex_assignment */
    public $is_complex_assignment = 0;

    /** @var int paper_format_id */
    public $paper_format_id;

    /** @var string paper_format_option */
    public $paper_format_option;

    /** @var int pages */
    public $pages = 0;

    /** @var int sources */
    public $sources = 0;

    /** @var string spacing */
    public $spacing = self::SPACING_DOUBLE;

    /** @var int views */
    public $views = 0;

    /** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
    public $date_add;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'sample',
        'primary' => 'sample_id',
        'fields' => array(
            'title' => array('type' => self::TYPE_STRING, 'required' => true),
            'instructions' => array('type' => self::TYPE_HTML, 'required' => true),
            'academic_level_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_option' => array('type' => self::TYPE_STRING),
            'topic_category_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'topic_category_option' => array('type' => self::TYPE_STRING),
            'is_complex_assignment' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'paper_format_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_format_option' => array('type' => self::TYPE_STRING),
            'pages' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'sources' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'spacing' => array('type' => self::TYPE_STRING),
            'views' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        )
    );
    
    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null)
    {
        parent::__construct($id);
    }

    public function add($autodate = true, $null_values = true) {
		if ( !$this->getNextSampleId() ) {
			$this->id = self::START_RESOURCE_ID;
		} else {
			$next_sample_id = Db::getInstance()->getValue('SELECT MAX(`sample_id`)+2 FROM ' . Db::prefix('sample'));
			$this->id = $next_sample_id;
        }

		$this->force_id = true;

		return parent::add($autodate, $null_values);
    }

    public function delete() {
        $success = parent::delete();
        Db::getInstance()->delete('sample_file', 'sample_id = ' . (int) $this->id );
    }

    /**
     * used to cache sample files.
     */
    protected $sampleFiles = null;

    public function getFiles() {
        if (is_null($this->sampleFiles)) {
            $sql = new DbQuery();
            $sql->select('pf.sample_file_id');
            $sql->from('sample_file', 'pf');
            $sql->where('pf.`sample_id` = ' . (int) $this->id );
            $sql->orderBy('pf.sample_file_id');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }

            $this->sampleFiles = $result;
        }

		return $this->sampleFiles;
    }
    
    /**
     * This method return the ID of the previous sample.
     *
     * @since 1.5.0.1
     *
     * @return int
     */
    public function getPreviousSampleId()
    {
        return Db::getInstance()->getValue('
            SELECT sample_id
            FROM ' . DB_PREFIX . 'sample
			WHERE sample_id < ' . (int) $this->id . '
            ORDER BY sample_id DESC');
    }
	
	/**
     * This method return the ID of the next sample.
     *
     * @since 1.0.0
     *
     * @return int
     */
    public function getNextSampleId()
    {
        return Db::getInstance()->getValue('
            SELECT sample_id
            FROM ' . DB_PREFIX . 'sample
			WHERE sample_id > ' . (int) $this->id . '
            ORDER BY sample_id ASC');
    }
}