<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Order\File;
use Proxim\Order\Order;
use Proxim\Pager;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\Discipline;
use Proxim\Preference\PaperFormat;
use Proxim\Preference\PaperType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Site\Site;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\Formatting;
use Proxim\Util\StringUtils;
use Proxim\Validate;

define('CURRENT_PWP_MODULE_DIR', realpath(dirname(__FILE__)));

require_once(CURRENT_PWP_MODULE_DIR . '/classes/PrePaper/PrePaper.php');
require_once(CURRENT_PWP_MODULE_DIR . '/classes/PrePaper/PrePaperFile.php');
require_once(CURRENT_PWP_MODULE_DIR . '/classes/PrePaper/PrePaperPreview.php');
require_once(CURRENT_PWP_MODULE_DIR . '/classes/PrePaper/PrePaperPayment.php');
require_once(CURRENT_PWP_MODULE_DIR . '/classes/Presenter/PrePaperLazyArray.php');
require_once(CURRENT_PWP_MODULE_DIR . '/classes/Presenter/PrePaperPresenter.php');

class PrePapers extends Module
{
    const SELL_PREWRITTEN_PAPERS = 'SELL_PREWRITTEN_PAPERS';
    const PREWRITTEN_PAPERS_TITLE = 'PREWRITTEN_PAPERS_TITLE';
    const PREWRITTEN_PAPERS_DESCRIPTION = 'PREWRITTEN_PAPERS_DESCRIPTION';
    const PREWRITTEN_PAPERS_HEADER_COLOR = 'PREWRITTEN_PAPERS_HEADER_COLOR';
    const PREWRITTEN_PAPERS_NAVLINK_COLOR = 'PREWRITTEN_PAPERS_NAVLINK_COLOR';
    const PREWRITTEN_PAPERS_BUTTON_PRIMARY = 'PREWRITTEN_PAPERS_BUTTON_PRIMARY';
    const PREWRITTEN_PAPERS_BUTTON_SECONDARY = 'PREWRITTEN_PAPERS_BUTTON_SECONDARY';

    public function __construct()
    {
        $this->name = 'prepapers';
        $this->icon = 'fa fa-book';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Pre Written Papers';
        $this->description = 'Sell your Pre Written Papers all from one place';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return ($user->isLogged() && ($user->is_admin || $user->is_sub_admin)) ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        } 

        $this->registerHook([
            'displayAdminNavBarBeforeEnd',
            'displayAdminEditSiteGeneralTabContent',
            'actionAfterSiteSettingsGeneralUpdate',
            'actionBeforeAdminDelete',
            'actionAfterFileUpload',
            'displayOrderActions'
        ]);
    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('prepaper') . " (
                `prepaper_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `slug` TEXT NOT NULL,  
                `title` TEXT NOT NULL,
                `question` LONGTEXT DEFAULT NULL,
                `academic_level_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_type_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_type_option` TEXT DEFAULT NULL,
                `topic_category_id` BIGINT(20) UNSIGNED NOT NULL,
                `topic_category_option` TEXT DEFAULT NULL,
                `is_complex_assignment` TINYINT(1) UNSIGNED NOT NULL DEFAULT '0',
                `paper_format_id` BIGINT(20) UNSIGNED NOT NULL,
                `paper_format_option` TEXT DEFAULT NULL,
                `pages` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `words` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `charts` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `slides` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `sources` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `spacing` enum('single','double') NOT NULL DEFAULT 'double',
                `price` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `sales` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `views` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `downloads` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`prepaper_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('prepaper_file') . " (
                `prepaper_file_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `prepaper_id` BIGINT(20) UNSIGNED NOT NULL,
                `name` VARCHAR(255) NOT NULL,
                `size` INT(10) NOT NULL DEFAULT 0,
                `source` TEXT NOT NULL,
                `extension` VARCHAR(20) DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`prepaper_file_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('prepaper_preview') . " (
                `prepaper_preview_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `prepaper_id` BIGINT(20) UNSIGNED NOT NULL,
                `name` VARCHAR(255) NOT NULL,
                `size` INT(10) NOT NULL DEFAULT 0,
                `source` TEXT NOT NULL,
                `extension` VARCHAR(20) DEFAULT NULL,
                `order` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`prepaper_preview_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('prepaper_payment') . " (
                `prepaper_payment_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `prepaper_id` BIGINT(20) UNSIGNED NOT NULL,
                `customer_id` BIGINT(20) UNSIGNED NOT NULL,
                `payment_method` VARCHAR(255) DEFAULT NULL,
                `payment_token` VARCHAR(255) DEFAULT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `is_paid` INT(10) UNSIGNED NOT NULL DEFAULT 0,
                `paid_at` DATETIME DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`prepaper_payment_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        try {
            Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('prepaper') . " ADD (
                    `order_id` BIGINT(20) UNSIGNED DEFAULT NULL
                );

                ALTER TABLE " . Db::prefix('prepaper_payment') . " ADD (
                    `order_id` BIGINT(20) UNSIGNED DEFAULT NULL
                );
            ");
        } catch(Exception $exception) {

        }

        try {
            Db::getInstance()->Execute("
                ALTER TABLE " . Db::prefix('prepaper_payment') . " ADD (
                    `order_id` BIGINT(20) UNSIGNED DEFAULT NULL
                );
            ");
        } catch(Exception $exception) {

        }

        return true;
    }
 
    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminNavBarBeforeEnd() {
        if($this->checkAccess()) {
            return $this->showTemplate('nav/nav_item');
        }
    }

    public function hookDisplayAdminEditSiteGeneralTabContent() {
        $application = $this->application;
        $smarty = $this->smarty;

        $prepapersFormValues = Configuration::getMultiple([
                self::SELL_PREWRITTEN_PAPERS,
                self::PREWRITTEN_PAPERS_TITLE,
                self::PREWRITTEN_PAPERS_DESCRIPTION,
                self::PREWRITTEN_PAPERS_HEADER_COLOR,
                self::PREWRITTEN_PAPERS_NAVLINK_COLOR,
                self::PREWRITTEN_PAPERS_BUTTON_PRIMARY,
                self::PREWRITTEN_PAPERS_BUTTON_SECONDARY
            ],
            $application->site->id
        );
        $smarty->assign('prepapersFormValues', $prepapersFormValues);
        
        return $this->showTemplate('nav/tab_content');
    }

    public function assignVairiables() {
        $smarty = $this->smarty;

        $objectPresenter = new ObjectPresenter();

        $disciplineGroups = Discipline::getDisciplinesByGroup();
        $paperFormats = PaperFormat::getPaperFormats();
        $paperTypesPreferences = PaperType::getPaperTypes();

        $paperTypes = array();
        foreach($paperTypesPreferences as $paperType) {
            $paperType = new PaperType( $paperType['paper_type_id'] );
            if(Validate::isLoadedObject($paperType)) {
                $paperTypes[] = $objectPresenter->present($paperType);
            }
        }

        $academicLevels = AcademicLevel::getAcademicLevels();

        $smarty->assign([
            'academicLevels' => $academicLevels,
            'paperTypes' => $paperTypes,
            'paperFormats' => $paperFormats,
            'disciplineGroups' => $disciplineGroups,
        ]);
    }

    public function hookDisplayOrderActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $order_id = ArrayUtils::get($params, 'order_id');

        if($order_id && $user->is_admin || $user->is_sub_admin) {
            $order = new Order( (int) $order_id );
            if(Validate::isLoadedObject($order)) {
                $can_import_order = $order->status_id == Order::FINISHED ? true : false;
                $order_imported = false;

                $prepaper = new PrePaper();
                $prepaper->getByOrderId($order->id);
                if(Validate::isLoadedObject($prepaper)) {
                    $order_imported = true;
                }

                $smarty->assign([
                    'order_id' => $order->id,
                    'order_imported' => $order_imported,
                    'can_import_order' => $can_import_order
                ]);

                return $this->showTemplate('actions.import');
            }
        }
    }

    public function import() {
        $app = $this->application;
        $controller = $app->controller;
        $order_id = $app->request->get('order_id');

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            $controller->setTemplate('404');
            $controller->display();
        }

        $spacing = $order->spacing == "single" ? 2 : 1;
        $prepaper = new PrePaper();
        $prepaper = $prepaper->getByOrderId( $order->id );

        $update = false;
        if(Validate::isLoadedObject($prepaper)) {
            $update = true;
        } else {
            $prepaper = new PrePaper();
        }

        $prepaper->order_id = $order->id;
        $prepaper->title = $order->title;
        $prepaper->question = Tools::nl2br($order->paper_details);
        $prepaper->academic_level_id = $order->academic_level_id;
        $prepaper->paper_type_id = (int) $order->paper_type_id;
        $prepaper->paper_type_option = $order->paper_type_option;
        $prepaper->topic_category_id = (int) $order->topic_category_id;
        $prepaper->topic_category_option = $order->topic_category_option;
        $prepaper->is_complex_assignment = $order->is_complex_assignment;
        $prepaper->paper_format_id = (int) $order->paper_format_id;
        $prepaper->paper_format_option = $order->paper_format_option;
        $prepaper->sources = (int) $order->sources;
        $prepaper->words = ($order->pages*275)*$spacing;
        $prepaper->pages = (int) $order->pages;
        $prepaper->charts = (int) $order->charts;
        $prepaper->slides = (int) $order->slides;
        $prepaper->spacing = $order->spacing;
        $prepaper->price = (float) $order->total;

        $orderFiles = $order->getFiles();

        if($update) {
            $prepaper->update();
        } else {
            $prepaper->slug = Formatting::sanitize_title_with_dashes($order->title , null, 'save');
            $slugExists = (new PrePaper())->getBySlug( $prepaper->slug, $prepaper->id );
            if(Validate::isLoadedObject($slugExists)) {
                $prepaper->slug = $prepaper->slug . '-' . Tools::randomGen(5, 'NUMERIC');
            }
            
            $prepaper->add();

            foreach($orderFiles as $orderFile) {
                $orderFile = new File( $orderFile['file_id'] );
                if(Validate::isLoadedObject($orderFile)) {
                    $prepaperFile = new PrePaperFile();
                    $prepaperFile->prepaper_id = $prepaper->id;
                    $prepaperFile->name = $orderFile->name;
                    $prepaperFile->size = $orderFile->size;
                    $prepaperFile->source = $orderFile->source;
                    $prepaperFile->extension = $orderFile->extension;
                    $prepaperFile->add();
                }
            }
        }

        $app->redirect('/prepapers/viewPaper?paperId='.$prepaper->id);
        return;
    }

    public function listPapers() {
        $app = $this->application;
        $smarty = $this->smarty;
        $params = $app->request->get();

        $is_search = ArrayUtils::has($params, 'search') ? true : false;
        $selected_page = ArrayUtils::has($params, 'page') ? (int) ArrayUtils::get($params, 'page') : 1;
        $paper_id = ArrayUtils::get($params, 'paper_id');
        $order_id = ArrayUtils::get($params, 'order_id');
        $title = ArrayUtils::get($params, 'toptitle');

        $sql = new DbQuery();
        $sql->select('DISTINCT p.*');
        $sql->from('prepaper', 'p');

        if ($is_search) {
            if ($paper_id) {
                $sql->where('p.prepaper_id LIKE \'%' . pSQL($paper_id) . '%\'');
            } 

            if ($order_id) {
                $sql->where('p.order_id LIKE \'%' . pSQL($order_id) . '%\'');
            } 

            if ($title) {
                $sql->where('p.title LIKE \'%' . pSQL($title) . '%\'');
            }
        }

        $sql->orderBy('p.prepaper_id DESC');
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);
        $total_prepapers = count($result);

        $prepapers = array();

        if( $total_prepapers > 0) {
            $params['total_items'] = $total_prepapers;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/prepapers/listPapers?page=%s";

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);
            
            $prepaperPresenter = new PrePaperPresenter(); 
            foreach( $result as $prepaper ) {
                $ppr = new PrePaper();
                $prepaper = $ppr->getObject( $prepaper );
                $prepapers[] = $prepaperPresenter->present( $prepaper );
            } 

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'prepapers',
            'sub_view' => 'listPapers',
            'toptitle' => $title,
            'paper_id' => $paper_id,
            'order_id' => $order_id,
            'toptitle' => $title,
            'prepapers' => $prepapers,
            'is_search' => $is_search
        ]);

        return $this->showTemplate('prepapers');
    }

    public function addNew() {
        $app = $this->application;
        $smarty = $this->smarty;

        $this->assignVairiables();

        $smarty->assign([
            'view' => 'prepapers',
            'sub_view' => 'addPaper'
        ]);

        return $this->showTemplate('prepapers');
    }

    public function viewPaper() {
        $app = $this->application;
        $user = $app->user;
        $controller = $app->user;
        $smarty = $this->smarty;
        $paperId = $app->request->get('paperId');

        $prepaper = new PrePaper( (int) $paperId );
        if(!Validate::isLoadedObject($prepaper)) {
            $controller->setTemplate('404');
            $controller->display();
        }

        $this->assignVairiables();

        $prepaperPresenter = new PrePaperPresenter();

        $smarty->assign([
            'view' => 'prepapers',
            'sub_view' => 'viewPaper',
            'prepaper' => $prepaperPresenter->present($prepaper)
        ]);

        return $this->showTemplate('prepapers');
    }

     public function apiAddNewPaper() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        $new_prepaper = true;

        if(ArrayUtils::has($payload, 'prepaper_id')) {
            $prepaper_id = ArrayUtils::get($payload, 'prepaper_id');
            $prepaper = new PrePaper( (int) $prepaper_id );
            if(!Validate::isLoadedObject($prepaper)) {
                return $app->sendResponse([
                    'error' => true,
                    'message' => "This prepaper does not exist"
                ]);
            }
            $new_prepaper = false;
        } else {
            $prepaper = new PrePaper();
        }

        $title = ArrayUtils::get($payload, 'title');
        if (!$title) {
			return $app->sendResponse([
                "error" => true,
                "message" => "Enter the title for your prepaper"
            ]);
        }

        $question = ArrayUtils::get($payload, 'question');
        if(!$question) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Enter the question of this prepaper"
            ]);
        }

        $price = ArrayUtils::get($payload, 'price');
        if (!$price) {
			return $app->sendResponse([
                "error" => true,
                "message" => "Enter the price for your prepaper"
            ]);
        }

        /** Pages / Charts / Slides */
        $words = (int) ArrayUtils::get($payload, 'words'); 
        $pages = (int) ArrayUtils::get($payload, 'pages'); 
        $charts = (int) ArrayUtils::get($payload, 'charts'); 
        $slides = (int) ArrayUtils::get($payload, 'slides');
        $sources = (int) ArrayUtils::get($payload, 'sources');

        // @todo both pages, charts, and slides can't be zero
		if ( $pages == 0 && $slides == 0 && $charts == 0) {
			return $app->sendResponse([
                "error" => true,
                "message" => "You need to add at least 1 page or 1 slide or 1 chart"
            ]);
        }

        /** Spacing */
        $spacing = ArrayUtils::get($payload, 'spacing'); 
        if ( !$spacing || !in_array($spacing, array(PrePaper::SPACING_SINGLE, PrePaper::SPACING_DOUBLE)) ) {
			return $app->sendResponse([
                "error" => true,
                "message" => "Select the spacing for this prepaper"
            ]);
        }

        /** Topic category */
        $topic_cat_id = (int) ArrayUtils::get($payload, 'topicCatId');
        $discipline = new Discipline( $topic_cat_id );
        if(Validate::isLoadedObject($discipline)) {
            $topic_cat_option = ArrayUtils::get($payload, 'topcatOption', $discipline->title);
			$isComplexAssignment = $discipline->is_complex_assignment;
        }
        
        /** Paper Type */
        $paper_type_id = (int) ArrayUtils::get($payload, 'paperTypeId');
        $paperType = new PaperType( $paper_type_id );
        if(Validate::isLoadedObject($paperType)) {
            $paper_type_option = ArrayUtils::get($payload, 'paperTypeOption', $paperType->title);
        }

        /** Paper Format */
        $paper_format_id = (int) ArrayUtils::get($payload, 'paperFormat');

        $paperFormat = new PaperFormat( $paper_format_id  );
        if(Validate::isLoadedObject($paperFormat)) {
            $paper_format_option = $paperFormat->title;
            $paper_format_option = ArrayUtils::get($payload, 'paperFormatOption', $paperFormat->title);
        }

        $academic_level_id = ArrayUtils::get($payload, 'academicLevelId');

        $slug = ArrayUtils::get($payload, 'slug');
        if($slug) {
            $prepaper->slug = Formatting::sanitize_title_with_dashes( $slug , null, 'save');
        } else {
            $prepaper->slug = Formatting::sanitize_title_with_dashes($title , null, 'save');
        }

        $slugExists = (new PrePaper())->getBySlug( $prepaper->slug, $prepaper->id );
        if(Validate::isLoadedObject($slugExists)) {
            $prepaper->slug = $prepaper->slug . '-' . Tools::randomGen(5, 'NUMERIC');
        }

        $prepaper->title = $title;
        $prepaper->question = $question;
        $prepaper->academic_level_id = $academic_level_id;
        $prepaper->paper_type_id = $paper_type_id;
        $prepaper->paper_type_option = $paper_type_option;
        $prepaper->topic_category_id = $topic_cat_id;
        $prepaper->topic_category_option = $topic_cat_option;
        $prepaper->is_complex_assignment = $isComplexAssignment;
        $prepaper->paper_format_id = $paper_format_id;
        $prepaper->paper_format_option = $paper_format_option;
        $prepaper->sources = $sources;
        $prepaper->words = $words;
        $prepaper->pages = $pages;
        $prepaper->charts = $charts;
        $prepaper->slides = $slides;
        $prepaper->spacing = $spacing;
        $prepaper->price = (float) $price;

        if($new_prepaper) {
            $prepaper->add();

            return $app->sendResponse([
                'callback' => 'window.location.href="/prepapers/viewPaper?paperId='.$prepaper->id.'";'
            ]);
        } else {
            $prepaper->update();

            return $app->sendResponse([
                'success' => true,
                'message' => 'PrePaper info has been updated'
            ]);
        }
    }

    public function hookActionAfterFileUpload($payload) {
        $handle = ArrayUtils::get($payload, 'handle');
        $id = ArrayUtils::get($payload, 'id');
        $file = ArrayUtils::get($payload, 'file');

        switch($handle) {
            case 'prepaper':
                $prepaper = new PrePaper( (int) $id);
                if(Validate::isLoadedObject($prepaper)) {
                    $prepaperFile = new PrePaperFile();
                    $prepaperFile->prepaper_id = $prepaper->id;
                    $prepaperFile->name = $file['name'];
                    $prepaperFile->size = $file['size'];
                    $prepaperFile->source = $file['source'];
                    $prepaperFile->extension = $file['extension'];
                    $prepaperFile->add();
                }
                break;

            case 'prepaper-preview':
                $prepaper = new PrePaper( (int) $id );
                if(Validate::isLoadedObject($prepaper)) {
                    $prepaperPreview = new PrePaperPreview();
                    $prepaperPreview->prepaper_id = $prepaper->id;
                    $prepaperPreview->name = $file['name'];
                    $prepaperPreview->size = $file['size'];
                    $prepaperPreview->source = $file['source_name'];
                    $prepaperPreview->extension = $file['extension'];
                    $prepaperPreview->add();
                }
                break;
        }
    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        switch($handle) {
            case "prepaper":
                $prepaper = new PrePaper( (int) $id );
                if(Validate::isLoadedObject($prepaper)) {
                    $prepaper->delete();
                }
                break;

            case "prepaper-file":
                $prepaperFile = new PrePaperFile( (int) $id );
                if(Validate::isLoadedObject($prepaperFile)) {
                    $prepaperFile->delete();
                }
                break;

            case 'prepaper-preview':
                $prepaperPreview = new PrePaperPreview( (int) $id );
                if(Validate::isLoadedObject($prepaperPreview)) {
                    $prepaperPreview->delete();
                }
                break;
        }
    }

    public function hookActionAfterSiteSettingsGeneralUpdate( $params ) {
        $site_id = ArrayUtils::get($params, 'site_id');
        $payload = ArrayUtils::get($params, 'payload');

        $site = new Site( (int) $site_id );
        if(Validate::isLoadedObject($site)) {
            Configuration::updateValue(self::SELL_PREWRITTEN_PAPERS, ArrayUtils::has($payload, 'sellPrewrittenPapers') ? true : false, false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_TITLE, ArrayUtils::get($payload, 'prepapers_title'), false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_DESCRIPTION, ArrayUtils::get($payload, 'prepapers_description'), false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_HEADER_COLOR, ArrayUtils::get($payload, 'prepapers_header_color'), false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_NAVLINK_COLOR, ArrayUtils::get($payload, 'prepapers_navlink_color'), false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_BUTTON_PRIMARY, ArrayUtils::get($payload, 'prepapers_btn_primary_color'), false, $site->id);
            Configuration::updateValue(self::PREWRITTEN_PAPERS_BUTTON_SECONDARY, ArrayUtils::get($payload, 'prepapers_btn_secondary_color'), false, $site->id);
        }
    }

    public function downloadPaper() {
        $app = $this->application;
        $controller = $app->controller;
        $fileId = $app->request->get('fileId');

        $prepaperFile = new PrePaperFile( (int) $fileId );
        if(!Validate::isLoadedObject($prepaperFile)) {
            $controller->setTemplate('404');
            $controller->display();
        }

        $file_path = PROX_DIR_UPLOADS . $prepaperFile->source;

		$downloadFileName = $prepaperFile->name;
		if($prepaperFile->extension && StringUtils::endsWith($downloadFileName, $prepaperFile->extension)) {
			$downloadFileName .= "." . $prepaperFile->extension;
		}

		$downloadFile = @fopen($file_path,"rb");
		if (!$downloadFile) {
            $controller->setTemplate('404');
            $controller->display();
		}

        $downloadFileName = $prepaperFile->name;
        
		if($prepaperFile->extension && !StringUtils::endsWith($downloadFileName, $prepaperFile->extension)) {
			$downloadFileName .= "." . $prepaperFile->extension;
		}

        $downloadFileSize = filesize($file_path);
        
        $mime_types = array(
	        "htm" => "text/html",
	        "exe" => "application/octet-stream",
	        "zip" => "application/zip",
	        "doc" => "application/msword",
	        "docx" => "application/msword",
	        "jpg" => "image/jpg",
	        "php" => "text/plain",
	        "xls" => "application/vnd.ms-excel",
	        "ppt" => "application/vnd.ms-powerpoint",
	        "gif" => "image/gif",
	        "pdf" => "application/pdf",
	        "txt" => "text/plain",
	        "html"=> "text/html",
	        "png" => "image/png",
	        "jpeg"=> "image/jpg",
			"js" => "application/x-javascript"
	    );

		if( isset($prepaperFile->extension, $mime_types) ) {
            $mime_type = $mime_types[$prepaperFile->extension];
        } else {
            $mime_type = "application/force-download";
        }

        @ob_end_clean();
        if( ini_get('zlib.output_compression') ) {
        	ini_set('zlib.output_compression', 'Off');
        }

        header('Pragma: public');
        header("Expires: 0");
        header('Connection: Keep-Alive');
        header("Cache-Control: public, must-revalidate, post-check=0, pre-check=0");
	    header('Content-Type: ' . $mime_type);
	    header('Content-Disposition: attachment; filename="'.$downloadFileName.'"');
	    header("Content-Transfer-Encoding: binary");
	    header('Accept-Ranges: bytes');

	    if( isset($_SERVER['HTTP_RANGE']) ) {
	        list($a, $range) = explode("=",$_SERVER['HTTP_RANGE'],2);
	        list($range) = explode(",",$range,2);
	        list($range, $range_end) = explode("-", $range);
	        $range=intval($range);
	        if(!$range_end) {
	            $range_end=$downloadFileSize-1;
	        } else {
	            $range_end=intval($range_end);
	        }

	        $new_length = $range_end-$range+1;
	        header("HTTP/1.1 206 Partial Content");
	        header("Content-Length: $new_length");
	        header("Content-Range: bytes $range-$range_end/$downloadFileSize");
	    } else {
	        $new_length = $downloadFileSize;
	        header("Content-Length: " . $downloadFileSize);
	    }

	    $chunksize = 1*(1024*1024);
	    $bytes_sent = 0;

        if(isset($_SERVER['HTTP_RANGE'])) {
	        fseek($downloadFile, $range);
	    }

        while(
        	!feof($downloadFile) && 
        	(!connection_aborted()) && 
        	($bytes_sent < $new_length) 
        ) {
            $buffer = fread($downloadFile, $chunksize);
            echo($buffer);
            flush();
            $bytes_sent += strlen($buffer);
        }

	    @fclose($downloadFile);
	    return;
    }

}