<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Database\DbQuery;
use Proxim\Application;
use Proxim\ObjectModel;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;
use Proxim\Tools;
use Proxim\Util\Formatting;

/**
 * PrePaper
 */
class PrePaper extends ObjectModel
{
    const START_RESOURCE_ID = 2000;

    const SPACING_SINGLE = 'single';
	const SPACING_DOUBLE = 'double';

    /** @var $id PrePaper ID */
    public $id;

    /** @var int order_id */
    public $order_id;
    
    /** @var string slug */
    public $slug;

    /** @var string title */
    public $title;

    /** @var string question */
    public $question;

    /** @var int academic_level_id */
    public $academic_level_id;

    /** @var int paper_type_id */
    public $paper_type_id;

    /** @var string paper_type_option */
    public $paper_type_option;

    /** @var int topic_category_id */
    public $topic_category_id;

    /** @var string topic_category_option */
    public $topic_category_option;

    /** @var bool is_complex_assignment */
    public $is_complex_assignment = 0;

    /** @var int paper_format_id */
    public $paper_format_id;

    /** @var string paper_format_option */
    public $paper_format_option;

    /** @var int words */
    public $words = 0;

    /** @var int pages */
    public $pages = 0;

    /** @var int charts */
    public $charts = 0;

    /** @var int slides */
    public $slides = 0;

    /** @var int sources */
    public $sources = 0;

    /** @var string spacing */
    public $spacing = self::SPACING_DOUBLE;

    /** @var float price */
    public $price = 0;

    /** @var int downloads */
    public $downloads = 0;

    /** @var int views */
    public $views = 0;

    /** @var float sales */
    public $sales = 0;

    /** @var string date_upd */
    public $date_upd;

    /** @var string date_add */
    public $date_add;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'prepaper',
        'primary' => 'prepaper_id',
        'fields' => array(
            'order_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'slug' => array('type' => self::TYPE_STRING, 'required' => true),
            'title' => array('type' => self::TYPE_STRING, 'required' => true),
            'question' => array('type' => self::TYPE_HTML, 'required' => true),
            'academic_level_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_type_option' => array('type' => self::TYPE_STRING),
            'topic_category_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'topic_category_option' => array('type' => self::TYPE_STRING),
            'is_complex_assignment' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'paper_format_id' => array('type' => self::TYPE_INT, 'validate' => 'isUnsignedInt'),
            'paper_format_option' => array('type' => self::TYPE_STRING),
            'words' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'pages' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'charts' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'slides' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'sources' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'spacing' => array('type' => self::TYPE_STRING),
            'price' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'views' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'downloads' => array('type' => self::TYPE_INT, 'validate' => 'isInt'),
            'sales' => array('type' => self::TYPE_FLOAT, 'validate' => 'isPrice'),
            'date_upd' => array('type' => self::TYPE_DATE, 'validate' => 'isDateOrNull'),
            'date_add' => array('type' => self::TYPE_DATE, 'validate' => 'isDate'),
        )
    );
    
    /**
     * constructor.
     *
     * @param null $id
     */
    public function __construct($id = null, $fullInfo = false)
    {
        parent::__construct($id);
    }

    public function add($autodate = true, $null_values = true) {
		if ( !$this->getNextPrePaperId() ) {
			$this->id = self::START_RESOURCE_ID;
		} else {
			$next_prepaper_id = Db::getInstance()->getValue('SELECT MAX(`prepaper_id`)+2 FROM ' . Db::prefix('prepaper'));
			$this->id = $next_prepaper_id;
        }

        if(!$this->words) {
            $wordsPerPage = $this->spacing == self::SPACING_DOUBLE ? 275 : 550;
            $this->words = $wordsPerPage*$this->pages;
        }

		$this->force_id = true;

		return parent::add($autodate, $null_values);
    }

    public function getObject( $prepaper = array() ) {
        if( !ArrayUtils::has($prepaper, 'prepaper_id') ) {
            return $this;
        }

        $this->id = $prepaper['prepaper_id'];

        foreach ($prepaper as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }

        return $this;
    }

    public function getBySlug( $slug, $notId = null ) {
        $sql = new DbQuery();
        $sql->select('p.*');
        $sql->from('prepaper', 'p');
        $sql->where('p.`slug` = \'' . pSQL($slug) . '\'');

        if(!is_null($notId)) {
            $sql->where('p.`prepaper_id` != ' . (int) $notId );
        }

        $result = Db::getInstance()->getRow($sql);

        if (!$result) {
            return false;
        }

        $this->id = $result['prepaper_id'];
        foreach ($result as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }

        return $this;
    }

    public function getByOrderId( $orderId ) {
        $sql = new DbQuery();
        $sql->select('p.*');
        $sql->from('prepaper', 'p');
        $sql->where('p.`order_id` = ' . (int) $orderId );
        $result = Db::getInstance()->getRow($sql);

        if (!$result) {
            return false;
        }

        $this->id = $result['prepaper_id'];
        foreach ($result as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }

        return $this;
    }

    public function delete() {
        $success = parent::delete();

        if($success) {
            Db::getInstance()->delete('prepaper_payment', 'prepaper_id = ' . (int) $this->id );
            Db::getInstance()->delete('prepaper_file', 'prepaper_id = ' . (int) $this->id );
            Db::getInstance()->delete('prepaper_preview', 'prepaper_id = ' . (int) $this->id );
        }
    }

    /**
     * used to cache prepaper files.
     */
    protected $prepaperFiles = null;

    public function getFiles() {
        if (is_null($this->prepaperFiles)) {
            $sql = new DbQuery();
            $sql->select('pf.prepaper_file_id');
            $sql->from('prepaper_file', 'pf');
            $sql->where('pf.`prepaper_id` = ' . (int) $this->id );
            $sql->orderBy('pf.prepaper_file_id');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }

            $this->prepaperFiles = $result;
        }

		return $this->prepaperFiles;
    }

    /**
     * used to cache prepaper previews.
     */
    protected $prepaperPreviews = null;

    public function getPreviews() {
        if (is_null($this->prepaperPreviews)) {
            $sql = new DbQuery();
            $sql->select('p_preview.prepaper_preview_id');
            $sql->from('prepaper_preview', 'p_preview');
            $sql->where('p_preview.`prepaper_id` = ' . (int) $this->id );
            $sql->orderBy('p_preview.prepaper_preview_id');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }
            
            $this->prepaperPreviews = $result;
        }

		return $this->prepaperPreviews;
    }

     /**
     * used to cache prepaper payments.
     */
    protected $prepaperPayments = null;

    public function getPayments() {
        if (is_null($this->prepaperPayments)) {
            $sql = new DbQuery();
            $sql->select('p_payments.prepaper_payment_id');
            $sql->from('prepaper_payment', 'p_payments');
            $sql->where('p_payments.`prepaper_id` = ' . (int) $this->id );
            $sql->where('p_payments.`is_paid` = 1' );
            $sql->orderBy('p_payments.prepaper_payment_id');

            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS( $sql );

            if (!$result) {
                return array();
            }
            
            $this->prepaperPayments = $result;
        }

		return $this->prepaperPayments;
    }
    
    /**
     * This method return the ID of the previous prepaper.
     *
     * @since 1.5.0.1
     *
     * @return int
     */
    public function getPreviousPrePaperId( $customerId = null )
    {
        return Db::getInstance()->getValue('
            SELECT prepaper_id
            FROM ' . DB_PREFIX . 'prepaper
			WHERE prepaper_id < ' . (int) $this->id . 
			($customerId ? ' AND customer_id = '. (int) $customerId : '') . '
            ORDER BY prepaper_id DESC');
    }
	
	/**
     * This method return the ID of the next prepaper.
     *
     * @since 1.0.0
     *
     * @return int
     */
    public function getNextPrePaperId( $customerId = null )
    {
        return Db::getInstance()->getValue('
            SELECT prepaper_id
            FROM ' . DB_PREFIX . 'prepaper
			WHERE prepaper_id > ' . (int) $this->id . 
			($customerId ? ' AND customer_id = '. (int) $customerId : '') . '
            ORDER BY prepaper_id ASC');
    }
}