<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class Pesapal extends PaymentModule
{
    /**
    * Pesapal configuration
    */
    const PESAPAL_ENABLED = 'PESAPAL_ENABLED';
    const PESAPAL_MODE = 'PESAPAL_MODE';
    const PESAPAL_CONSUMER_KEY = 'PESAPAL_CONSUMER_KEY';
    const PESAPAL_CONSUMER_SECRET = 'PESAPAL_CONSUMER_SECRET';

    public function __construct()
    {
        $this->name = 'pesapal';
        $this->icon = 'fa fa-credit-card';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Pesapal';
        $this->description = 'Set up Pesapal as a payment method on your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $pesapalConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::PESAPAL_ENABLED,
                self::PESAPAL_MODE,
                self::PESAPAL_CONSUMER_KEY,
                self::PESAPAL_CONSUMER_SECRET,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'pesapalConfiguration' => $pesapalConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updatePesapalSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        if( ArrayUtils::has($payload, 'default_method_enabled') ) {
            Configuration::updateValue('PAYMENT_METHOD_DEFAULT', $this->name , false, $site_id);
        }

        Configuration::updateValue( self::PESAPAL_ENABLED, (ArrayUtils::get($payload, 'pesapal_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue( self::PESAPAL_MODE, ArrayUtils::get($payload, 'pesapal_mode'), false, $site_id);
        Configuration::updateValue( self::PESAPAL_CONSUMER_KEY, ArrayUtils::get($payload, 'pesapal_consumer_key'), false, $site_id);
        Configuration::updateValue( self::PESAPAL_CONSUMER_SECRET, ArrayUtils::get($payload, 'pesapal_consumer_secret'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}