<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class PayPal extends PaymentModule
{
    /**
    * PayPal configuration
    */
    const PAYPAL_ENABLED = 'PAYPAL_ENABLED';
    const PAYPAL_MODE = 'PAYPAL_MODE';
    const PAYPAL_CLIENT_ID = 'PAYPAL_CLIENT_ID';
    const PAYPAL_SECRET_KEY = 'PAYPAL_SECRET_KEY';
    const PAYPAL_INTEGRATION_MODE = 'PAYPAL_INTEGRATION_MODE';
    const PAYPAL_EMAIL = 'PAYPAL_EMAIL';

    public function __construct()
    {
        $this->name = 'paypal';
        $this->icon = 'fab fa-paypal';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'PayPal';
        $this->description = 'Set up PayPal as a payment method and enable users to pay without leaving your website';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $paypalConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::PAYPAL_ENABLED,
                self::PAYPAL_MODE,
                self::PAYPAL_CLIENT_ID,
                self::PAYPAL_SECRET_KEY,
                self::PAYPAL_EMAIL,
                'SITE_ID'
            ],
            $application->site->id
        );

        $paypalConfiguration[self::PAYPAL_INTEGRATION_MODE] = Configuration::get(self::PAYPAL_INTEGRATION_MODE, $application->site->id, 'paypal_api_keys');

        $this->smarty->assign([
            'paypalConfiguration' => $paypalConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updatePayPalSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        Configuration::updateValue('PAYMENT_METHOD_DEFAULT', ArrayUtils::has($payload, 'default_method_enabled') ? $this->name : '' , false, $site_id);
        Configuration::updateValue(self::PAYPAL_ENABLED, (ArrayUtils::get($payload, 'paypal_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue(self::PAYPAL_MODE, ArrayUtils::get($payload, 'paypal_mode'), false, $site_id);
        Configuration::updateValue(self::PAYPAL_CLIENT_ID, ArrayUtils::get($payload, 'paypal_client_id'), false, $site_id);
        Configuration::updateValue(self::PAYPAL_SECRET_KEY, ArrayUtils::get($payload, 'paypal_secret_key'), false, $site_id);

        if(
            ArrayUtils::get($payload, 'paypal_integration_mode') == 'paypal_email' && 
            !Validate::isEmail(ArrayUtils::get($payload, 'paypal_email'))) 
        {
            return $app->sendResponse([
                "error" => true,
                "message" => "Please enter a valid PayPal email"
            ]);
        }

        Configuration::updateValue(self::PAYPAL_INTEGRATION_MODE, ArrayUtils::get($payload, 'paypal_integration_mode'), false, $site_id);
        Configuration::updateValue(self::PAYPAL_EMAIL, ArrayUtils::get($payload, 'paypal_email'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}