<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Cookie;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Tools;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Validate; 

define('CURRENT_MESSENGER_MODULE_DIR', realpath(dirname(__FILE__)));

require_once CURRENT_MESSENGER_MODULE_DIR . '/classes/Conversation.php';
require_once CURRENT_MESSENGER_MODULE_DIR . '/classes/ConversationUser.php';
require_once CURRENT_MESSENGER_MODULE_DIR . '/classes/ConversationMessage.php';

class Messenger extends Module
{
    /**
     * @var string
     */
    private $css_path;

    /**
     * @var string
     */
    private $js_path;

    /**
     * @var string 
     */
    private $module_path;

    /**
     * @var bool
     */
    private $online_users_installed = false;

    public $cookie;

    public $active_theme = PROX_ACTIVE_THEME;

    public function __construct()
    {
        $this->name = 'messenger';
        $this->icon = 'fas fa-comment-dots';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Messenger';
        $this->description = 'Send messages between users in the admin panel.';

        $this->css_path = $this->_path . 'css/';
        $this->js_path = $this->_path . 'js/';
        $this->module_path = $this->local_path . 'views';

        $this->online_users_installed = (bool) Module::isInstalled('online_users');

        $this->cookie = $this->application->cookie;

        if(!isset($this->cookie->chat_sidebar_closed)) {
            $this->cookie->chat_sidebar_closed = false;
        } 

        @session_start();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->isLogged() ? true : false;
    }

    public function showChatSidebar() {
        $app = $this->application;
        $user = $app->user;

        $routeName = $app->router()->getCurrentRoute()->getName();
        if($routeName == 'tickets') return false;

        $detect = new Mobile_Detect();
        return ($user->is_admin || $user->is_sub_admin) && !($detect->isMobile() && !$detect->isTablet());
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        }

        $this->registerHook([
            'loadMoreBefore',
            'displayFooterAfter',
            'displayHeaderNavbarNav',
            'actionDispatcherBeforeRun',
            'actionControllerSetMedia',
            'actionAfterLiveData',
            'displayEmployeeEditActions',
            'actionEmployeeLogoutAfter'
        ]);
    }

    public function createTables() {
        $sql = '';

        $result = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('employee'));
        if(!isset($result['live_messenger_lastid'])) {
            $sql .= "
                ALTER TABLE " . Db::prefix('employee') . " ADD (
                    `live_messenger_lastid` BIGINT(20) UNSIGNED DEFAULT NULL,
                    `live_messenger_counter` INT(10) UNSIGNED NOT NULL DEFAULT '0'
                );
            ";
        }

        $sql .= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('conversation') . "  (
                `conversation_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `last_message_id` BIGINT(20) UNSIGNED DEFAULT NULL,
                `color` VARCHAR(32) DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY (`conversation_id`)
            ) ENGINE=InnoDB ROW_FORMAT=DYNAMIC;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('conversation_message') . "  (
                `message_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `conversation_id` BIGINT(20) UNSIGNED NOT NULL,
                `user_id` BIGINT(20) UNSIGNED NOT NULL,
                `message` LONGTEXT DEFAULT NULL,
                `image` VARCHAR(256) DEFAULT NULL,
                `voice_note` VARCHAR(256) DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY (`message_id`)
            ) ENGINE=InnoDB ROW_FORMAT=DYNAMIC;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('conversation_user') . " (
                `conversation_user_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `conversation_id` BIGINT(20) UNSIGNED NOT NULL,
                `user_id` BIGINT(20) UNSIGNED NOT NULL,
                `seen` INT(10) NOT NULL DEFAULT 0,
                `typing` INT(10) NOT NULL DEFAULT 0,
                `deleted` INT(10) NOT NULL DEFAULT 0,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY (`conversation_user_id`),
                UNIQUE KEY `conversation_id_user_id` (`conversation_id`, `user_id`)
            ) ENGINE=InnoDB ROW_FORMAT=DYNAMIC;
        ";
        
        try {
            if (!Db::getInstance()->Execute($sql)) {
                return false;
            }
        } catch (Exception $e) {
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookActionControllerSetMedia() {
        $app = Application::getInstance();
        $user = $app->user;

        if($user->isLogged()) {
            if(!$this->online_users_installed) {
                $app->controller->addCss($this->css_path . $this->active_theme . '/chat.sidebar.css');
                $app->controller->addJs($this->js_path . $this->active_theme . '/chart.sidebar.js');
            }
    
            $app->controller->addCss($this->css_path . $this->active_theme . '/messenger.css');
            $app->controller->addJs($this->js_path . $this->active_theme . '/messenger.js');
        }
    }

    public function hookActionEmployeeLogoutAfter() {
        session_destroy();
    }

    public function hookDisplayEmployeeEditActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $employee_id = ArrayUtils::get($params, 'employee_id');
        $employee = new Employee( (int) $employee_id );
        if(Validate::isLoadedObject($employee) && $employee->id != $user->id ) {
            $smarty->assign([
                'employee_id' => $employee->id,
                'employee_name' => Tools::ucfirst($employee->first_name) . ' '. Tools::ucfirst($employee->last_name)
            ]);
            return $this->showTemplate('messenger.actions');
        }
    }

    public function hookDisplayHeaderNavbarNav() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        if($this->checkAccess()) {
            $userMessenger = Db::getInstance()->getRow('SELECT live_messenger_lastid, live_messenger_counter FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $user->id );
            
            $conversations = $this->getConversations();
            $smarty->assign([
                'conversation_tpl_path' => $this->module_path . '/' .  $this->active_theme . '/__feeds_conversation.tpl',
                'live_messenger_counter' => $userMessenger['live_messenger_counter'],
                'max_results_limit' => Configuration::get('MAX_RESULTS', null, 10)*2,
                'messenger_conversations' => $conversations
            ]);
            return $this->showTemplate('messenger.header.navbar');
        }
    }

    public function hookLoadMoreBefore($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $payload = ArrayUtils::get($params, 'payload');
        $get = ArrayUtils::get($payload, 'get');
        $offset = ArrayUtils::get($payload, 'offset');

        // initialize the attach type
        $append = true;

        $return = array();
        /* get conversations */
        if ($get == "conversations") {
            $conversations = $this->getConversations( $offset );
            $smarty->assign([
                'conversation_tpl_path' => $this->module_path . '/' .  $this->active_theme . '/__feeds_conversation.tpl',
                'conversations' => $conversations
            ]);
          
            $return['data'] = $this->getTemplateContent("ajax.live.conversations");
        /* get conversation messages */
        } elseif ($get == "messages") {
            $conversation_id = ArrayUtils::get($payload, 'id');
            /* get conversation */
            $conversation = $this->getConversation( (int) $conversation_id);
            if($conversation) {
                $append = false;
                $messages = $this->getConversationMessages($conversation['conversation_id'] , $offset);
                $smarty->assign([
                    'messages' => $messages,
                    'conversation' => $conversation
                ]);

                $return['data'] = $this->getTemplateContent("ajax.chat.messages");
            }
        }

        /* assign variables */
        $smarty->assign('offset', $offset );
        $smarty->assign('get', $get );
        /* return */
        $return['append'] = $append;

        // return & exit
        return $return;
    }

    public function hookDisplayFooterAfter() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;

        $smarty->assign([
            'chat_sidebar_closed' => (bool) $this->cookie->chat_sidebar_closed,
            'chat_templates_path' => $this->module_path . '/' .  $this->active_theme . '/chat.templates.tpl',
            'module_path' => $this->module_path
        ]);

        if(!$this->online_users_installed) {
            if( $this->showChatSidebar() ) {
                /* get online users */
                $online_users = $this->getOnlineUsers();
                /* get offline users */
                $offline_users = $this->getOfflineUsers();
                /* get sidebar users */
                $sidebar_users = array_merge( $online_users, $offline_users );
    
                /* assign variables */
                $smarty->assign([
                    'chat_master_sidebar_tpl' =>  $this->module_path . '/' .  $this->active_theme . '/ajax.chat.master.sidebar.tpl',
                    'sidebar_users' => $sidebar_users,
                    'online_writers_count' => count($online_users)
                ]);
    
                return $this->showTemplate('chat.sidebar');
            } else {
                return $this->showTemplate('chat.templates');
            }
        } else {
            return $this->showTemplate('chat.templates');
        }
    } 

    /* ------------------------------- */
    /* Chat */
    /* ------------------------------- */

    /**
     * getOnlineUsers
     * 
     * @return array
     */
    public function getOnlineUsers() {
        $app = $this->application;
        $activeUser = $app->user;
        
        /* get online users */
        $online_writers = [];

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where( 'e.last_activity >= SUBTIME(NOW(), SEC_TO_TIME(60))' );
        $sql->where( 'e.employee_id != ' . (int) $activeUser->id );
        $sql->where( 'e.is_started = 1' );
        $sql->orderBy( 'e.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's users */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $user) {
            $user = new Employee( (int) $user['employee_id'] );
            if(Validate::isLoadedObject($user)) {
                $online_writer = $objectPresenter->present($user);
                $online_writer['is_online'] = '1';
                $online_writer['order_stats'] = $user->order_stats;
                $online_writer['user_picture_default'] = !$user->avatar ? true : false;
                $online_writer['user_picture'] = $user->getPicture($user->avatar, $user->gender);
                $online_writers[] = $online_writer;
            }
        }

        return $online_writers;
    }


    /**
     * getOfflineUsers
     * 
     * @return array
     */
    public function getOfflineUsers() {
        $app = $this->application;
        $activeUser = $app->user;

        /* get offline users */
        $offline_writers = [];

        $sql = new DbQuery();
        $sql->select('e.employee_id');
        $sql->from('employee', 'e');
        $sql->where( 'e.last_activity < SUBTIME(NOW(), SEC_TO_TIME(60))' );
        $sql->where( 'e.employee_id != ' . (int) $activeUser->id );
        $sql->where( 'e.is_started = 1' );
        $sql->orderBy( 'e.last_activity DESC' );
        $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql);

        /* make a list from viewer's users */
        $objectPresenter = new ObjectPresenter();

        foreach($result as $user) {
            $user = new Employee( (int) $user['employee_id'] );
            if(Validate::isLoadedObject($user)) {
                $offline_writer = $objectPresenter->present($user);
                $offline_writer['is_online'] = '0';
                $offline_writer['order_stats'] = $user->order_stats;
                $offline_writer['user_picture_default'] = !$user->avatar ? true : false;
                $offline_writer['user_picture'] = $user->getPicture($user->avatar, $user->gender);
                $offline_writers[] = $offline_writer;
            }
        }

        return $offline_writers;
    }

    /**
     * userOnline
     * 
     * @param integer $user_id
     * @return boolean
     */
    public function userOnline($user_id) {
        /* check if the target user is online & enable the chat */ 
        $get_user_status = Db::getInstance()->getValue("SELECT COUNT(*) FROM " . Db::prefix('employee') . " WHERE employee_id = " . (int) $user_id . " AND last_activity >= SUBTIME(NOW(), SEC_TO_TIME(60))");
        /* if no > return false */
        if($get_user_status == 0) return false;

        return true;
    }

    /**
     * get_conversations_new
     * 
     * @return array
     */
    public function getNewConversations() {
        $app = $this->application;
        $user = $app->user;

        if(!isset($_SESSION['chat_boxes_opened'])) {
            $_SESSION['chat_boxes_opened'] = array();
        }

        if(count($_SESSION['chat_boxes_opened'])) {
            /* make list from opened chat boxes keys (conversations ids) */
            $chat_boxes_opened_list = implode(',', $_SESSION['chat_boxes_opened']);
            $userConversations = Db::getInstance()->executeS(sprintf("SELECT conversation_id FROM ". Db::prefix('conversation_user') ." WHERE user_id = %s AND seen = '0' AND deleted = '0' AND conversation_id NOT IN (%s)", $user->id , $chat_boxes_opened_list ));
        } else {
            $userConversations = Db::getInstance()->executeS(sprintf("SELECT conversation_id FROM ". Db::prefix('conversation_user') ." WHERE user_id = %s AND seen = '0' AND deleted = '0'", $user->id ));
        }

        $conversations = [];
        foreach($userConversations as $conversation) {
            Db::getInstance()->update(
                'conversation_user',
                ['seen' => 1],
                sprintf('conversation_id = %s AND user_id = %s', (int) $conversation['conversation_id'], $user->id )
            );

            $conversations[] = $this->getConversation($conversation['conversation_id']);
        }
        
        return $conversations;
    }

    /**
     * getConversations
     * 
     * @param integer $offset
     * @return array
     */
    public function getConversations($offset = 0) {
        $app = $this->application;
        $user = $app->user;

        $max_results = (int) Configuration::get('MAX_RESULTS', null, 10);
        $offset *= $max_results;

        $sql = new DbQuery();
        $sql->select('conversation.conversation_id');
        $sql->from('conversation', 'conversation');
        $sql->innerJoin('conversation_user', 'conversation_user', 'conversation.conversation_id = conversation_user.conversation_id');
        $sql->where('');
        $sql->where('conversation_user.user_id = ' . (int) $user->id);
        $sql->where("conversation_user.deleted = '0'");
        $sql->orderBy('conversation.last_message_id DESC');
        $sql->limit($max_results, $offset);
        $result = Db::getInstance()->executeS($sql);

        $conversations = [];
        foreach($result as $conversation) {
            $conversation = $this->getConversation($conversation['conversation_id']);
            if($conversation) {
                $conversations[] = $conversation;
            }
        }

        return $conversations;
    }

    /**
     * getConversation
     * 
     * @param integer $conversation_id
     * @return array
     */
    public function getConversation($conversation_id) {
        $app = $this->application;
        $user = $app->user;

        $sql = new DbQuery();
        $sql->select('conversation.*, conversation_message.message, conversation_message.image, conversation_message.voice_note, conversation_message.date_add, conversation_user.seen');
        $sql->from('conversation', 'conversation');
        $sql->innerJoin('conversation_message', 'conversation_message', 'conversation.last_message_id = conversation_message.message_id');
        $sql->innerJoin('conversation_user', 'conversation_user', 'conversation.conversation_id = conversation_user.conversation_id');
        $sql->where('conversation_user.user_id = ' . (int) $user->id );
        $sql->where('conversation.conversation_id = ' . (int) $conversation_id );
        $conversation = Db::getInstance()->getRow($sql);

        if(!$conversation) return false;

        /* get recipients */
        $sql = new DbQuery();
        $sql->select('conversation_user.seen, conversation_user.typing, employee.employee_id, employee.first_name, employee.last_name, employee.gender, employee.avatar');
        $sql->from('conversation_user', 'conversation_user');
        $sql->innerJoin('employee', 'employee', 'conversation_user.user_id = employee.employee_id');
        $sql->where('conversation_user.conversation_id = ' . (int) $conversation['conversation_id'] );
        $sql->where('conversation_user.user_id != ' . (int) $user->id );
        $recipients = Db::getInstance()->executeS($sql);

        $recipients_num = count($recipients);
        if($recipients_num == 0) return false;

        $conversation['name_list'] = $conversation['typing_name_list'] = $conversation['seen_name_list'] = '';

        $i = 1;
        foreach($recipients as $recipient) {
            /* get recipient picture */
            $recipient['avatar'] = $user->getPicture($recipient['avatar'], $recipient['gender']);
            /* add to conversation recipients */
            $conversation['recipients'][] = $recipient;
            /* prepare typing recipients */
            if($recipient['typing']) {
                /* check if recipient typing but offline */
                $get_recipient_status = Db::getInstance()->getValue('SELECT COUNT(*) FROM '. Db::prefix('employee') .' WHERE employee_id = '. (int) $recipient['employee_id'] .' AND last_activity >= SUBTIME(NOW(), SEC_TO_TIME(60))');
                if($get_recipient_status == 0) {
                    /* recipient offline -> remove his typing status */
                    Db::getInstance()->update(
                        'conversation_user',
                        ['typing' => '0'],
                        sprintf('conversation_id = %s AND user_id = %s', (int) $conversation_id, (int) $recipient['employee_id'] )
                    );
                } else {
                    /* recipient online -> return his typing status */
                    if($conversation['typing_name_list']) $conversation['typing_name_list'] .= ", ";
                    $conversation['typing_name_list'] .= Tools::htmlentitiesUTF8($recipient['first_name']) . ' ' . Tools::htmlentitiesUTF8($recipient['last_name']);
                }
            } 

            /* prepare seen recipients */
            if($recipient['seen']) {
                if($conversation['seen_name_list']) $conversation['seen_name_list'] .= ", ";
                $conversation['seen_name_list'] .= Tools::htmlentitiesUTF8($recipient['first_name']) . ' ' . Tools::htmlentitiesUTF8($recipient['last_name']);
            }

            /* prepare conversation name_list */
            $conversation['name_list'] .= Tools::htmlentitiesUTF8($recipient['first_name']) . ' ' . Tools::htmlentitiesUTF8($recipient['last_name']);
            if($i < $recipients_num) {
                $conversation['name_list'] .= ", ";
            }
            $i++;
        }

        /* prepare conversation with multiple_recipients */
        if($recipients_num > 1) {
            /* multiple recipients */
            $conversation['multiple_recipients'] = true;
            $conversation['link'] = $conversation['conversation_id'];
            $conversation['picture_left'] = $conversation['recipients'][0]['avatar'];
            $conversation['picture_right'] = $conversation['recipients'][1]['avatar'];
            if($recipients_num > 2) {
                $conversation['name'] = Tools::htmlentitiesUTF8($conversation['recipients'][0]['first_name']) . ", " . Tools::htmlentitiesUTF8($conversation['recipients'][1]['first_name'])." & ".($recipients_num - 2)." more";
            } else {
                $conversation['name'] = Tools::htmlentitiesUTF8($conversation['recipients'][0]['first_name']). " & " . Tools::htmlentitiesUTF8($conversation['recipients'][1]['first_name']);
            }
        } else {
            /* one recipient */
            $one_recipient = $conversation['recipients'][0];
            $conversation['multiple_recipients'] = false;
            $conversation['user_id'] = $one_recipient['employee_id'];
            $conversation['link'] = $one_recipient['employee_id'];
            $conversation['avatar'] = $one_recipient['avatar'];
            $conversation['name'] = Tools::htmlentitiesUTF8($one_recipient['first_name']) . ' ' . Tools::htmlentitiesUTF8($one_recipient['last_name']);
        }

        /* get total number of messages */
        $conversation['total_messages'] = $this->getConversationTotalMessages($conversation_id);
        
        /* decode message text */
        $conversation['message_orginal'] = $conversation['message'];
        $conversation['message'] = $conversation['message'];
        
        /* return */
        return $conversation;
    }

    /**
     * getMutualConversation
     * 
     * @param array $recipients
     * @param boolean $check_deleted
     * @return integer
     */
    public function getMutualConversation($recipients, $check_deleted = false) {
        $app = $this->application;
        $user = $app->user;

        $recipients_array = (array) $recipients;
        $recipients_array[] = $user->id;
        $recipients_list = implode(',', $recipients_array);

        $mutual_conversations = Db::getInstance()->executeS(sprintf('SELECT conversation_id FROM ' . Db::prefix('conversation_user') . ' WHERE user_id IN (%s) GROUP BY conversation_id HAVING COUNT(conversation_id) = %s', $recipients_list, count($recipients_array) ));
        if(count($mutual_conversations) == 0) {
            return false;
        }

        foreach($mutual_conversations as $mutual_conversation) {
            /* get recipients */
            $where_statement = ($check_deleted)? " AND deleted != '1' ": "";
            $recipients_count = Db::getInstance()->getValue(sprintf("SELECT COUNT(*) FROM "  . Db::prefix('conversation_user') .  " WHERE conversation_id = %s" . $where_statement, (int) $mutual_conversation['conversation_id'] ));
            if($recipients_count == count($recipients_array)) {
                return $mutual_conversation['conversation_id'];
            }
        }
    }

    /**
     * getConversationMessages
     * 
     * @param integer $conversation_id
     * @param integer $offset
     * @param integer $last_message_id
     * @return array
     */
    public function getConversationMessages($conversation_id, $offset = 0, $last_message_id = null) {
        $app = $this->application;
        $user = $app->user;

        /* check if user authorized to see this conversation */
        $check_conversation = Db::getInstance()->getValue(sprintf("SELECT COUNT(*) FROM " . Db::prefix('conversation_user') . " WHERE conversation_id = %s AND user_id = %s", (int) $conversation_id, $user->id ));
        if($check_conversation == 0) {
            return false;
        }

        $max_results = Configuration::get('MAX_RESULTS', null, 10);
        $offset *= $max_results;
        if($last_message_id !== null) {
            /* get all messages after the last_message_id */
            $get_messages = Db::getInstance()->executeS(sprintf("SELECT conversation_message.message_id, conversation_message.message, conversation_message.image, conversation_message.voice_note, conversation_message.date_add, user.employee_id, user.first_name, user.last_name, user.gender, user.avatar FROM " . Db::prefix('conversation_message') . " AS conversation_message INNER JOIN " . Db::prefix('employee') . " AS user ON conversation_message.user_id = user.employee_id WHERE conversation_message.conversation_id = %s AND conversation_message.message_id > %s", (int) $conversation_id, (int) $last_message_id ));
        } else {
            $get_messages = Db::getInstance()->executeS(sprintf("SELECT * FROM ( SELECT conversation_message.message_id, conversation_message.message, conversation_message.image, conversation_message.voice_note, conversation_message.date_add, user.employee_id, user.first_name, user.last_name, user.gender, user.avatar FROM " . Db::prefix('conversation_message') . " AS conversation_message INNER JOIN " . Db::prefix('employee') . " AS user ON conversation_message.user_id = user.employee_id WHERE conversation_message.conversation_id = %s ORDER BY conversation_message.message_id DESC LIMIT %s, %s ) messages ORDER BY messages.message_id ASC", (int) $conversation_id, (int) $offset, (int) $max_results ));
        }

        $messages = [];

        foreach($get_messages as $message) {
            $message['avatar'] = $user->getPicture($message['avatar'], $message['gender']);
            $message['message'] = $message['message'];
            /* return */
            $messages[] = $message;
        }

        return $messages;
    }

    /**
     * getConversationTotalMessages
     * 
     * @param integer $conversation_id
     * @return integer
     */
    public function getConversationTotalMessages($conversation_id) {
        $total_messages = Db::getInstance()->getValue("SELECT COUNT(*) FROM " . Db::prefix('conversation_message') . " WHERE conversation_id = " . (int) $conversation_id );
        return $total_messages;
    }

    /**
     * postConversationMessage
     * 
     * @param string $message
     * @param string $image
     * @param integer $conversation_id
     * @param array $recipients
     * @return array
     */
    public function postConversationMessage($message, $image, $voice_note, $conversation_id = null, $recipients = null) {
        $app = $this->application;
        $user = $app->user;

        /* check if posting the message to (new || existed) conversation */
        if($conversation_id == null) {
            /* [first] check previous conversation between (viewer & recipients) */
            $mutual_conversation = $this->getMutualConversation($recipients);
            if(!$mutual_conversation) {
                /* [1] there is no conversation between viewer and the recipients -> start new one */
                /* insert conversation */
                $conversation = new Conversation();
                $conversation->last_message_id = 0;
                $conversation->add();

                $conversation_id = $conversation->id;
                
                /* insert the sender (viewer) */
                $conversation_user = new ConversationUser();
                $conversation_user->conversation_id = (int) $conversation->id;
                $conversation_user->user_id = $user->id;
                $conversation_user->seen = 1;
                $conversation_user->add();

                /* insert recipients */
                foreach($recipients as $recipient) {
                    Db::getInstance()->insert(
                        'conversation_user',
                        [
                            'conversation_id' => (int) $conversation->id,
                            'user_id' => $recipient
                        ]
                    );
                }
            } else {
                /* [2] there is a conversation between the viewer and the recipients */
                /* set the conversation_id */
                $conversation_id = $mutual_conversation;
            }
        } else {
            /* [3] post the message to -> existed conversation */
            /* check if user authorized */
            $conversation = $this->getConversation($conversation_id);
            if(!$conversation) {
                return false;
            }

            /* update sender (viewer) as seen and not deleted if any */
            Db::getInstance()->update(
                'conversation_user',
                [
                    'seen' => 1,
                    'deleted' => 0
                ],
                sprintf('conversation_id = %s AND user_id = %s', (int) $conversation_id, (int) $user->id )
            );

            /* update recipients as not seen and not deleted if any */
            Db::getInstance()->update(
                'conversation_user',
                [
                    'seen' => 0,
                    'deleted' => 0
                ],
                sprintf('conversation_id = %s AND user_id != %s', (int) $conversation_id, (int) $user->id )
            );
        }

        /* insert message */
        $image = ($image != '')? $image : '';
        $voice_note = ($voice_note != '')? $voice_note : '';

        $conversation_message = new ConversationMessage();
        $conversation_message->conversation_id = (int) $conversation_id;
        $conversation_message->user_id = $user->id;
        $conversation_message->message = $message;
        $conversation_message->image = $image;
        $conversation_message->voice_note = $voice_note;
        $conversation_message->add();

        /* update the conversation with last message id */
        $conversation = new Conversation( (int) $conversation_id );
        if(Validate::isLoadedObject($conversation)) {
            $conversation->last_message_id = $conversation_message->id;
            $conversation->update();
        }

        /* update sender (viewer) with last message id */
        Db::getInstance()->update(
            'employee',
            [
                'live_messenger_lastid' => $conversation_message->id
            ],
            'employee_id = ' . (int) $user->id
        );
        
        /* get conversation */
        $conversation = $this->getConversation( (int) $conversation_id );

        /* update all recipients with last message id & only offline recipient messages counter */
        foreach($conversation['recipients'] as $recipient) {
            Db::getInstance()->update(
                'employee',
                [
                    'live_messenger_lastid' => $conversation_message->id,
                    'live_messenger_counter' => 'live_messenger_counter+1'
                ],
                'employee_id = ' . (int) $recipient['employee_id']
            );
        }
        
        /* update typing status of the viewer for this conversation */
        $is_typing = '0';
        Db::getInstance()->query(sprintf("UPDATE " . Db::prefix('conversation_user') . " SET typing = %s WHERE conversation_id = %s AND user_id = %s", $is_typing, (int) $conversation_id, (int) $user->id ));
        
        /* return with conversation */
        return $conversation;       
    }

    /**
     * deleteConversation
     * 
     * @param integer $conversation_id
     * @return void
     */
    public function deleteConversation($conversation_id) {
        $app = $this->application;
        $user = $app->user;

        /* check if user authorized */
        $conversation = Db::getInstance()->getRow(sprintf("SELECT conversation_id FROM " . Db::prefix('conversation_user') . " WHERE conversation_id = %s AND user_id = %s", (int) $conversation_id, (int) $user->id ));
        if(!$conversation) return false;

        /* update typing status of the viewer for this conversation */
        Db::getInstance()->update(
            'conversation_user',
            ['typing' => 0],
            sprintf('conversation_id = %s AND user_id = %s', (int) $conversation_id, (int) $user->id)
        );

        /* update conversation as deleted */
        Db::getInstance()->update(
            'conversation_user',
            ['deleted' => 1],
            sprintf('conversation_id = %s AND user_id = %s', (int) $conversation_id, (int) $user->id)
        );
    }

    /**
     * setConversationColor
     * 
     * @param integer $conversation_id
     * @param string $color
     * @return void
     */
    public function setConversationColor($conversation_id, $color) {
        $app = $this->application;
        $user = $app->user;

        /* check if user authorized */
        $conversation = Db::getInstance()->getRow(sprintf("SELECT conversation_id FROM " . Db::prefix('conversation_user') . " WHERE conversation_id = %s AND user_id = %s", (int) $conversation_id, (int) $user->id ));
        if(!$conversation) return false;

        Db::getInstance()->update(
            'conversation',
            ['color' => $color],
            'conversation_id = ' . (int) $conversation['conversation_id']
        );
    }

    /**
     * updateConversationTypingStatus
     * 
     * @param integer $conversation_id
     * @param boolean $is_typing
     * @return void
     */
    public function updateConversationTypingStatus($conversation_id, $is_typing) {
        $app = $this->application;
        $user = $app->user;

        /* check if user authorized */
        $check_conversation = Db::getInstance()->getRow('SELECT COUNT(*) as count FROM ' . Db::prefix('conversation_user') . ' WHERE conversation_id = ' . (int) $conversation_id . ' AND user_id = ' . (int) $user->id );
        if($check_conversation == 0) {
            return;
        }

        /* update typing status of the viewer for this conversation */
        Db::getInstance()->update(
            'conversation_user',
            ['typing' => $is_typing],
            sprintf('conversation_id = %s AND user_id = %s', (int) $conversation_id, (int) $user->id )
        );
    }

    /**
     * updateConversationSeenStatus
     * 
     * @param array $conversation_ids
     * @return void
     */
    public function updateConversationSeenStatus($conversation_ids) {
        $app = $this->application;
        $user = $app->user;

        $conversations_array = [];
        /* check if user authorized */
        foreach( (array) $conversation_ids as $conversation_id ) {
            $conversations = Db::getInstance()->getValue('SELECT COUNT(*) FROM ' . Db::prefix('conversation_user') . ' WHERE conversation_id = ' . (int) $conversation_id . ' AND user_id = ' . (int) $user->id );
            if($conversations > 0) {
                $conversations_array[] = $conversation_id;
            }
        }

        if(!$conversations_array) return;
        $conversations_list = implode(',', $conversations_array);
        /* update seen status of the viewer to these conversation(s) */
        Db::getInstance()->update(
            'conversation_user',
            ['seen' => 1],
            sprintf('conversation_id IN (%s) AND user_id = %s', $conversations_list, (int) $user->id )
        );
    }

    public function chatMessages() {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $params = $app->request->get();

        if(!isset($_SESSION['chat_boxes_opened'])) {
            $_SESSION['chat_boxes_opened'] = array();
        }

        $conversation_id = ArrayUtils::get($params, 'conversation_id');
        $user_id = ArrayUtils::get($params, 'user_id');

        if(!$conversation_id && !$user_id) {
            return $app->sendResponse([
                'error' => true
            ]);
        }

        // initialize the return array
        $return = array();

        // initialize the conversation
        $conversation = array();

        // get conversation messages
	    /* check single user's chat status */
        if($user_id) {
            $return['user_online'] = ($this->userOnline($user_id)) ? true : false;
        }

        /* if conversation_id not set -> check if there is a mutual conversation */
        if(!$conversation_id) {
            $mutual_conversation = $this->getMutualConversation( (array) $user_id );
            if(!$mutual_conversation) {
                /* there is no mutual conversation -> return & exit */
                return $app->sendResponse($return);
            }

            /* set the conversation_id */
            $conversation_id = $mutual_conversation;
            /* return [conversation_id: to set it as chat-box cid] */
            $return['conversation_id'] = $mutual_conversation;
        }

        /* get convertsation details */
        $conversation = $this->getConversation($conversation_id);

        /* get conversation messages */
        $conversation['messages'] = $this->getConversationMessages($conversation_id);

        /* check if last message sent by the viewer */
        $last_message = end($conversation['messages']);
        if( $conversation['seen_name_list'] && $last_message['employee_id'] == $user->id ) {
            $smarty->assign('last_seen_message_id', $last_message['message_id']);
        }

        /* return [color] */
        $return['color'] = $conversation['color'];

        /* return [messages] */
        $smarty->assign('conversation', $conversation);
        $return['messages'] = $this->getTemplateContent("ajax.conversation.messages");

        /* add conversation to opened chat boxes session if not */
        if(!in_array($conversation['conversation_id'], $_SESSION['chat_boxes_opened'] )) {
            $_SESSION['chat_boxes_opened'][] = $conversation['conversation_id'];
        }

        // return & exit
        return $app->sendResponse($return);
    }

    public function postChat() {
        $app = $this->application;
        $user = $app->user;
        $payload = $app->request->post();

        if(!isset($_SESSION['chat_boxes_opened'])) {
            $_SESSION['chat_boxes_opened'] = array();
        }

        $message = ArrayUtils::get($payload, 'message');
        $photo = ArrayUtils::get($payload, 'photo');

        if(!$message && !$photo) return $app->sendResponse(['error' => true]);

        if($photo) $photo = json_decode($photo);
        $voice_note = ArrayUtils::get($payload, 'voice_note');
        if($voice_note) $photo = json_decode($voice_note);

        /* if both (conversation_id & recipients) not set */
        if(!ArrayUtils::has($payload, 'conversation_id') && !ArrayUtils::has($payload, 'recipients')) {
            return $app->sendResponse(['error' => true]);
        }

        $conversation_id = ArrayUtils::get($payload, 'conversation_id');
        $recipients = ArrayUtils::get($payload, 'recipients');

        /* if recipients not array */
        if($recipients) {
            $recipients = json_decode($recipients);
            if(!is_array($recipients)) {
                return $app->sendResponse(['error' => true]);
            }

            /* recipients must contain numeric values only */
            $recipients = array_filter($recipients, 'is_numeric');
        }

        // post message
        /* post conversation message */
        $conversation = $this->postConversationMessage($message, $photo, $voice_note, $conversation_id, $recipients);

        /* remove typing status */
        $this->updateConversationTypingStatus($conversation['conversation_id'], false);
        
        /* add conversation to opened chat boxes session if not */
        if(!in_array($conversation['conversation_id'], $_SESSION['chat_boxes_opened'] )) {
            $_SESSION['chat_boxes_opened'][] = $conversation['conversation_id'];
        }

        // return & exit
        return $app->sendResponse($conversation);
    }

    public function chatReaction() {
        $app = $this->application;
        $user = $app->user;
        $payload = $app->request->post();

        if(!isset($_SESSION['chat_boxes_opened'])) {
            $_SESSION['chat_boxes_opened'] = array();
        }

        $do =  ArrayUtils::get($payload, 'do');
        $conversation_id = (int) ArrayUtils::get($payload, 'conversation_id');
     
        // initialize the return array
	    $return = array(
            'success' => true
        );

        switch($do) {
            case 'close':
                // close chatbox
                /* unset from opened chat boxes & return */
                if(($key = array_search($conversation_id, $_SESSION['chat_boxes_opened'])) !== false) {
                    unset($_SESSION['chat_boxes_opened'][$key]);
                    $_SESSION['chat_boxes_opened'] = array_values($_SESSION['chat_boxes_opened']);
                    /* remove typing status */
                    $this->updateConversationTypingStatus($conversation_id, false);
                }
                break;

            case 'delete':
                // delete converstaion
                $this->deleteConversation($conversation_id);

                /* unset from opened chat boxes & return */
                if(($key = array_search($conversation_id, $_SESSION['chat_boxes_opened'])) !== false) {
                    unset($_SESSION['chat_boxes_opened'][$key]);
                    $_SESSION['chat_boxes_opened'] = array_values($_SESSION['chat_boxes_opened']);
                }
    
                // return
                $return['callback'] = 'window.location = "'.$app->base_uri.'/messenger"';
                break;
    
            case 'color':
                $color =  ArrayUtils::get($payload, 'color');
                // color converstaion
                $this->setConversationColor($conversation_id, $color);
                break;
    
            case 'typing':
                $is_typing = (bool) ArrayUtils::get($payload, 'is_typing');
                // update typing status
                $this->updateConversationTypingStatus($conversation_id, $is_typing);
                break;
    
            case 'seen':
                $ids =  ArrayUtils::get($payload, 'ids');
                // update seen status
                $this->updateConversationSeenStatus((array) $ids );
                break;
        }

        // return & exit
        return $app->sendResponse($return);
    }

    public function liveChat() {
        $app = $this->application;
        $user = $app->user;
        $smarty = $app->smarty;
        $payload = $app->request->post();

        if(!isset($_SESSION['chat_boxes_opened'])) {
            $_SESSION['chat_boxes_opened'] = array();
        }

        $chat_boxes_opened_client = array();
        if(ArrayUtils::has($payload, 'chat_boxes_opened_client')) {
            $chat_boxes_oc = ArrayUtils::get($payload, 'chat_boxes_opened_client');
            $chat_boxes_oc = Tools::jsonDecode($chat_boxes_oc, true);
            /* filter the client opened chat boxes */
            foreach($chat_boxes_oc as $key => $value) {
                if(Validate::isInt($key) && Validate::isInt($value)) {
                    $chat_boxes_opened_client[$key] = $value;
                }
            }
        }

        // initialize the return array
	    $return = array(
            'success' => true
        );

        $detect = new Mobile_Detect();

        if( !( $detect->isMobile() && !$detect->isTablet()) ) {
            // [1] [update] master chat sidebar (online & offline friends list)
            /* get online users */
            $online_users = $this->getOnlineUsers();
            /* get offline users */
            $offline_users = $this->getOfflineUsers();
            /* get sidebar friends */
            $sidebar_users = array_merge( $online_users, $offline_users );
            // assign variables
            $smarty->assign('sidebar_users', $sidebar_users);
            /* return */
            $return['master']['sidebar'] = $this->getTemplateContent("ajax.chat.master.sidebar");

            /* prepare session */
            if(($key = array_search(NULL, $_SESSION['chat_boxes_opened'])) !== false) {
                unset($_SESSION['chat_boxes_opened'][$key]);
            }

            // [3] & [4] & [5]
            if(!(empty($chat_boxes_opened_client) && empty($_SESSION['chat_boxes_opened']))) {
                // [3] [get] closed chat boxes
                $chat_boxes_closed = array_diff(array_keys($chat_boxes_opened_client), $_SESSION['chat_boxes_opened']);
                if(count($chat_boxes_closed) > 0) {
                    $return['chat_boxes_closed'] = $chat_boxes_closed;
                }

                // [4] [get] opened chat boxes
                $chat_boxes_pre_opened = array_diff($_SESSION['chat_boxes_opened'], array_keys($chat_boxes_opened_client));
                
                $chat_boxes_opened = [];
                foreach($chat_boxes_pre_opened as $opened_conversation_id) {
                    /* get conversation */
                    $conversation = $this->getConversation($opened_conversation_id);
                    if($conversation) {
                        $chat_boxes_opened[] = $conversation;
                    }
                }
                
                if(count($chat_boxes_opened) > 0) {
                    $return['chat_boxes_opened'] = $chat_boxes_opened;
                }

                // [5] [get] updated chat boxes
                $chat_boxes_pre_updated = array_intersect($_SESSION['chat_boxes_opened'], array_keys($chat_boxes_opened_client));
                $chat_boxes_updated = [];
                foreach($chat_boxes_pre_updated as $updated_conversation_id) {
                    /* get conversation */
                    $conversation = $this->getConversation($updated_conversation_id);
                    if($conversation) {
                        $return_this = false;
                        /* [1] check for a new messages for this chat box */
                        if($conversation['last_message_id'] != $chat_boxes_opened_client[$conversation['conversation_id']]) {
                            $return_this = true;
                            /* get new messages */
                            $messages = $this->getConversationMessages($conversation['conversation_id'], 0, $chat_boxes_opened_client[$conversation['conversation_id']]);
                            /* assign variables */
                            $smarty->assign('messages', $messages);
                            /* return */
                            $last_message = end($messages);
                            $conversation['is_me'] = ( $last_message['employee_id'] == $user->id )? true: false;
                            $conversation['messages_count'] = count($messages);
                            $conversation['messages'] = $this->getTemplateContent("ajax.chat.messages");
                        }

                        /* [2] check if any recipient typing */
                        if($conversation['typing_name_list']) {
                            $return_this = true;
                        }
                        /* [3] check if any recipient seen */
                        if($conversation['seen_name_list']) {
                            $return_this = true;
                        }
                        /* [4] check single user's chat status (online|offline) */
                        if(!$conversation['multiple_recipients']) {
                            $return_this = true;
                            /* update single user's chat status */
                            $conversation['user_online'] = ($this->userOnline($conversation['recipients'][0]['employee_id']))? true: false;
                        }
                        /* return */
                        if($return_this) {
                            $chat_boxes_updated[] = $conversation;
                        }
                    }
                }
                
                if(count($chat_boxes_updated) > 0) {
                    $return['chat_boxes_updated'] = $chat_boxes_updated;
                }
            }

            // [6] [get] new chat boxes
            $chat_boxes_new = $this->getNewConversations();
            if(count($chat_boxes_new) > 0) {
                $return['chat_boxes_new'] = $chat_boxes_new;
            }
        }

        /* if opened_thread isset */
        if(ArrayUtils::has($payload, 'opened_thread')) {
            $opened_thread = ArrayUtils::get($payload, 'opened_thread');
            $opened_thread = Tools::jsonDecode($opened_thread, true);

            if( 
                is_array($opened_thread) && 
                (int) ArrayUtils::has($opened_thread, 'conversation_id') && 
                (int) ArrayUtils::has($opened_thread, 'last_message_id')
            ) {

                /* get conversation */
                $conversation = $this->getConversation( (int) $opened_thread['conversation_id']);
                if($conversation) {
                    $return_this = false;
                    /* [1] check for a new messages for this converstaion */
                    if(ArrayUtils::get($conversation, 'last_message_id') != ArrayUtils::get($opened_thread, 'last_message_id')) {
                        $return_this = true;
                        /* get new messages */
                        $messages = $this->getConversationMessages((int) $conversation['conversation_id'], 0, (int) ArrayUtils::get($opened_thread, 'last_message_id'));
                        /* assign variables */
                        $smarty->assign('messages', $messages);
                        /* return */
                        $last_message = end($messages);

                        $conversation['is_me'] = (ArrayUtils::get($last_message, 'employee_id') == $user->id) ? true: false;
                        $conversation['messages_count'] = count($messages);
                        $conversation['messages'] = $this->getTemplateContent("ajax.chat.messages");
                    }

                    /* [2] check if any recipient typing */
                    if($conversation['typing_name_list']) {
                        $return_this = true;
                    }
                    /* [3] check if any recipient seen */
                    if($conversation['seen_name_list']) {
                        $return_this = true;
                    }
                    /* return */
                    if($return_this) {
                        $return['thread_updated'] = $conversation;
                    }
                }
            }
        }

        // return & exit
        return $app->sendResponse($return);
    }

    public function conversation() {
        $app = $this->application;
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        $conversation_id = ArrayUtils::get($params, 'conversation_id');

        // initialize the return array
	    $return = array(
            'success' => true
        );

        // get conversation
        $conversation = $this->getConversation( (int) $conversation_id );
        if($conversation) {
            /* get conversation messages */
            $conversation['messages'] = $this->getConversationMessages($conversation['conversation_id']);
            /* assign variables */
            $smarty->assign([
                'ajax_conversation_messages_tpl' => $this->module_path . '/' .  $this->active_theme . '/ajax.conversation.messages.tpl',
                'conversation' => $conversation
            ]);
            /* return */
            $return['conversation'] = $conversation;
            $return['conversation_html'] = $this->getTemplateContent("ajax.chat.conversation");
        }

        return $app->sendResponse($return);
    }

    public function hookActionAfterLiveData($payload) {
        $app = $this->application;
        $user = $app->user;
        $smarty = $app->smarty;

        $last_message = (int) ArrayUtils::get($payload, 'last_messenger');

        $userMessenger = Db::getInstance()->getRow('SELECT live_messenger_lastid, live_messenger_counter FROM ' . Db::prefix('employee') . ' WHERE employee_id = ' . (int) $user->id );

        $return = array();

        // [2] check for new messgaes
        if($last_message != $userMessenger['live_messenger_lastid'] ) {
            $conversations = $this->getConversations();
            /* assign variables */
            $smarty->assign([
                'conversation_tpl_path' => $this->module_path . '/' .  $this->active_theme . '/__feeds_conversation.tpl',
                'conversations' => $conversations
            ]);
            /* return */
            $return['conversations_count'] = (int) $userMessenger['live_messenger_counter'];
            $return['conversations'] = $this->getTemplateContent("ajax.live.conversations");
        }

        return $return;
    }

    public function toggleChatHeader() {
        $app = $this->application;
        $user = $app->user;
        $smarty = $app->smarty;
        $payload = $app->request->post();

        $closed = (bool) ArrayUtils::get($payload, 'closed');

        $this->cookie->chat_sidebar_closed = ($closed == true) ? false : true;

        return $app->sendResponse(['success' => true]);
    }

    public function getContent() {
        $app = $this->application;
        $user = $app->user;
        $smarty = $app->smarty;
        $params = $app->request->get();

        $sub_view = (!isset($params['new'])) ? 'message' : 'new';

        $conversations = $this->getConversations();

        if($sub_view == 'message') {

            if (ArrayUtils::has($params, 'cid')) {
                $cid = ArrayUtils::get($params, 'cid');
                $conversation = $this->getConversation( (int) $cid );
                $conversation['messages'] = $this->getConversationMessages($conversation['conversation_id']);

                // assign variables
                $smarty->assign('conversation', $conversation);
            } else {
                if($conversations) {
                    $conversation = $conversations[0];
                    $conversation['messages'] = $this->getConversationMessages($conversation['conversation_id']);
                    // assign variables
                    $smarty->assign('conversation', $conversation);
                }
            }
    
        } elseif ($sub_view == 'new') {
            /* get recipient */
            if(isset($params['uid'])) {
                $recipient = new Employee( (int) $params['uid'] );
                if(Validate::isLoadedObject($recipient)) {
                    /* assign variables */
                    $smarty->assign('recipient', [
                        'employee_id' => $recipient->id,
                        'name' => $recipient->first_name . ' ' . $recipient->last_name,
                    ]);
                }
            }
            
        }

        $smarty->assign([
            'sub_view' => $sub_view,
            'module_no_header' => true,
            'conversation_tpl_path' => $this->module_path . '/' .  $this->active_theme . '/__feeds_conversation.tpl',
            'ajax_conversation_messages_tpl' => $this->module_path . '/' .  $this->active_theme . '/ajax.conversation.messages.tpl',
            'chat_conversation_tpl_path' => $this->module_path . '/' .  $this->active_theme . '/ajax.chat.conversation.tpl',
            'max_results_limit' => Configuration::get('MAX_RESULTS', null, 10)*2,
            'conversations' => $conversations
        ]);

        return $this->getTemplateContent('messenger');
    }
}  