<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Hook;
use Proxim\Module\Module;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class Leavetip extends Module
{
    /**
    * LeaveTip configuration
    */
    const LEAVE_TIP_ENABLED = 'LEAVE_TIP_ENABLED';
    const TIP_WRITER_PERCENT = 'TIP_WRITER_PERCENT';
    const TIP_EDITOR_PERCENT = 'TIP_EDITOR_PERCENT';
    const TIP_WRITER_PERCENT_VALUE = 20;
    const TIP_EDITOR_PERCENT_VALUE = 10;

    public function __construct()
    {
        $this->name = 'leavetip';
        $this->icon = 'fa fa-refund';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Leave Tip';
        $this->description = 'Allow your customers to tip leave a tip';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_sub_admin ? false : true;
    }

    public function install()
    {
        $this->registerHook(['displayOrderAdditionalInfo', 'actionBeforeAdminDelete']);

        Configuration::updateValue(self::LEAVE_TIP_ENABLED, true);
        Configuration::updateValue(self::TIP_WRITER_PERCENT, self::TIP_WRITER_PERCENT_VALUE);

        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('order_tip') . " ( 
                `order_tip_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `order_id` BIGINT(20) UNSIGNED NOT NULL,
                `currency_id` BIGINT(20) UNSIGNED DEFAULT NULL,
                `transaction_id` VARCHAR(255) DEFAULT NULL,
                `payment_method` VARCHAR(255) DEFAULT NULL,
                `amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `writer_tip_amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `editor_tip_amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`order_tip_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;

            CREATE TABLE IF NOT EXISTS " . Db::prefix('tip_option') . " (
                `tip_option_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `type` enum('amount', 'percent') NOT NULL DEFAULT 'amount',
                `value` INT(14) NOT NULL DEFAULT 0,
                `date_add` DATETIME DEFAULT NULL,
            PRIMARY KEY(`tip_option_id`)) ENGINE = InnoDB DEFAULT CHARSET=utf8 ROW_FORMAT=DYNAMIC;
        ";

        try {
            if (!Db::getInstance()->Execute($sql)) {
                return false;
            }
        } catch (Exception $e) {

        }
        
        $results = Db::getInstance()->getRow("SELECT * FROM ". Db::prefix('order_tip'));
        if(!isset($results['writer_tip_amount'])) {
            try {
                if (!Db::getInstance()->Execute("
                    ALTER TABLE " . Db::prefix('order_tip') . " ADD (
                        `writer_tip_amount` FLOAT(14,2) NOT NULL DEFAULT 0.00,
                        `editor_tip_amount` FLOAT(14,2) NOT NULL DEFAULT 0.00
                    );
                ")) {
                    return false;
                }
            } catch (Exception $e) {

            }

            $writer_tip_percent = Configuration::get(self::TIP_WRITER_PERCENT, null, 0);
            $editor_tip_percent = Configuration::get(self::TIP_EDITOR_PERCENT, null, 0);

            $tip_results = Db::getInstance()->executeS("SELECT * FROM ". Db::prefix('order_tip'));

            foreach($tip_results as $tip_res) {
                $writer_tip_amount = ($tip_res['amount'] * $writer_tip_percent) / 100;
                $editor_tip_amount = ($tip_res['amount'] * $editor_tip_percent) / 100;
                
                $success = Db::getInstance()->update(
                    'order_tip',
                    [
                        'writer_tip_amount' => $writer_tip_amount,
                        'editor_tip_amount' => $editor_tip_amount
                    ],
                    'order_tip_id = ' . (int) $tip_res['order_tip_id']
                );
            }
        }

        

        if (!parent::install()) {
            return false;
        }
    }

    public function uninstall()
    {
        if (!parent::uninstall()) {
            return false;
        }
        
        Configuration::updateValue(self::LEAVE_TIP_ENABLED, false);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayOrderAdditionalInfo( $params ) {
        $order_id = ArrayUtils::get($params, 'order_id');

        $sql = new DbQuery();
        $sql->select('ot.*');
        $sql->from('order_tip', 'ot');
        $sql->where('ot.`order_id` = ' . (int) $order_id );
        $sql->orderBy('ot.order_tip_id DESC');
        $result = Db::getInstance()->getRow($sql);

        if($result) {
            $amount = ArrayUtils::get($result, 'amount', 0);
            $writer_tip_amount = ArrayUtils::get($result, 'writer_tip_amount', 0);
            $editor_tip_amount = ArrayUtils::get($result, 'editor_tip_amount', 0);
            
            $this->smarty->assign([
                'amountFormatted' => formatPrice($amount),
                'writerTipAmount' => $writer_tip_amount,
                'writerTipAmountFormatted' => formatPrice($writer_tip_amount),
                'editorTipAmount' => $editor_tip_amount,
                'editorTipAmountFormatted' => formatPrice($editor_tip_amount),
            ]);

            $this->showTemplate('order.tip');
        }
    }

    public function getContent()
    {
        $smarty = $this->smarty;

        $sql = new DbQuery();
        $sql->select('*');
        $sql->from('tip_option');
        $sql->orderBy('type ASC');
        $result = Db::getInstance()->executeS($sql);

        $tipPresets = array();
        foreach($result as $tipPreset) {
            $tipPresets[] = array(
                'id' => $tipPreset['tip_option_id'],
                'type' => $tipPreset['type'],
                'value' => $tipPreset['value'],
            );
        }
 
        $tipConfiguration = Configuration::getMultiple([
            self::TIP_WRITER_PERCENT,
            self::TIP_EDITOR_PERCENT
        ]);

        $smarty->assign([
            'tipConfiguration' => $tipConfiguration,
            'tipPresets' => $tipPresets
        ]);
        
        return $this->getTemplateContent('configure');
    }

    public function updateTipSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        Configuration::updateValue(self::TIP_WRITER_PERCENT, ArrayUtils::get($payload, 'writer_percentage'), self::TIP_WRITER_PERCENT_VALUE);
        Configuration::updateValue(self::TIP_EDITOR_PERCENT, ArrayUtils::get($payload, 'editor_percentage'), self::TIP_EDITOR_PERCENT_VALUE);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }

    public function addTipPreset() {
        $app = $this->application;
        $payload = $app->request->post();

        $tip_type = ArrayUtils::get($payload, 'tip_type');
        $tip_amount = ArrayUtils::get($payload, 'tip_amount');
        $tip_percent = ArrayUtils::get($payload, 'tip_percent');

        $tip_value = 0;
        if($tip_type == "amount") {
            if(!$tip_amount || !Validate::isPrice($tip_amount)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Enter a valid amount"
                ]);
            }

            $tip_value = $tip_amount;
        } else {
            if(!$tip_percent || !Validate::isPercentage($tip_percent)) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => "Enter a valid percentage"
                ]);
            }

            $tip_value = $tip_percent;
        }

        $ok = Db::getInstance()->insert(
            'tip_option',
            [
                'type' => $tip_type,
                'value' => (float) $tip_value,
                'date_add' => DateUtils::now(),
            ]
        );

        if(!$ok) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Could not add preset"
            ]);
        }

        return $app->sendResponse([
            'callback' => 'window.location.reload();'
        ]);

    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        if($handle == "tip-preset") {
            Db::getInstance()->delete(
                'tip_option',
                'tip_option_id = ' . (int) $id
            );
        }
    }

}