<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Module\PaymentModule;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class Jengacheckout extends PaymentModule
{
    /**
    * JengaCheckout configuration
    */
    const JENGA_ENABLED = 'JENGA_ENABLED';
    const JENGA_MODE = 'JENGA_MODE';
    const JENGA_USERNAME = 'JENGA_USERNAME';
    const JENGA_PASSWORD = 'JENGA_PASSWORD';
    const JENGA_API_KEY = 'JENGA_API_KEY';
    const JENGA_OUTLET_CODE = 'JENGA_OUTLET_CODE';
    const JENGA_MERCHANT_CODE = 'JENGA_MERCHANT_CODE';

    public function __construct()
    {
        $this->name = 'jengacheckout';
        $this->icon = 'fab fa-credit-card';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Jenga Checkout';
        $this->description = 'Build a payment experience that adapts to your customers';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    { 
        return parent::install() && 
        $this->registerHook('paymentOptions') &&
        $this->registerHook('verifyPayment') &&
        $this->registerHook('displayClientHeader') &&
        $this->registerHook('displayClientFooter');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    public function getContent()
    {
        $application = $this->application;

        $jengaConfiguration = Configuration::getMultiple([
                'PAYMENT_METHOD_DEFAULT',
                self::JENGA_ENABLED,
                self::JENGA_MODE,
                self::JENGA_USERNAME,
                self::JENGA_PASSWORD,
                self::JENGA_API_KEY,
                self::JENGA_OUTLET_CODE,
                self::JENGA_MERCHANT_CODE,
                'SITE_ID'
            ],
            $application->site->id
        );

        $this->smarty->assign([
            'jengaConfiguration' => $jengaConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateJengaSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $site_id = (int) $app->request->post('site_id');
        if( ArrayUtils::has($payload, 'default_method_enabled') ) {
            Configuration::updateValue('PAYMENT_METHOD_DEFAULT', $this->name , false, $site_id);
        }

        Configuration::updateValue(self::JENGA_ENABLED, (ArrayUtils::get($payload, 'jenga_enabled') == 'on') ? true : false , false, $site_id);
        Configuration::updateValue(self::JENGA_MODE, ArrayUtils::get($payload, 'jenga_mode'), false, $site_id);
        Configuration::updateValue(self::JENGA_USERNAME, ArrayUtils::get($payload, 'jenga_username'), false, $site_id);
        Configuration::updateValue(self::JENGA_PASSWORD, ArrayUtils::get($payload, 'jenga_password'), false, $site_id);
        Configuration::updateValue(self::JENGA_API_KEY, ArrayUtils::get($payload, 'jenga_api_key'), false, $site_id);
        Configuration::updateValue(self::JENGA_OUTLET_CODE, ArrayUtils::get($payload, 'jenga_outlet_code'), false, $site_id);
        Configuration::updateValue(self::JENGA_MERCHANT_CODE, ArrayUtils::get($payload, 'jenga_merchant_code'), false, $site_id);

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}