<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Site\Site;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class Google_Signin extends Module
{
    const GOOGLE_SIGNIN_ENABLED = 'GOOGLE_SIGNIN_ENABLED';
    const GOOGLE_SIGNIN_CLIENT_ID = 'GOOGLE_SIGNIN_CLIENT_ID';

    public function __construct()
    {
        $this->name = 'google_signin';
        $this->icon = 'fa fa-google';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Sign In With Google';
        $this->description = 'This addon gives your customers the ability to sign in with their Google account.';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'displayAdminEditSiteGeneralTabContent',
            'actionAfterSiteSettingsGeneralUpdate'
        ]);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminEditSiteGeneralTabContent() {
        $application = $this->application;
        $smarty = $this->smarty;

        $googleSignInFormValues = Configuration::getMultiple([
                self::GOOGLE_SIGNIN_ENABLED,
                self::GOOGLE_SIGNIN_CLIENT_ID
            ],
            $application->site->id
        );
        $smarty->assign('googleSignInFormValues', $googleSignInFormValues);
        
        return $this->showTemplate('tab_content');
    }

    public function hookActionAfterSiteSettingsGeneralUpdate( $params ) {
        $site_id = ArrayUtils::get($params, 'site_id');
        $payload = ArrayUtils::get($params, 'payload');

        $site = new Site( (int) $site_id );
        if(Validate::isLoadedObject($site)) {
            Configuration::updateValue(self::GOOGLE_SIGNIN_ENABLED, ArrayUtils::has($payload, 'google_signin_enabled') ? true : false, false, $site->id);
            Configuration::updateValue(self::GOOGLE_SIGNIN_CLIENT_ID, ArrayUtils::get($payload, 'google_signin_client_id'), false, $site->id);
        }
    }
}