<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Util\ArrayUtils;

class First_Order_Discount extends Module
{
    const FOD_DISCOUNT_PERCENTAGE = 'FOD_DISCOUNT_PERCENTAGE';
    const FOD_DEFAULT_DISCOUNT = 15;

    public function __construct()
    {
        $this->name = 'first_order_discount';
        $this->icon = 'fa fa-tag';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'First Order Discount';
        $this->description = 'Offer discounts to your customers on their first order with you.';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook([
            'actionCustomerAccountAdd',
            'processPayment'
        ]);

        Configuration::updateValue(self::FOD_DISCOUNT_PERCENTAGE, self::FOD_DEFAULT_DISCOUNT );
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent() {
        $fodConfiguration = Configuration::getMultiple([
            self::FOD_DISCOUNT_PERCENTAGE
        ]);

        $this->smarty->assign([
            'fodConfiguration' => $fodConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function updateSettings() {
        $app = $this->application;
        $payload = $app->request->post();

        $discount_percentage = (float) ArrayUtils::get($payload, 'discount_percentage');

        Configuration::updateValue(self::FOD_DISCOUNT_PERCENTAGE, $discount_percentage );

        return $app->sendResponse([
            "success" => true,
			"message" => "System settings have been updated"
        ]);
    }
}