<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\Mail;
use Proxim\MailTemplate;
use Proxim\Module\Module;
use Proxim\Order\File;
use Proxim\Order\Order;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

class email_files_button extends Module
{
    public function __construct()
    {
        $this->name = 'email_files_button';
        $this->icon = 'fa fa-envelope-open';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Quick Email Customer Files Button';
        $this->description = 'One click button that emails the final paper to the customer';
        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return ($user->is_sub_admin || $user->is_admin) ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        $this->registerHook(['displayOrderWriterFilesActions']);
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayOrderWriterFilesActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $order_id = ArrayUtils::get($params, 'order_id');

        if(!$user->is_admin && !$user->is_sub_admin) {
            return;
        }

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return;
        }

        $smarty->assign([
            'order' => $order->present(),
        ]);

        return $this->showTemplate('actions.email');
    }

    public function saveMail() {
        if ($send && Configuration::get('LOG_EMAILS')) {
            $mail = new Mail();
            $mail->template = Tools::substr($template, 0, 62);
            $mail->subject = Tools::substr($subject, 0, 255);
            $mail->message = $templateHtml;

            $recipientsTo = $message->getTo();
            $recipientsCc = $message->getCc();
            $recipientsBcc = $message->getBcc();
            if (!is_array($recipientsTo)) {
                $recipientsTo = [];
            }
            if (!is_array($recipientsCc)) {
                $recipientsCc = [];
            }
            if (!is_array($recipientsBcc)) {
                $recipientsBcc = [];
            }
            foreach (array_merge($recipientsTo, $recipientsCc, $recipientsBcc) as $email => $recipient_name) {
                /* @var Swift_Address $recipient */
                $mail->id = null;
                $mail->recipient = Tools::substr($email, 0, 255);
                $mail->add();
            }
        }
    }

    public function emailCustomerFiles() {
        $app = $this->application;
        $controller = $app->controller;
        $user = $app->user;
        $order_id = $app->request->post('order_id');

        if(!$user->is_admin && !$user->is_sub_admin) {
            return $controller->modal("ERROR", "Access Denied", "You don't have enough permissions");
        }

        $order = new Order( (int) $order_id );
        if(!Validate::isLoadedObject($order)) {
            return $controller->modal("ERROR", "Not Found", "This order does not exists");
        }

        $order_files = $order->getFiles();

        $fileAttachments = array();
        $customer = $order->getCustomer();

        foreach( $order_files as $file ) {
            $file = new File( (int) $file['file_id']); 
            if( Validate::isLoadedObject($file) ) {
                if ( $file->uploader_id != $customer->id ) {
                    $fileAttachments[] = $file->attach();
                }
            } 
        } 

        try {
            $order_url = Configuration::get('SITE_DOMAIN', $order->site_id) . '/dashboard/order/' . $order->id;
            Mail::send(
                MailTemplate::TEMPLATE_ORDER_FILES,
                'Order '. $order->id . ': Please download your paper',
                array(
                    'order_domain' => Configuration::get('SITE_DOMAIN', $order->site_id),
                    'order_url' => $order_url,
                    'order_id' => $order->id,
                    'order_title' => $order->title,
                    'username' => $customer->name,
                ),
                $customer->email,
                null,
                null,
                null,
                $fileAttachments,
                null,
                null,
                null,
                null,
                null,
                null,
                $customer->site_id
            );  
        } catch( \Exception $e ) {
        }

        return $controller->modal("SUCCESS", "Successful", "Email has been sent successfully");
    }
}