<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Module\Module;
use Proxim\Order\Order;
use Proxim\Site\Site;
use Proxim\Tools;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class BluePost extends Module
{
    const BLUEPOST_SITE_URLS = 'BLUEPOST_SITE_URLS';
    const BLUEPOST_CUSTOM_URLS = 'BLUEPOST_CUSTOM_URLS';

    public function __construct()
    {
        $this->name = 'bluepost';
        $this->icon = 'fab fa-wordpress';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'WP Import';
        $this->description = 'Import your orders to all client websites as WordPress posts in one click';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin || $user->is_sub_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        } 

        $this->registerHook([
            'displayOrderActions' 
        ]);
    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('bluepost') . " (
                `bluepost_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `post_id` BIGINT(20) UNSIGNED NOT NULL,
                `order_id` BIGINT(20) UNSIGNED NOT NULL,
                `site_id` BIGINT(20) UNSIGNED NOT NULL,
                `website_url` VARCHAR(255) DEFAULT NULL,
                `date_add` DATETIME DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY(`bluepost_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function getContent() {
        $bluepostConfiguration = Configuration::getMultiple([
            self::BLUEPOST_CUSTOM_URLS,
            self::BLUEPOST_SITE_URLS
        ]);

        $this->smarty->assign([
            'bluepostConfiguration' => $bluepostConfiguration
        ]);

        return $this->getTemplateContent('configure');
    }

    public function hookDisplayOrderActions($params) {
        $app = $this->application;
        $smarty = $this->smarty;
        $user = $app->user;
        $order_id = ArrayUtils::get($params, 'order_id');

        if($order_id && $user->is_admin || $user->is_sub_admin) {
            $order = new Order( (int) $order_id );
            if(Validate::isLoadedObject($order) && $order->service_type == Order::COMPLEX_ASSIGNMENT) {
                $order_imported = (bool) Db::getInstance(PROX_USE_SQL_SLAVE)->getValue('SELECT post_id FROM ' . Db::prefix('bluepost') . ' WHERE order_id = ' . $order->id);
                $smarty->assign([
                    'order' => $order->present(),
                    'order_imported' => $order_imported,
                    'can_import_order' => $order_imported ? false : true
                ]);
                
                return $this->showTemplate('actions.import');
            }
        }
    }

    public function updateWebsiteUrls() {
        $app = $this->application;
        $payload = $app->request->post();

        $wp_urls = ArrayUtils::get($payload, 'website_urls');
        if(!$wp_urls) {
            return $app->sendResponse([
                'error' => true,
                'message' => "Please enter at least one url"
            ]);
        }

        $website_urls = explode(PHP_EOL, $wp_urls);

        if(is_array($website_urls) && !empty($website_urls)) {
            foreach($website_urls as $url) {
                $url = trim($url);
                $url = rtrim($url, '/\\');
                if(!Validate::isAbsoluteUrl($url)) {
                    return $app->sendResponse([
                        'error' => true,
                        'message' => "<strong>" . $url . "</strong> is not a valid url"
                    ]);
                }
            }
        }

        Configuration::updateValue(self::BLUEPOST_CUSTOM_URLS, ArrayUtils::has($payload, 'custom_urls'));
        Configuration::updateValue(self::BLUEPOST_SITE_URLS, $wp_urls, true);

        return $app->sendResponse([
            'success' => true,
            'message' => "Website URLs successfuly updated"
        ]);
    }

    public function publishPost($data = array()) {
        $domain = ArrayUtils::get($data, 'domain');
        $title = ArrayUtils::get($data, 'title');
        $category = ArrayUtils::get($data, 'category');
        $content = ArrayUtils::get($data, 'content');
        $order_id = (int) ArrayUtils::get($data, 'order_id');

        $return = '';

        try {
            $ch = curl_init( $domain . '/wp-json/bluepost/v1/publish' );
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, array(
                'domain' => $domain,
                'post_title' => $title,
                'post_category' => $category,
                'post_content' => $content,
                'order_id' => $order_id
            ));

            $response = curl_exec($ch);

            // close the connection, release resources used
            curl_close($ch);
            $response = Tools::jsonDecode($response, true);

            if(ArrayUtils::get($response, 'success') == false) {
                $return = $domain . " : " . ArrayUtils::get($response, 'message');
            } else {
                $return = $domain . " : Order successfully imported";
            }
        } catch (Exception $e) {
            $return = $domain . " : Import unsuccessful";
        }

        return $return;
    }

    public function publish() {
        $app = $this->application;
        $payload = $app->request->post();

        $order_id = ArrayUtils::get($payload, 'order_id');
        $title = ArrayUtils::get($payload, 'title');
        $category = ArrayUtils::get($payload, 'category');
        $content = ArrayUtils::get($payload, 'instructions');

        if(!$title || !$category || !$content) {
            return $app->sendResponse([
                'error' => true,
                'message' => "Please fill in all the fields"
            ]);
        }

        if(Configuration::get(self::BLUEPOST_CUSTOM_URLS)) {
            $return = array();

            $website_urls = Configuration::get(self::BLUEPOST_SITE_URLS);
            if(!empty($website_urls)) {
                $website_urls = explode(PHP_EOL, $website_urls);
                if(is_array($website_urls) && !empty($website_urls)) {
                    foreach($website_urls as $url) {
                        $url = trim($url);
                        $url = rtrim($url, '/\\');
                        if(Validate::isAbsoluteUrl($url)) {
                            $return[] = $this->publishPost([
                                'domain' => $url,
                                'order_id' => $order_id,
                                'title' => $title,
                                'category' => $category,
                                'content' => $content
                            ]);
                        } else {
                            $return[] = "<strong>" . $url . "</strong> is not a valid url";
                        }
                    }
                }
            }

            return $app->sendResponse([
                'success' => true,
                'message' => implode("<br>", $return)
            ]);
        } else {
            $return = array();

            $sites = Site::getSites();
            foreach($sites as $site) {
                $return[] = $this->publishPost([
                    'domain' => $site['domain'],
                    'order_id' => $order_id,
                    'title' => $title,
                    'category' => $category,
                    'content' => $content
                ]);
            }
    
            return $app->sendResponse([
                'success' => true,
                'message' => implode("<br>", $return)
            ]);
        }
    }
}