<?php

class BackupRestoreDatabase {
    /**
     * Database core functions for the MySQLi library
     *
     * Note: PHP version 5 states that the MySQL library is "Maintenance only, Long term deprecation announced."
     * It recommends using the PDO::MySQL or the MySQLi library instead.
     *
     * @package core
     */
    // force UTF-8 Ø

    const DATABASE_SOFTWARE = 'MySQLi';
    const DATABASE_MIN_VERSION = '5.0.0';
    const DATABASE_DESIRED_VERSION = '5.5.0';

    private $connection = null;
    private $db_details = array();

    /**
     * Connect to the database server and select the database.
     * @param array $db_details the db configuration parameters
     * @param bool $errorstop set to false to omit error messages
     * @return true if successful connection
     */
    public function __construct($errorstop = true) {
        $this->db_details = array('server' => DB_SERVER, 'user' => DB_USER, 'password' => DB_PASSWORD, 'database' => DB_NAME, 'prefix' => DB_PREFIX);

        if (function_exists('mysqli_connect')) {
            $this->connection = @mysqli_connect($this->db_details['server'], $this->db_details['user'], $this->db_details['password']);
        } else {
            $this->connection = NULL;
        }
        if (!$this->connection) {
            if ($errorstop) {
                throw new Exception(gettext('MySQLi Error: App could not instantiate a connection.'));
            }
            return false;
        }
        if (!$this->connection->select_db($this->db_details['database'])) {
            if ($errorstop) {
                throw new Exception(sprintf(gettext('MySQLi Error: MySQLi returned the error %1$s when App tried to select the database %2$s.'), $this->connection->error, $this->db_details['database']));
            }

            return false;
        }

        if (array_key_exists('UTF-8', $this->db_details) && $this->db_details['UTF-8']) {
            $this->connection->set_charset("utf8");
        }

        // set the sql_mode to relaxed (if possible)
        @$this->connection->query('SET SESSION sql_mode="";');

        return $this->connection;
    }

    /**
     * The main query function. Runs the SQL on the connection and handles errors.
     * @param string $sql sql code
     * @param bool $errorstop set to false to supress the error message
     * @return results of the sql statements
     * @since 0.6
     */
    public function query($sql, $errorstop = true) {
        if ($result = $this->connection->query($sql)) {
            return $result;
        }

        if ($errorstop) {
            $sql = str_replace('`' . $this->db_details['prefix'], '`[' . gettext('prefix') . ']', $sql);
            $sql = str_replace($this->db_details['database'], '[' . gettext('DB') . ']', $sql);
            throw new Exception(sprintf('%1$s Error: ( %2$s ) failed. %1$s returned the error %3$s'), self::DATABASE_SOFTWARE, $sql);
        }
        return false;
    }

    /**
     * Runs a SQL query and returns an associative array of the first row.
     * Doesn't handle multiple rows, so this should only be used for unique entries.
     * @param string $sql sql code
     * @param bool $errorstop set to false to supress the error message
     * @return results of the sql statements
     * @since 0.6
     */
    public function query_single_row($sql, $errorstop = true) {
        $result = $this->query($sql, $errorstop);
        if (is_object($result)) {
            $row = $result->fetch_assoc();
            mysqli_free_result($result);
            return $row;
        } else {
            return false;
        }
    }

    /**
     * Runs a SQL query and returns an array of associative arrays of every row returned.
     * @param string $sql sql code
     * @param bool $errorstop set to false to supress the error message
     * @param string $key optional array index key
     * @return results of the sql statements
     * @since 0.6
     */
    public function query_full_array($sql, $errorstop = true, $key = NULL) {
        $result = $this->query($sql, $errorstop);
        if (is_object($result)) {
            $allrows = array();
            if (is_null($key)) {
                while ($row = $result->fetch_assoc()) {
                    $allrows[] = $row;
                }
            } else {
                while ($row = $result->fetch_assoc()) {
                    $allrows[$row[$key]] = $row;
                }
            }
            mysqli_free_result($result);
            return $allrows;
        } else {
            return false;
        }
    }

    /**
     * mysqli_real_escape_string standin that insures the DB connection is passed.
     *
     * @param string $string
     * @return string
     */
    public function db_quote($string, $addquotes = true) {
        $escaped = $this->connection->real_escape_string($string);
        if ($addquotes) {
            return "'" . $escaped . "'";
        } else {
            return $escaped;
        }
    }

    /*
    * Fetch a result row as an associative array
    */

    public function db_fetch_assoc($resource) {
        if ($resource) {
            return $resource->fetch_assoc();
        }
        return false;
    }

    /*
    * Returns the text of the error message from previous operation
    */
    public function db_error() {
        if (is_object($this->connection)) {
            return mysqli_error($this->connection);
        }
        if (!$msg = mysqli_connect())
            $msg = sprintf(gettext('%s not connected'), self::DATABASE_SOFTWARE);
        return $msg;
    }

    /*
    * Get number of affected rows in previous operation
    */

    public function db_affected_rows() {
        return $this->connection->affected_rows;
    }

    /*
    * Get a result row as an enumerated array
    */

    public function db_fetch_row($result) {
        if (is_object($result)) {
            return $result->fetch_row();
        }
        return false;
    }

    /*
    * Get number of rows in result
    */

    public function db_num_rows($result) {
        return $result->num_rows;
    }

    /**
     * Closes the database
     */
    public function db_close() {
        if ($this->connection) {
            $rslt = $this->connection->close();
        } else {
            $rslt = true;
        }
        $this->connection = NULL;
        return $rslt;
    }

    /*
    * report the software of the database
    */

    public function db_software() {
        $dbversion = trim(@$this->connection->get_server_info());
        preg_match('/[0-9,\.]*/', $dbversion, $matches);
        return array('application' => self::DATABASE_SOFTWARE, 'required' => self::DATABASE_MIN_VERSION, 'desired' => self::DATABASE_DESIRED_VERSION, 'version' => $matches[0]);
    }

    /**
     * create the database
     */
    public function db_create() {
        $sql = 'CREATE DATABASE IF NOT EXISTS ' . '`' . $this->db_details['database'] . '`' . $this->db_collation();
        return $this->query($sql, false);
    }

    /**
     * Returns user's permissions on the database
     */
    public function db_permissions() {
        $sql = "SHOW GRANTS FOR " . $this->db_details['user'] . ";";
        $result = $this->query($sql, false);
        if (!$result) {
            $result = $this->query("SHOW GRANTS;", false);
        }
        if (is_object($result)) {
            $db_results = array();
            while ($onerow = $this->db_fetch_row($result)) {
                $db_results[] = $onerow[0];
            }
            return $db_results;
        } else {
            return false;
        }
    }

    /**
     * Sets the SQL session mode to empty
     */
    public function db_setSQLmode() {
        return $this->query('SET SESSION sql_mode=""', false);
    }

    /**
     * Queries the SQL session mode
     */
    public function db_getSQLmode() {
        $result = $this->query('SELECT @@SESSION.sql_mode;', false);
        if (is_object($result)) {
            $row = $this->db_fetch_row($result);
            return $row[0];
        }
        return false;
    }

    public function db_collation() {
        $collation = ' CHARACTER SET utf8 COLLATE utf8_unicode_ci';
        return $collation;
    }

    public function db_create_table(&$sql) {
        return $this->query($sql, false);
    }

    public function db_table_update(&$sql) {
        return $this->query($sql, false);
    }

    public function db_show($what, $aux = '') {
        switch ($what) {
            case 'tables':
                $sql = "SHOW TABLES FROM `" . $this->db_details['database'] . "` LIKE '" . $this->db_LIKE_escape($this->db_details['prefix']) . "%'";
                return $this->query($sql, false);
            case 'columns':
                $sql = 'SHOW FULL COLUMNS FROM `' . $this->db_details['prefix'] . $aux . '`';
                return $this->query($sql, true);
            case 'variables':
                $sql = "SHOW VARIABLES LIKE '$aux'";
                return $this->query_full_array($sql);
            case 'index':
                $sql = "SHOW INDEX FROM `" . $this->db_details['database'] . '`.' . $aux;
                return $this->query_full_array($sql);
        }
    }

    public function db_list_fields($table) {
        $result = $this->db_show('columns', $table);
        if (is_object($result)) {
            $fields = array();
            while ($row = $this->db_fetch_assoc($result)) {
                $fields[] = $row;
            }
            return $fields;
        } else {
            return false;
        }
    }

    public function db_truncate_table($table) {
        $sql = 'TRUNCATE ' . $this->db_details['prefix'] . $table;
        return $this->query($sql, false);
    }

    public function db_LIKE_escape($str) {
        return strtr($str, array('_' => '\\_', '%' => '\\%'));
    }

    public function db_free_result($result) {
        return mysqli_free_result($result);
    }
}