<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

use Proxim\Application;
use Proxim\Configuration;
use Proxim\FileSystem;
use Proxim\Module\Module;
use Proxim\Util\ArrayUtils;
use Proxim\Validate;

define('CURRENT_BR_MODULE_DIR', realpath(dirname(__FILE__)));

require_once(CURRENT_BR_MODULE_DIR . '/classes/BackupRestoreDatabase.php');

class Backup_Restore extends Module
{
    const HEADER = '__HEADER__';
    const RECORD_SEPARATOR = ':****:';
    const TABLE_SEPARATOR = '::';
    const RESPOND_COUNTER = 1000;
    const BACKUPFOLDER = PROX_DIR_CONTENT . 'backup' . PROX_DS;

    public $handle;
    public $buffer;
    public $counter;
    public $file_version;
    public $compression_handler;

    public function __construct()
    {
        $this->name = 'backup_restore';
        $this->icon = 'fas fa-database';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';
        $this->displayName = 'Backup & Restore';
        $this->description = 'Backup & Restore your database';

        $this->bootstrap = true;
        parent::__construct();
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin === true;
    }

    public function install()
    {
        return parent::install() && $this->registerHook([
            'displayFooterAfter'
        ]);
    }

    public function hookDisplayFooterAfter()
    {
        $app = $this->application;

        $app->controller->addJS(($this->_path) . 'js/backuprestore.js');
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . $templateName.'.tpl');
    }

    /**
     * Generates a selection list from files found on disk
     *
     * @param strig $currentValue the current value of the selector
     * @param string $root directory path to search
     * @param string $suffix suffix to select for
     * @param bool $descending set true to get a reverse order sort
     */
    private function generateListFromFiles($currentValue, $root, $suffix, $descending = false) {
        if (is_dir($root)) {
            $curdir = getcwd();
            chdir($root);
            $filelist = $this->safeGlob('*' . $suffix);
            $list = array();
            foreach ($filelist as $file) {
                $list[] = str_replace($suffix, '', $file);
            }
            return $list;
        }
    }

    /**
     * Provide an alternative to glob which does not return filenames with accented charactes in them
     *
     * NOTE: this function ignores "hidden" files whose name starts with a period!
     *
     * @param string $pattern the 'pattern' for matching files
     * @param bit $flags glob 'flags'
     */
    private function safeGlob($pattern, $flags = 0) {
        $split = explode('/', $pattern);
        $match = '/^' . strtr(addcslashes(array_pop($split), '\\.+^$(){}=!<>|'), array('*' => '.*', '?' => '.?')) . '$/i';
        $path_return = $path = implode('/', $split);
        if (empty($path)) {
            $path = '.';
        } else {
            $path_return = $path_return . '/';
        }
        if (!is_dir($path))
            return array();
        if (($dir = opendir($path)) !== false) {
            $glob = array();
            while (($file = readdir($dir)) !== false) {
                if (@preg_match($match, $file) && $file[0] != '.') {
                    if (is_dir("$path/$file")) {
                        if ($flags & GLOB_MARK)
                            $file.='/';
                        $glob[] = $path_return . $file;
                    } else if (!is_dir("$path/$file") && !($flags & GLOB_ONLYDIR)) {
                        $glob[] = $path_return . $file;
                    }
                }
            }
            closedir($dir);
            if (!($flags & GLOB_NOSORT))
                sort($glob);
            return $glob;
        } else {
            return array();
        }
    }

    public function getContent() {
        $app = $this->application;
        $smarty = $this->smarty;

        $backup_files = array();
        if(is_dir(self::BACKUPFOLDER)) {
            $backup_files = $this->generateListFromFiles('', self::BACKUPFOLDER, '.zdb', true);
        }

        $smarty->assign([
            'view' => 'backup_restore',
            'backup_files' => $backup_files
        ]);

        return $this->getTemplateContent('configure');
    }

    public function extendExecution() {
        @set_time_limit(30);
        echo ' ';
    }

    public function fillbuffer($handle) {
        $record = fread($handle, 8192);
        if ($record === false || empty($record)) {
            return false;
        }
        $this->buffer .= $record;
        return true;
    }
    
    public function getrow($handle) {
        if ($this->file_version == 0 || substr($this->buffer, 0, strlen(self::HEADER)) == self::HEADER) {
            $end = strpos($this->buffer, self::RECORD_SEPARATOR);
            while ($end === false) {
                if ($end = $this->fillbuffer($handle)) {
                    $end = strpos($this->buffer, self::RECORD_SEPARATOR);
                } else {
                    return false;
                }
            }
            $result = substr($this->buffer, 0, $end);
            $this->buffer = substr($this->buffer, $end + strlen(self::RECORD_SEPARATOR));
        } else {
            $i = strpos($this->buffer, ':');
            if ($i === false) {
                $this->fillbuffer($handle);
                $i = strpos($this->buffer, ':');
            }
            $end = substr($this->buffer, 0, $i) + $i + 1;
            while ($end >= strlen($this->buffer)) {
                if (!$this->fillbuffer($handle))
                    return false;
            }
            $result = substr($this->buffer, $i + 1, $end - $i - 1);
            $this->buffer = substr($this->buffer, $end);
        }

        return $result;
    }
    
    function decompressField($str) {
        switch ($this->compression_handler) {
            default:
                return $str;
            case 'bzip2':
                return bzdecompress($str);
            case 'gzip':
                return gzuncompress($str);
        }
    }
    
    public function compressRow($str, $lvl) {
        switch ($this->compression_handler) {
            default:
                return $str;
            case 'bzip2_row':
                return bzcompress($str, $lvl);
            case 'gzip_row':
                return gzcompress($str, $lvl);
        }
    }
    
    public function decompressRow($str) {
        switch ($this->compression_handler) {
            default:
                return $str;
            case 'bzip2_row':
                return bzdecompress($str);
            case 'gzip_row':
                return gzuncompress($str);
        }
    }
    
    public function writeHeader($type, $value) {
        return fwrite($this->handle, self::HEADER . $type . '=' . $value . self::RECORD_SEPARATOR);
    }

    public function createBackup() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        $prefix = trim(DB_PREFIX, '`');
        $prefixLen = strlen($prefix);

        $compression_level = ArrayUtils::get($payload, 'compress');

		if ($compression_level > 0) {
			if (function_exists('bzcompress')) {
				$this->compression_handler = 'bzip2_row';
			} else {
				$this->compression_handler = 'gzip_row';
			}
		} else {
			$this->compression_handler = 'no';
        }

        $db_connection = new BackupRestoreDatabase();
        $tables = array();
        $result = $db_connection->db_show('tables');
		if ($result) {
			while ($row = $db_connection->db_fetch_assoc($result)) {
				$tables[] = $row;
            }
            $db_connection->db_free_result($result);
		}

		if (!empty($tables)) {
            $folder = self::BACKUPFOLDER;
			$filename = $folder . '/backup-' . date('D_M_Y-H_i_s') . '.zdb';
			if (!is_dir($folder)) {
				mkdir($folder, FileSystem::DEFAULT_MODE_FOLDER);
            }
            
			@chmod($folder, FileSystem::DEFAULT_MODE_FOLDER);
            $writeresult = $this->handle = @fopen($filename, 'w');
            
			if ($this->handle === false) {
                return $app->sendResponse([
                    "error" => true,
                    "message" => sprintf('Failed to open %s for writing.', $filename)
                ]);
			} else {
				$writeresult = $this->writeHeader('file_version', 1);
				$writeresult = $writeresult && $this->writeHeader('compression_handler', $this->compression_handler);
				if ($writeresult === false) {
                    return $app->sendResponse([
                        "error" => true,
                        "message" => 'failed writing to backup!'
                    ]);
				}

				$this->counter = 0;
				$writeresult = true;
				foreach ($tables as $row) {
					$table = array_shift($row);
					$unprefixed_table = substr($table, strlen($prefix));
					$sql = 'SELECT * from `' . $table . '`';
					$result = $db_connection->query($sql);
					if ($result) {
						while ($tablerow = $db_connection->db_fetch_assoc($result)) {
							$this->extendExecution();
							$storestring = serialize($tablerow);
							$storestring = $this->compressRow($storestring, $compression_level);
							$storestring = $unprefixed_table . self::TABLE_SEPARATOR . $storestring;
							$storestring = strlen($storestring) . ':' . $storestring;
							$writeresult = fwrite($this->handle, $storestring);
							if ($writeresult === false) {
                                return $app->sendResponse([
                                    "error" => true,
                                    "message" => 'failed writing to backup!'
                                ]);
							}
							$this->counter++;
							if ($this->counter >= self::RESPOND_COUNTER) {
								echo ' ';
								$this->counter = 0;
							}
						}
						$db_connection->db_free_result($result);
					}
					if ($writeresult === false)
						break;
				}
				fclose($this->handle);
				@chmod($filename, 0660 & FileSystem::DEFAULT_MODE_FOLDER);
			}
		} else {
            return $app->sendResponse([
                "error" => true,
                "message" => 'SHOW TABLES failed!'
            ]);
        }
        
		if ($writeresult) {
			if (ArrayUtils::has($payload, 'autobackup')) {
                Configuration::updateValue('LAST_BACKUP_RUN', time());
			}

			if ($compression_level > 0) {
                return $app->sendResponse([
                    "success" => true,
                    "message" => sprintf('backup completed using <em>%1$s(%2$s)</em> compression', $this->compression_handler, $compression_level)
                ]);
			} else {
                return $app->sendResponse([
                    "success" => true,
                    "message" => 'backup completed'
                ]);
            }
		} else {
            return $app->sendResponse([
                "error" => true,
                "message" => "Backup Failed"
            ]);
        }
    }

    public function restoreBackup() {
        $app = $this->application;
        $payload = $app->request->post();
        $user = $app->user;

        $db_connection = new BackupRestoreDatabase();

        $messages = '';
        $errors = array(gettext('No backup set found.'));
        $prefix = trim(DB_PREFIX, '`');
        $prefixLen = strlen($prefix);

        if(ArrayUtils::has($payload, 'backupfile')) {
            $backupfile = ArrayUtils::get($payload, 'backupfile');
			$this->file_version = 0;
			$this->compression_handler = 'gzip';
            $folder = self::BACKUPFOLDER;
            
            $filename = $folder . $backupfile . '.zdb';
            
			if (file_exists($filename)) {
				$handle = fopen($filename, 'r');
				if ($handle !== false) {
					$resource = $db_connection->db_show('tables');
					if ($resource) {
						$result = array();
						while ($row = $db_connection->db_fetch_assoc($resource)) {
							$result[] = $row;
                        }
                        
						$db_connection->db_free_result($resource);
					} else {
						$result = false;
					}

					$unique = $tables = array();
					$table_cleared = array();
					if (is_array($result)) {
						foreach ($result as $row) {
							$this->extendExecution();
							$table = array_shift($row);
							$tables[$table] = array();
							$table_cleared[$table] = false;
							$result2 = $db_connection->db_list_fields(substr($table, $prefixLen));
							if (is_array($result2)) {
								foreach ($result2 as $row) {
									$tables[$table][] = $row['Field'];
								}
							}
							$result2 = $db_connection->db_show('index', $table);
							if (is_array($result2)) {
								foreach ($result2 as $row) {
									if (is_array($row)) {
										if (array_key_exists('Non_unique', $row) && !$row['Non_unique']) {
											$unique[$table][] = $row['Column_name'];
										}
									}
								}
							}
						}
                    }

					$errors = array();
                    $string = $this->getrow($handle);
                    
					while (substr($string, 0, strlen(self::HEADER)) == self::HEADER) {
						$string = substr($string, strlen(self::HEADER));
						$i = strpos($string, '=');
						$type = substr($string, 0, $i);
						$what = substr($string, $i + 1);
						switch ($type) {
							case 'compression_handler':
								$this->compression_handler = $what;
								break;
							case 'file_version':
								$this->file_version = $what;
                        }
                        
						$string = $this->getrow($handle);
                    }

					$counter = 0;
					$missing_table = array();
					$missing_element = array();
					while (!empty($string) && count($errors) < 100) {
						$this->extendExecution();
						$sep = strpos($string, self::TABLE_SEPARATOR);
                        $table = substr($string, 0, $sep);

						if (array_key_exists($prefix . $table, $tables)) {
							if (!$table_cleared[$prefix . $table]) {
								if (!$db_connection->db_truncate_table($table)) {
									$errors[] = gettext('Truncate table<br />') . $db_connection->db_error();
								}
								$table_cleared[$prefix . $table] = true;
							}
							$row = substr($string, $sep + strlen(self::TABLE_SEPARATOR));
							$row = $this->decompressRow($row);
							$row = unserialize($row);

							foreach ($row as $key => $element) {
								if ($this->compression_handler == 'bzip2' || $this->compression_handler == 'gzip') {
									if (!empty($element)) {
										$element = $this->decompressField($element);
									}
								}
								if (array_search($key, $tables[$prefix . $table]) === false) {
									//	Flag it if data will be lost
									$missing_element[] = $table . '->' . $key;
									unset($row[$key]);
								} else {
									if (is_null($element)) {
										$row[$key] = 'NULL';
									} else {
										$row[$key] = $db_connection->db_quote($element);
									}
								}
							}
							if (!empty($row)) {
								$sql = 'INSERT INTO ' . Db::prefix($table) . ' (`' . implode('`,`', array_keys($row)) . '`) VALUES (' . implode(',', $row) . ')';
								if (isset($unique[$prefix . $table])) {
									foreach ($unique[$prefix . $table] as $exclude) {
										unset($row[$exclude]);
									}
								}
								
								if (count($row) > 0) {
									$sqlu = ' ON DUPLICATE KEY UPDATE ';
									foreach ($row as $key => $value) {
										$sqlu .= '`' . $key . '`=' . $value . ',';
									}
									$sqlu = substr($sqlu, 0, -1);
								} else {
									$sqlu = '';
								}
								if (!$db_connection->query($sql . $sqlu, false)) {
									$errors[] = $sql . $sqlu . '<br />' . $db_connection->db_error();
								}
							}
						} else {
							$missing_table[] = $table;
						}
						$counter++;
						if ($counter >= self::RESPOND_COUNTER) {
							echo ' ';
							$counter = 0;
						}
						$string = $this->getrow($handle);
					}
				}
				fclose($handle);
			}
		}

		if (!empty($missing_table) || !empty($missing_element)) {
			$messages = gettext("Restore encountered exceptions");
			if (!empty($missing_table)) {
				$messages .= gettext('The following tables were not restored because the table no longer exists:');
				foreach (array_unique($missing_table) as $item) {
					$messages .= '<em>' . $item . '</em>,';
				}
			}
			if (!empty($missing_element)) {
				$messages .= gettext('The following fields were not restored because the field no longer exists:');

				foreach (array_unique($missing_element) as $item) {
					$messages .= '<em>' . $item . '</em>,';
				}
			}
		} else if (count($errors) > 0) {
			if (count($errors) >= 100) {
				$messages .= gettext('The maximum error count was exceeded and the restore aborted.');
				unset($_GET['compression']);
			} else {
				$messages .= gettext("Restore encountered the following errors:");
			}
			foreach ($errors as $msg) {
				$messages .= $msg;
			}
		} else {
            return $app->sendResponse([
                "success" => true,
                "message" => "Restore was successfull"
            ]);
        }
        
        return $app->sendResponse([
            "error" => true,
            "message" => $messages
        ]);
    }

    public function downloadBackup() {
        $app = $this->application;
        $params = $app->request->get();

        $backup_file = ArrayUtils::get($params, 'backup');

        if(!@fopen(self::BACKUPFOLDER . $backup_file,"rb")) {
            return $app->sendResponse([
                "error" => true,
                "message" => "Not Found"
            ]);
        }

        $downloadFile = self::BACKUPFOLDER . $backup_file;

        $hostname = $app->request->getHost();

        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $hostname . '-' . basename($downloadFile).'"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($downloadFile));
        flush(); // Flush system output buffer
        readfile($downloadFile);
        die();
    }

    public function deleteBackup() {
        $app = $this->application;

        if(is_dir(self::BACKUPFOLDER)) {
            deleteDir(self::BACKUPFOLDER);
        }

        return $app->sendResponse([
            "success" => true,
            "message" => 'Backup Deleted'
        ]);
    }
}