<?php
use Proxim\Application;
use Proxim\Configuration;
use Proxim\Cookie;
use Proxim\Crypto\Hashing;
use Proxim\Database\DbQuery;
use Proxim\Module\Module;
use Proxim\Pager;
use Proxim\Tools;
use Proxim\User\Customer;
use Proxim\User\Employee;
use Proxim\Util\ArrayUtils;
use Proxim\Util\DateUtils;
use Proxim\Validate;

class ActivityLog extends Module
{
    public function __construct()
    {
        $this->name = 'activitylog';
        $this->icon = 'fas fa-user-clock';
        $this->version = '1.0.0';
        $this->prox_versions_compliancy = array('min' => '1.0.0', 'max' => PROX_VERSION);
        $this->author = 'Davison Pro';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Activity Log';
        $this->description = 'Keep an activity log of everything that happens on your site';
    }

    public function checkAccess() {
        $user = $this->application->user;
        return $user->is_admin ? true : false;
    }

    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        if (!$this->createTables()) {
            return false;
        } 

        return $this->registerHook([
            'activitylog',
            'actionObjectAddAfter',
            'actionObjectUpdateBefore',
            'actionBeforeAdminDelete',
            'actionAuthentication',
            'actionEmployeeLogoutAfter',
            'displayAdminNavBarBeforeEnd'
        ]);
    }

    public function createTables() {
        $sql= "
            CREATE TABLE IF NOT EXISTS " . Db::prefix('activitylog') . " (
                `activitylog_id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `user_object` VARCHAR(64) NOT NULL,
                `user_id` BIGINT(20) UNSIGNED NOT NULL,
                `user_ip` VARCHAR(64) NOT NULL,
                `object` VARCHAR(64) NOT NULL,
                `object_id` BIGINT(20) UNSIGNED NOT NULL,
                `event` VARCHAR(64) NOT NULL,
                `data` TEXT DEFAULT NULL,
                `date_upd` DATETIME DEFAULT NULL,
                `date_add` DATETIME DEFAULT NULL,
                PRIMARY KEY(`activitylog_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        if (!Db::getInstance()->Execute($sql)) {
            return false;
        }

        return true;
    }

    /**
     * Echoes a template.
     *
     * @param string $templateName Template name
     */
    public function showTemplate($templateName)
    {
        $this->application->response()->header('Content-Type', 'text/html; charset=utf-8');
        echo $this->getTemplateContent($templateName);
    }

    /**
     * Return a template.
     *
     * @param string $templateName          Template name
     * @param array  $additionnalParameters Additionnal parameters to inject on the Twig template
     *
     * @return string Parsed template
     */
    private function getTemplateContent($templateName, $additionnalParameters = array())
    {
        $this->smarty->assign($additionnalParameters);
        return $this->fetch(__DIR__ . '/views/' . PROX_ACTIVE_THEME . '/' . $templateName.'.tpl');
    }

    public function hookDisplayAdminNavBarBeforeEnd() {
        if($this->checkAccess()) {
            return $this->showTemplate('nav_item');
        }
    }

    public function hookActivitylog( $params ) {
        $app = $this->application;
        $user = $app->user;

        $object = ArrayUtils::get($params, 'object');
        $object_id = ArrayUtils::get($params, 'object_id');
        $event = ArrayUtils::get($params, 'event');

        Db::getInstance()->insert(
            'activitylog',
            array(
                'user_object' => 'employee',
                'user_id' => $user->id,
                'user_ip' => Tools::getIp(),
                'object' => $object,
                'object_id' => (int) $object_id,
                'event' => $event,
                'date_upd' => DateUtils::now(),
                'date_add' => DateUtils::now()
            )
        );
    }

    public function getContent()
    {
        $app = $this->application;
        $smarty = $this->smarty;
        $payload = $app->request->get();

        $date_from = ArrayUtils::get($payload, 'from');
        $date_to = ArrayUtils::get($payload, 'to');
        $query = ArrayUtils::get($payload, 'query');

        if(ArrayUtils::has($payload, 'page')) {
            $selected_page = (int) ArrayUtils::get($payload, 'page');
            ArrayUtils::remove($payload, 'page');
        } else {
            $selected_page = 1;
        }

        $sql = new DbQuery();
        $sql->from('activitylog', 'activitylog');
        $sql->orderBy('activitylog_id DESC');

        if ($query) {
            $sql->where('activitylog.user_id LIKE \'%' . pSQL($query) . '%\' OR 
                        activitylog.user_ip LIKE \'%' . pSQL($query) . '%\' OR 
                        activitylog.event LIKE \'%' . pSQL($query) . '%\' OR 
                        activitylog.object_id LIKE \'%' . pSQL($query) . '%\'');
        }

        if ($date_from || $date_to) {
            if ( $date_from ) {
                $dateFromSQL = DateUtils::date(strtotime($date_from));
            } 

            if ( $date_to ) {
                $dateToSQL = DateUtils::date(strtotime($date_to));
            } 

            if ( $date_from && $date_to ) {
                $sql->where('activitylog.`date_add` BETWEEN "' . pSQL($dateFromSQL) . '" AND "' . pSQL($dateToSQL) . '"');
            } elseif( $date_from ) {
                $sql->where('activitylog.`date_add` >= "' . pSQL($dateFromSQL) . '"');
            }  elseif( $date_to ) {
                $sql->where('activitylog.`date_add` <= "' . pSQL($dateFromSQL) . '"');
            }
        } 

        $numRowsSql = clone $sql;
        $sql->select('activitylog.*');
        $numRowsSql->select('COUNT(activitylog_id) AS count');

        $total_results = Db::getInstance(PROX_USE_SQL_SLAVE)->getValue($numRowsSql);

        $activities = array(); 
        if( $total_results > 0) {
            $params['total_items'] = $total_results;
            $params['selected_page'] = $selected_page;
            $params['items_per_page'] = Configuration::get('MAX_RESULTS', null, 10)*2;
            $params['url'] = "/activitylog?page=%s";
            if( \is_array($payload) && !empty($payload)) {
                $params['url'] .= "&" . http_build_query($payload);
            }

            $pager = new Pager( $params );
            $limit_query = $pager->getLimitSql();
            $result = Db::getInstance(PROX_USE_SQL_SLAVE)->executeS($sql . $limit_query);

            foreach($result as $activity) {
                if($activity['user_object'] == 'employee') {
                    $activityUser = new Employee( (int) $activity['user_id']);
                    if(Validate::isLoadedObject($activityUser)) {
                        $activity['name'] = $activityUser->first_name . ' ' . $activityUser->last_name;
                        $activities[] = $activity;
                    }

                } else {
                    $customer = new Customer( (int) $activity['user_id']);
                    if(Validate::isLoadedObject($customer)) {
                        $activity['name'] = $customer->name;
                        $activities[] = $activity;
                    }
                }
            }

            $smarty->assign([
                'pager' => $pager->getPager()
            ]);
        }

        $smarty->assign([
            'view' => 'activitylog',
            'selected_page' => $selected_page,
            'query' => $query,
            'date_from' => $date_from,
            'date_to' => $date_to,
            'total_rows' => $total_results,
            'activities' => $activities,
            'module_no_header' => true
        ]); 

        return $this->getTemplateContent('activitylog');
    }

    public function hookActionBeforeAdminDelete( $params ) {
        $id = ArrayUtils::get($params, 'id');
        $handle = ArrayUtils::get($params, 'handle');

        if($handle == 'activitylog') {
            Db::getInstance()->delete(
                'activitylog',
                'activitylog_id = ' . (int) $id
            );
        }
    }

}