<?php
/**
 * @package    Proxim
 * @author     Davison Pro <davis@davisonpro.dev | https://davisonpro.dev>
 * @copyright  2019 Proxim
 * @version    1.5.0
 * @since      File available since Release 1.0.0
 */

 // One minute to execute the script
@set_time_limit(300);

/* Improve PHP configuration to prevent issues */
ini_set('default_charset', 'utf-8');

@date_default_timezone_set(@date_default_timezone_get());

// Short name
define('PROX_DS', DIRECTORY_SEPARATOR);

// Modes
define('PROX_IS_CLI_MODE', 'cli' === PHP_SAPI);

// Common end-of-line
define('PROX_EOL', PROX_IS_CLI_MODE ? "\n" : '<br />');

define('PROX_NAMESPACE',            'Proxim');

define('PROX_DIR',                    dirname(__FILE__));
define('PROX_DIR_ROOT',               rtrim(PROX_DIR, PROX_DS) . PROX_DS);
define('PROX_DIR_INCLUDES',           PROX_DIR_ROOT . 'prox-includes' . PROX_DS);
define('PROX_DIR_CLASSES',            PROX_DIR_INCLUDES . 'classes' . PROX_DS);
define('PROX_DIR_CONTROLLERS',        PROX_DIR_INCLUDES . 'controllers' . PROX_DS);
define('PROX_DIR_IMAGES',             PROX_DIR_ROOT . 'images' . PROX_DS);
define('PROX_DIR_CONTENT',            PROX_DIR_ROOT . 'content' . PROX_DS);
define('PROX_DIR_MODULE',             PROX_DIR_CONTENT . 'modules' . PROX_DS);
define('PROX_DIR_THEMES',             PROX_DIR_CONTENT . 'themes' . PROX_DS);
 
define('PROX_API_URL', 'proxim.craftyworks.co' );
define('PROX_ADDONS_DOMAIN', 'addons.craftyworks.co' );
define('PROX_UPDATES_URL', 'https://' . PROX_API_URL  . '/releases/admin' );
define('PROX_API_MODULES_LIST', PROX_ADDONS_DOMAIN . '/xml/modules_list.xml' );

define('PROX_OS_WINDOWS', 'WIN' === strtoupper(substr(PHP_OS, 0, 3)));
define('PROX_IS_PHP_7', version_compare(PHP_VERSION, '7.0.0', '>='));

if (!defined('PROX_MAGIC_QUOTES_GPC')) {
    define('PROX_MAGIC_QUOTES_GPC', false);
}

if ( file_exists(PROX_DIR . PROX_DS . 'vendor' . PROX_DS . 'autoload.php') ) {
    require_once PROX_DIR . PROX_DS . 'vendor' . PROX_DS . 'autoload.php';
}

// check the config file
if(!file_exists(PROX_DIR_ROOT . 'config.php')) {
    /* the config file doesn't exist -> start the installer */
    header('Location: /install');
    exit;
}

require_once PROX_DIR_ROOT . 'autoload.php';

require_once PROX_DIR_INCLUDES . 'globals.php';
require_once PROX_DIR_INCLUDES . 'functions.php';
require_once PROX_DIR_ROOT . 'config.php';

@session_start();

if (PROX_DEBUG) {
    // ini_set('display_errors', 1);
    // ini_set('display_startup_errors', 1);
    // error_reporting(E_ALL);
    define('PROX_DEBUG_SQL', false);
} else {
    define('PROX_DEBUG_SQL', false);
}

use Proxim\Application;
use Proxim\Tools;
use Proxim\Validate;
use Proxim\Configuration;
use Proxim\Database\DbQuery;
use Proxim\Database\DbCore;
use Proxim\Database\DbPDOCore;
use Proxim\Database\DbMySQLiCore;
use Proxim\Cache\CacheMemcachedCore;
use Proxim\Cache\CacheApcCore;
use Proxim\Currency;
use Proxim\Module\Module;
use Proxim\Preference\AcademicLevel;
use Proxim\Preference\UploadType;
use Proxim\Presenter\Object\ObjectPresenter;
use Proxim\Site\Site;
use Proxim\User\Employee;
use Proxim\Util\DateUtils;

abstract class Db extends DbCore {};
class DbPDO extends DbPDOCore {};
class DbMySQLi extends DbMySQLiCore {};
class CacheMemcached extends CacheMemcachedCore {};
class CacheApc extends CacheApcCore {};

/* set db time to UTC */
Db::getInstance()->query("SET time_zone = '+0:00'");

if (Tools::convertBytes(ini_get('upload_max_filesize')) < Tools::convertBytes('100M')) {
    ini_set('upload_max_filesize', '100M');
    ini_set('post_max_size', '100M');
}

// Define some $_SERVER variables like HTTP_HOST if PHP is launched with php-cli
if (Tools::isPHPCLI()) {
    if (!isset($_SERVER['HTTP_HOST']) || empty($_SERVER['HTTP_HOST'])) {
        $_SERVER['HTTP_HOST'] = '';
    }
    if (!isset($_SERVER['SERVER_NAME']) || empty($_SERVER['SERVER_NAME'])) {
        $_SERVER['SERVER_NAME'] = '';
    }
    if (!isset($_SERVER['REMOTE_ADDR']) || empty($_SERVER['REMOTE_ADDR'])) {
        $_SERVER['REMOTE_ADDR'] = '127.0.0.1';
    }
}

/* Redefine REQUEST_URI if empty (on some webservers...) */
if (!isset($_SERVER['REQUEST_URI']) || empty($_SERVER['REQUEST_URI'])) {
    if (!isset($_SERVER['SCRIPT_NAME']) && isset($_SERVER['SCRIPT_FILENAME'])) {
        $_SERVER['SCRIPT_NAME'] = $_SERVER['SCRIPT_FILENAME'];
    }
    if (isset($_SERVER['SCRIPT_NAME'])) {
        if (basename($_SERVER['SCRIPT_NAME']) == 'index.php' && empty($_SERVER['QUERY_STRING'])) {
            $_SERVER['REQUEST_URI'] = dirname($_SERVER['SCRIPT_NAME']).'/';
        } else {
            $_SERVER['REQUEST_URI'] = $_SERVER['SCRIPT_NAME'];
            if (isset($_SERVER['QUERY_STRING']) && !empty($_SERVER['QUERY_STRING'])) {
                $_SERVER['REQUEST_URI'] .= '?'.$_SERVER['QUERY_STRING'];
            }
        }
    }
}

/* Trying to redefine HTTP_HOST if empty (on some webservers...) */
if (!isset($_SERVER['HTTP_HOST']) || empty($_SERVER['HTTP_HOST'])) {
    $_SERVER['HTTP_HOST'] = @getenv('HTTP_HOST');
} 

if(!Configuration::get('PROX_VERSION')) {
    Configuration::updateValue('PROX_VERSION', Application::VERSION);
}

define('PROX_VERSION', Configuration::get('PROX_VERSION'));

$app = new Application([
    'mode' => PROX_ENVIRONMENT,
    'debug' => PROX_DEBUG
]); 

define('PROX_ACTIVE_THEME',           $app->theme->getName());
define('PROX_DIR_CURR_THEME',         PROX_DIR_THEMES . PROX_ACTIVE_THEME . PROX_DS);
define('PROX_DIR_TEMPLATES',          PROX_DIR_CURR_THEME . 'templates' . PROX_DS);
define('PROX_DIR_CACHE',              PROX_DIR_CURR_THEME . 'cache' . PROX_DS);
define('PROX_DIR_MAIL',               PROX_DIR_TEMPLATES . 'mail' . PROX_DS);
define('PROX_DIR_PDF',                PROX_DIR_TEMPLATES . 'pdf' . PROX_DS);

$cloud_storage = false;
/* set system uploads */
if(Configuration::get('S3_ENABLED')) {
    $cloud_storage = true;
    $s3_region = Configuration::get('S3_REGION');
    $s3_bucket = Configuration::get('S3_BUCKET');
    $endpoint = "https://s3.".$s3_region.".amazonaws.com/".$s3_bucket;
    $uploads_path = $endpoint . "/uploads";
} elseif (Configuration::get('DIGITALOCEAN_ENABLED')) {
    $cloud_storage = true;
    $digitalocean_space_name = Configuration::get('DIGITALOCEAN_SPACE_NAME');
    $digitalocean_space_region = Configuration::get('DIGITALOCEAN_SPACE_REGION');
    $endpoint = "https://".$digitalocean_space_name.".".$digitalocean_space_region.".digitaloceanspaces.com";
    $uploads_path = $endpoint."/uploads";
} else {
    $uploads_path = $app->base_uri .'/uploads';
}

Configuration::set('UPLOADS_PATH', $uploads_path);

# add extensions
$allowed_extensions = Configuration::get('ALLOWED_EXTENSIONS');
if( $allowed_extensions ) {
    $extensions = explode(',', $allowed_extensions);
    $extensions = array_merge($extensions, $globals['extensions']);
    $globals['extensions'] = $extensions;
}

$cookie = $app->cookie;
$smarty = $app->smarty;

$regions = array();
$zones = timezone_identifiers_list();
foreach ($zones as $zone) {
    $zone = explode('/', $zone);
    if(in_array($zone[0], array("Africa", "America", "Antarctica", "Arctic", "Asia", "Atlantic", "Australia", "Europe", "Indian"))) {
        if (isset($zone[1]) != '') {
            $regions[$zone[0]][$zone[0]. '/' . $zone[1]] = str_replace('_', ' ', $zone[1]);
        }
    }
}

if (
    (isset($cookie->user_id) && 
    isset($cookie->session_token) && 
    isset($cookie->session_browser) && 
    isset($cookie->session_os) && 
    isset($cookie->password)) && 
    (int) $cookie->user_id
) {
    $sql = new DbQuery();
    $sql->select('e.employee_id, e.employee_group');
    $sql->from('employee', 'e');
    $sql->innerJoin('user_session', 'us', 'us.user_id = e.employee_id');
    $sql->where('e.`employee_id` = ' . (int) $cookie->user_id );
    $sql->where('us.`session_token` = \'' . pSQL($cookie->session_token) . '\'');
    $sql->where('us.`user_browser` = \'' . pSQL($cookie->session_browser) . '\'');
    $sql->where('us.`user_os` = \'' . pSQL($cookie->session_os) . '\'');
    $checkUser = Db::getInstance(PROX_USE_SQL_SLAVE)->getRow($sql);

    if($checkUser) {
        $user = new Employee( (int) $checkUser['employee_id'] );
        if ( Validate::isLoadedObject($user) && $user->password == $cookie->password ) {
            $user->last_activity = DateUtils::now();
            $user->logged = true;
            $user->update();
        } else {
            $cookie->logout();
        }
    } else {
        $cookie->logout();
        $user = new Employee();
    }
} else {
    $user = new Employee();
}

$is_authorize = false;
if (isset($cookie->authorize_id) && (int) $cookie->authorize_id) {
    $admin = new Employee( (int) $cookie->authorize_id );
    if(Validate::isLoadedObject($admin)) {
        $is_authorize = true;
        $smarty->assign([
            'admin' => (array) $admin
        ]);
    }
}

# upload types
$objectPresenter = new ObjectPresenter();
$configUploadTypes = UploadType::getUploadTypes();

$uploadTypes = array();
foreach($configUploadTypes as $uploadType) {
    $uploadType = new UploadType( (int) $uploadType['upload_type_id'] );
    if(Validate::isLoadedObject($uploadType)) {
        if($user->is_writer && $uploadType->writer) {
            $uploadTypes[] = $objectPresenter->present($uploadType);
        } elseif ($user->is_editor && $uploadType->editor) {
            $uploadTypes[] = $objectPresenter->present($uploadType);
        } else {
            $uploadTypes[] = $objectPresenter->present($uploadType);
        }
    }
}

$app->cookie = $cookie;
$app->user = $user;

// run cron job
$app->runCronjob();

// site
$site = new Site(PROX_SITE_ID);
$app->site = $site;

$show_client_cost = $show_writer_info = true;

if($user->is_sub_admin) {
    $show_client_cost = (bool) Configuration::get('SUBADMIN_CLIENT_COST', PROX_SITE_ID, true);
    $show_writer_info = (bool) Configuration::get('SUBADMIN_WRITER_INFO', PROX_SITE_ID, true);
}

$themeOptionsDefaultValues = array(
    'style' => 'ui-default',
    'header' => 'is-light',
    'aside' => 'is-light',
    'skin' => 'default'
);

$themeOptions = array(
    'style' => Configuration::get('CUSTOMIZE_UI_STYLE', PROX_SITE_ID, $themeOptionsDefaultValues['style']),
    'header' => Configuration::get('CUSTOMIZE_HEADER_STYLE', PROX_SITE_ID, $themeOptionsDefaultValues['header']),
    'aside' => Configuration::get('CUSTOMIZE_ASIDE_STYLE', PROX_SITE_ID, $themeOptionsDefaultValues['aside']),
    'skin' => Configuration::get('CUSTOMIZE_UI_SKIN', PROX_SITE_ID, $themeOptionsDefaultValues['skin'])
);

$writer_currency_format = [];
$writer_currency_id = Configuration::get('WRITER_PAY_CURRENCY', PROX_SITE_ID);
$writer_currency = new Currency( (int) $writer_currency_id );
if(Validate::isLoadedObject($writer_currency)) {
    $writer_currency_format = [
        'currency' => [
            'symbol' => $writer_currency->symbol,
            'rate' => $writer_currency->conversion_rate
        ]
    ];
} else {
    $currencyId = Currency::getIdByIsoCode('USD');
    $currency = new Currency( (int) $currencyId );
    $writer_currency_format = [
        'currency' => [
            'symbol' => $currency->symbol,
            'rate' => $currency->conversion_rate
        ]
    ];
} 

$app->writer_currency = $writer_currency_format;

$smarty->assign([
    'writer_currency_format' => $writer_currency_format,
    /* set system theme (day|night) mode */
    'theme_mode_night' => (bool) $cookie->night_mode,
    'user_messages' => $user->getMessages(),
    'user_notifications' => $user->getNotifications(),
    'stats' => $user->order_stats,
    'system_version' => Configuration::get('PROX_VERSION'),
    'new_version' => Configuration::get('PROX_NEW_VERSION'),
    'site_logo' => Configuration::get('SITE_LOGO'),
    'site_dark_logo' => Configuration::get('SITE_DARK_LOGO'),
    'site_name' => Configuration::get('SITE_NAME'),
    'site_email' => Configuration::get('SITE_EMAIL'),
    'css_customize' => Configuration::get('CSS_CUSTOMIZE', null, false),
    'header_color' => Configuration::get('HEADER_COLOR'),
    'color_button_primary' => Configuration::get('BUTTON_PRIMARY'),
    'menu_background' => Configuration::get('MENU_BACKGROUND'),
    'registration_enabled' => Configuration::get('REGISTRATION_ENABLED', null, true),
    'custom_css' => Configuration::get('CUSTOM_CSS', null, ''),
    'site_favicon' => Configuration::get('SITE_FAVICON', null, false),
    'pwa_enabled' => Configuration::get('PWA_ENABLED'),
    'pwa_background_color' => Configuration::get('PWA_BACKGROUND_COLOR'),
    'pwa_theme_color' => Configuration::get('PWA_THEME_COLOR'),
    'pwa_app_icon' => Configuration::get('PWA_APP_ICON'),
    'active_theme' => PROX_ACTIVE_THEME,
    'themeOptions' => $themeOptions,
    'base_uri' => $app->base_uri,
    'theme_uri' => $app->theme_uri,
    'user' => (array) $user,
    'smarty' => $smarty,
    'is_authorize' => $is_authorize,
    'max_results_limit' => Configuration::get('MAX_RESULTS', null, 10),
    'cloud_storage' => (bool) $cloud_storage,
    'upload_types' => array_reverse($uploadTypes),
    'deadline_highlight' => Configuration::get('HIGHLIGHT_ORDER_DEADLINE'),
    'regions' => $regions,
    'default_region' => preg_replace("/(.*)\/.*/", "$1", Configuration::get('DEFAULT_TIMEZONE', PROX_SITE_ID, 'America') ),
    'default_timezone' => Configuration::get('DEFAULT_TIMEZONE'),
    'show_client_cost' => (bool) $show_client_cost,
    'show_writer_info' => (bool) $show_writer_info,
    'smsnotifications_enabled' => Module::isInstalled('smsnotifications'),
    'chat_enabled' => ($user->is_admin || $user->is_sub_admin) && Module::isInstalled('messenger'),
    'is_programming_enabled' => Module::isInstalled('programming'),
    'is_calculations_enabled' => Module::isInstalled('calculations'),
    'is_articlewriting_enabled' => Module::isInstalled('articlewriting'),
    'is_mailtemplates_enabled' => Module::isInstalled('mail_templates'),
    'cookie_consent_enabled' => Configuration::get('COOKIE_CONSENT_ENABLED', PROX_SITE_ID, true),
    'pay_delivered_orders' => Configuration::get('PAY_DELIVERED_ORDERS', PROX_SITE_ID, false),
    'ifs_enabled' => (bool) Configuration::get('INTERNAL_FILE_STORAGE'),
    'editor_account_enabled' => (bool) Configuration::get('EDITOR_ACCOUNT_ENABLED', PROX_SITE_ID, false),
    'confirm_order_assign' => (bool) Configuration::get('CONFIRM_ORDER_ASSIGN', PROX_SITE_ID, false),
    'order_manager_enabled' => (bool) Configuration::get('ORDER_MANAGER_ENABLED', PROX_SITE_ID, false),
    'order_manager_public' => (bool) Configuration::get('ORDER_MANAGER_PUBLIC', PROX_SITE_ID, false),
    'uploads_path' => Configuration::get('UPLOADS_PATH'),
    'view' => '',
    'sub_view' => '',
    'page_title' => ''
]);